/*=========================================================================

  Program:   ParaView
  Module:    vtkPVColorMap.cxx
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

Copyright (c) 2000-2001 Kitware Inc. 469 Clifton Corporate Parkway,
Clifton Park, NY, 12065, USA.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither the name of Kitware nor the names of any contributors may be used
   to endorse or promote products derived from this software without specific 
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/
#include "vtkPVColorMap.h"

#include "vtkCommand.h"
#include "vtkKWChangeColorButton.h"
#include "vtkKWCheckButton.h"
#include "vtkKWEntry.h"
#include "vtkKWImageLabel.h"
#include "vtkKWLabel.h"
#include "vtkKWLabeledEntry.h"
#include "vtkKWLabeledFrame.h"
#include "vtkKWMenuButton.h"
#include "vtkKWOptionMenu.h"
#include "vtkKWPushButton.h"
#include "vtkKWScale.h"
#include "vtkKWWidget.h"
#include "vtkKWTextProperty.h"
#include "vtkKWTkUtilities.h"
#include "vtkLookupTable.h"
#include "vtkObjectFactory.h"
#include "vtkPVApplication.h"
#include "vtkPVData.h"
#include "vtkPVGenericRenderWindowInteractor.h"
#include "vtkPVRenderView.h"
#include "vtkPVSource.h"
#include "vtkPVSourceCollection.h"
#include "vtkPVWindow.h"
#include "vtkRenderer.h"
#include "vtkScalarBarActor.h"
#include "vtkScalarBarWidget.h"

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPVColorMap);
vtkCxxRevisionMacro(vtkPVColorMap, "1.24.2.19");

int vtkPVColorMapCommand(ClientData cd, Tcl_Interp *interp,
                     int argc, char *argv[]);

/* 
 * This part was generated by ImageConvert from image:
 *    presets.png (zlib, base64)
 */
#define image_presets_width         16
#define image_presets_height        16
#define image_presets_pixel_size    4
#define image_presets_buffer_length 264

static unsigned char image_presets[] = 
  "eNrdkusNhSAMhZkHhtAlXAKHgCFkCZeAIXAIXeJcikiqQaN/L8kJr349LQGAQJG1FleB3T"
  "+JYkMIecvnNznuWJ7jqbay/jSIEULgzIvHOvjc4t+yR0//4P9VV3+SUgp932MYBoxaw6S7"
  "yTnM84zgA5ZlwbZtObblr6RMfJd5PSbeGDg3Zd4TH2Pl+f858d3ur/XB7/4+eETGt6SUrP"
  "WTP+WuvPfJn+pf63tz0Zm89G9N4qfSf/Kn/tfi3+K/vf+Z/wENH5fO";

//===========================================================================
//***************************************************************************
class vtkScalarBarWidgetObserver : public vtkCommand
{
public:
  static vtkScalarBarWidgetObserver *New() 
    {return new vtkScalarBarWidgetObserver;};

  vtkScalarBarWidgetObserver()
    {
      this->PVColorMap = 0;
    }

  virtual void Execute(vtkObject* wdg, unsigned long event,  
                       void* calldata)
    {
      if ( this->PVColorMap )
        {
        this->PVColorMap->ExecuteEvent(wdg, event, calldata);
        }
    }

  vtkPVColorMap* PVColorMap;
};

//***************************************************************************
//===========================================================================
//vtkCxxSetObjectMacro(vtkPVColorMap,PVRenderView,vtkPVRenderView);
//----------------------------------------------------------------------------
// No register count because of reference loop.
void vtkPVColorMap::SetPVRenderView(vtkPVRenderView *rv)
{
  this->PVRenderView = rv;
}

//----------------------------------------------------------------------------
vtkPVColorMap::vtkPVColorMap()
{
  this->CommandFunction = vtkPVColorMapCommand;

  // Used to be in vtkPVActorComposite
  static int instanceCount = 0;

  this->ScalarBarTitle = NULL;
  this->ScalarBarLabelFormat = NULL;
  this->ArrayName = NULL;
  this->ScalarBarVisibility = 0;
  this->ScalarRange[0] = 0.0;
  this->ScalarRange[1] = 1.0;
  this->Initialized = 0;

  this->StartHSV[0] = 0.0;
  this->StartHSV[1] = 1.0;
  this->StartHSV[2] = 1.0;
  this->EndHSV[0] = 0.6667;
  this->EndHSV[1] = 1.0;
  this->EndHSV[2] = 1.0;

  this->NumberOfColors = 256;

  this->PVRenderView = NULL;
  this->LookupTableTclName = NULL;
  this->LookupTable = NULL;
  this->ScalarBar = NULL;
  this->ScalarBarObserver = NULL;

  // Create a unique id for creating tcl names.
  ++instanceCount;
  this->InstanceCount = instanceCount;    
  
  this->NumberOfVectorComponents = 1;
  this->VectorComponent = 0;

  // User interaface.
  this->ColorMapFrame = vtkKWLabeledFrame::New();
  this->ArrayNameLabel = vtkKWLabel::New();
  this->NumberOfColorsScale = vtkKWScale::New();  
  this->ColorEditorFrame = vtkKWWidget::New();
  this->StartColorButton = vtkKWChangeColorButton::New();
  this->Map = vtkKWImageLabel::New();
  this->EndColorButton = vtkKWChangeColorButton::New();
  // Stuff for setting the range of the color map.
  this->ColorRangeFrame = vtkKWWidget::New();
  this->ColorRangeResetButton = vtkKWPushButton::New();
  this->ColorRangeMinEntry = vtkKWLabeledEntry::New();
  this->ColorRangeMaxEntry = vtkKWLabeledEntry::New();

  // Stuff for manipulating the scalar bar.
  this->ScalarBarFrame = vtkKWLabeledFrame::New();
  this->ScalarBarCheck = vtkKWCheckButton::New();
  this->ScalarBarTitleFrame = vtkKWWidget::New();
  this->ScalarBarTitleLabel = vtkKWLabel::New();
  this->ScalarBarTitleEntry = vtkKWEntry::New();
  this->ScalarBarLabelFormatFrame = vtkKWWidget::New();
  this->ScalarBarLabelFormatLabel = vtkKWLabel::New();
  this->ScalarBarLabelFormatEntry = vtkKWEntry::New();

  this->BackButton = vtkKWPushButton::New();

  this->MapData = NULL;
  this->MapDataSize = 0;
  this->MapHeight = 25;
  this->MapWidth = 20;

  this->PresetsMenuButton = vtkKWMenuButton::New();

  this->VisitedFlag = 0;

  this->TitleTextPropertyWidget = vtkKWTextProperty::New();
  this->LabelTextPropertyWidget = vtkKWTextProperty::New();
}

//----------------------------------------------------------------------------
vtkPVColorMap::~vtkPVColorMap()
{
  // Used to be in vtkPVActorComposite........
  vtkPVApplication *pvApp = this->GetPVApplication();

  if (this->ArrayName)
    {
    delete [] this->ArrayName;
    this->ArrayName = NULL;
    }
  if (this->ScalarBarTitle)
    {
    delete [] this->ScalarBarTitle;
    this->ScalarBarTitle = NULL;
    }
  if (this->ScalarBarLabelFormat)
    {
    delete [] this->ScalarBarLabelFormat;
    this->ScalarBarLabelFormat = NULL;
    }

  this->SetPVRenderView(NULL);

  if (this->LookupTableTclName)
    {
    if ( pvApp )
      {
      pvApp->BroadcastScript("%s Delete", this->LookupTableTclName);
      }
    this->SetLookupTableTclName(NULL);
    this->LookupTable = NULL;
    }

  if (this->ScalarBar)
    {
    this->ScalarBar->Delete();
    this->ScalarBar = NULL;
    }

  if (this->ScalarBarObserver)
    {
    this->ScalarBarObserver->Delete();
    this->ScalarBarObserver = NULL;
    }
    
  // User interaface.
  this->ColorMapFrame->Delete();
  this->ColorMapFrame = NULL;
  this->ArrayNameLabel->Delete();
  this->ArrayNameLabel = NULL;
  this->NumberOfColorsScale->Delete();
  this->NumberOfColorsScale = NULL;

  this->ColorEditorFrame->Delete();
  this->ColorEditorFrame = NULL;
  this->StartColorButton->Delete();
  this->StartColorButton = NULL;
  this->Map->Delete();
  this->Map = NULL;
  this->EndColorButton->Delete();
  this->EndColorButton = NULL;
  
  // Stuff for setting the range of the color map.
  this->ColorRangeFrame->Delete();
  this->ColorRangeFrame = NULL;
  this->ColorRangeResetButton->Delete();
  this->ColorRangeResetButton = NULL;
  this->ColorRangeMinEntry->Delete();
  this->ColorRangeMinEntry = NULL;
  this->ColorRangeMaxEntry->Delete();      
  this->ColorRangeMaxEntry = NULL;     


  this->ScalarBarFrame->Delete();
  this->ScalarBarFrame = NULL;
  this->ScalarBarCheck->Delete();
  this->ScalarBarCheck = NULL;
  this->ScalarBarTitleFrame->Delete();
  this->ScalarBarTitleFrame = NULL;
  this->ScalarBarTitleLabel->Delete();
  this->ScalarBarTitleLabel = NULL;
  this->ScalarBarTitleEntry->Delete();
  this->ScalarBarTitleEntry = NULL;
  this->ScalarBarLabelFormatFrame->Delete();
  this->ScalarBarLabelFormatFrame = NULL;
  this->ScalarBarLabelFormatLabel->Delete();
  this->ScalarBarLabelFormatLabel = NULL;
  this->ScalarBarLabelFormatEntry->Delete();
  this->ScalarBarLabelFormatEntry = NULL;

  this->BackButton->Delete();
  this->BackButton = NULL;

  if (this->MapData)
    {
    delete [] this->MapData;
    this->MapDataSize = 0;
    this->MapWidth = 0;
    this->MapHeight = 0;
    }

  if ( this->PresetsMenuButton )
    {
    this->PresetsMenuButton->Delete();
    }

  this->TitleTextPropertyWidget->Delete();
  this->TitleTextPropertyWidget = NULL;

  this->LabelTextPropertyWidget->Delete();
  this->LabelTextPropertyWidget = NULL;
}


//----------------------------------------------------------------------------
void vtkPVColorMap::Create(vtkKWApplication *app)
{
  vtkPVApplication *pvApp = vtkPVApplication::SafeDownCast(app);
  const char* wname;
  
  if (this->Application)
    {
    vtkErrorMacro("PVColorMap already created");
    return;
    }
  // Superclass create takes a KWApplication, but we need a PVApplication.
  if (pvApp == NULL)
    {
    vtkErrorMacro("Need a PV application");
    return;
    }
  this->SetApplication(app);
  this->CreateParallelTclObjects(pvApp);
  
  // create the top level
  wname = this->GetWidgetName();
  this->Script("frame %s -borderwidth 0 -relief flat", wname);
  
  // Now for the UI.

  // Color map

  this->ColorMapFrame->SetParent(this);
  this->ColorMapFrame->ShowHideFrameOn();
  this->ColorMapFrame->Create(this->Application);
  this->ColorMapFrame->SetLabel("Color Map");

  // Color map: parameter name

  this->ArrayNameLabel->SetParent(this->ColorMapFrame->GetFrame());
  this->ArrayNameLabel->Create(this->Application, "-anchor w");
  this->ArrayNameLabel->SetLabel("Parameter: ");

  // Color map: range

  this->ColorRangeFrame->SetParent(this->ColorMapFrame->GetFrame());
  this->ColorRangeFrame->Create(this->Application, "frame", "");

  this->ColorRangeResetButton->SetParent(this->ColorRangeFrame);
  this->ColorRangeResetButton->Create(this->Application, 
                                      "-text {Reset Range}");
  this->ColorRangeResetButton->SetCommand(this, "ResetScalarRange");

  this->ColorRangeMinEntry->SetParent(this->ColorRangeFrame);
  this->ColorRangeMinEntry->Create(this->Application);
  this->ColorRangeMinEntry->SetLabel("Min:");
  this->ColorRangeMinEntry->GetEntry()->SetWidth(10);
  this->Script("bind %s <KeyPress-Return> {%s ColorRangeEntryCallback}",
               this->ColorRangeMinEntry->GetEntry()->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ColorRangeEntryCallback}",
               this->ColorRangeMinEntry->GetEntry()->GetWidgetName(),
               this->GetTclName()); 

  this->ColorRangeMaxEntry->SetParent(this->ColorRangeFrame);
  this->ColorRangeMaxEntry->Create(this->Application);
  this->ColorRangeMaxEntry->SetLabel("Max:");
  this->ColorRangeMaxEntry->GetEntry()->SetWidth(
    this->ColorRangeMinEntry->GetEntry()->GetWidth());
  this->Script("bind %s <KeyPress-Return> {%s ColorRangeEntryCallback}",
               this->ColorRangeMaxEntry->GetEntry()->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ColorRangeEntryCallback}",
               this->ColorRangeMaxEntry->GetEntry()->GetWidgetName(),
               this->GetTclName());

  this->Script("pack %s -side left -expand f",
               this->ColorRangeResetButton->GetWidgetName());

  this->Script("pack %s %s -side left -expand t -fill x",
               this->ColorRangeMinEntry->GetWidgetName(),
               this->ColorRangeMaxEntry->GetWidgetName());

  // Color map: gradient editor

  this->ColorEditorFrame->SetParent(this->ColorMapFrame->GetFrame());
  this->ColorEditorFrame->Create(this->Application, "frame", "");

  this->StartColorButton->SetParent(this->ColorEditorFrame);
  this->StartColorButton->SetText("");
  this->StartColorButton->Create(this->Application, "");
  this->StartColorButton->SetColor(1.0, 0.0, 0.0);
  this->StartColorButton->SetCommand(this, "StartColorButtonCallback");
  this->StartColorButton->SetBalloonHelpString("Select the minimum color.");

  this->Map->SetParent(this->ColorEditorFrame);
  this->Map->Create(this->Application, 
                    "-relief flat -bd 0 -highlightthickness 0 -padx 0 -pady 0");
  this->Script("bind %s <Configure> {%s MapConfigureCallback %s}", 
               this->Map->GetWidgetName(), 
               this->GetTclName(), "%w %h");

  this->EndColorButton->SetParent(this->ColorEditorFrame);
  this->EndColorButton->SetText("");
  this->EndColorButton->Create(this->Application, "");
  this->EndColorButton->SetColor(0.0, 0.0, 1.0);
  this->EndColorButton->SetCommand(this, "EndColorButtonCallback");
  this->EndColorButton->SetBalloonHelpString("Select the maximum color.");

  this->PresetsMenuButton->SetParent(this->ColorEditorFrame);
  this->PresetsMenuButton->Create(this->Application, "-indicator 0");
  this->PresetsMenuButton->SetBalloonHelpString("Select a preset color map.");
  this->PresetsMenuButton->AddCommand(
    "Red to Blue", 
    this, "SetColorSchemeToRedBlue", "Set Color Scheme to Red-Blue");
  this->PresetsMenuButton->AddCommand(
    "Blue to Red", 
    this, "SetColorSchemeToBlueRed", "Set Color Scheme to Blue-Red");
  this->PresetsMenuButton->AddCommand(
    "Grayscale", 
    this, "SetColorSchemeToGrayscale", "Set Color Scheme to Grayscale");

  ostrstream presets;
  presets << this->PresetsMenuButton->GetWidgetName() << ".presetsimg" << ends;
  this->Script("image create photo %s", presets.str());
  if (!vtkKWTkUtilities::UpdatePhoto(this->Application->GetMainInterp(),
                                     presets.str(), 
                                     image_presets, 
                                     image_presets_width, 
                                     image_presets_height, 
                                     image_presets_pixel_size,
                                     image_presets_buffer_length,
                                     this->PresetsMenuButton->GetWidgetName()))
    {
    vtkWarningMacro(<< "Error creating photo (presets)");
    this->PresetsMenuButton->SetButtonText("Presets");
    }
  else
    {
    this->Script("%s configure -image %s", 
                 this->PresetsMenuButton->GetWidgetName(), presets.str());
    }
  presets.rdbuf()->freeze(0);

  this->Script("grid %s %s %s %s -sticky news -padx 1 -pady 2",
               this->PresetsMenuButton->GetWidgetName(),
               this->StartColorButton->GetWidgetName(),
               this->Map->GetWidgetName(),
               this->EndColorButton->GetWidgetName());

  this->Script("grid columnconfigure %s 2 -weight 1",
               this->Map->GetParent()->GetWidgetName());
  this->Script("grid %s -ipadx 1",
               this->PresetsMenuButton->GetWidgetName());

  // Color map: resolution

  this->NumberOfColorsScale->SetParent(this->ColorMapFrame->GetFrame());
  this->NumberOfColorsScale->Create(this->Application, "");
  this->NumberOfColorsScale->SetRange(2, 256);
  this->NumberOfColorsScale->SetValue(256);
  this->NumberOfColorsScale->DisplayLabel("Resolution:");
  this->NumberOfColorsScale->DisplayEntry();
  this->NumberOfColorsScale->DisplayEntryAndLabelOnTopOff();
  this->NumberOfColorsScale->SetEndCommand(this, 
                                           "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetEntryCommand(this, 
                                             "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetBalloonHelpString(
    "Select the discrete number of colors in the color map.");

  // Color map: pack

  this->Script("pack %s %s %s %s -side top -expand t -fill x -anchor nw",
               this->ArrayNameLabel->GetWidgetName(),
               this->ColorRangeFrame->GetWidgetName(),
               this->ColorEditorFrame->GetWidgetName(),
               this->NumberOfColorsScale->GetWidgetName());

  // Scalar bar frame

  this->ScalarBarFrame->SetParent(this);
  this->ScalarBarFrame->ShowHideFrameOn();
  this->ScalarBarFrame->Create(this->Application);
  this->ScalarBarFrame->SetLabel("Scalar Bar");

  ostrstream onchangecommand;
  onchangecommand << "[" << this->GetTclName() 
                  << " GetPVRenderView] EventuallyRender" << ends;

  const char *grid_settings = "-padx 1 -pady 2";
  const char *label_settings = "-anchor w";

  // Scalar bar : Visibility

  this->ScalarBarCheck->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarCheck->Create(this->Application, "-text Visibility");
  this->Application->Script(
    "%s configure -command {%s ScalarBarCheckCallback}",
    this->ScalarBarCheck->GetWidgetName(),
    this->GetTclName());

  // Scalar bar : Title control

  this->ScalarBarTitleFrame->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarTitleFrame->Create(this->Application, "frame", "-bd 0");

  this->ScalarBarTitleLabel->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleLabel->SetLabel("Title:");
  this->ScalarBarTitleLabel->Create(this->Application, label_settings);
  
  this->ScalarBarTitleEntry->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleEntry->Create(this->Application, "");
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName()); 

  this->TitleTextPropertyWidget->SetParent(this->ScalarBarTitleFrame);
  this->TitleTextPropertyWidget->SetTextProperty(
    this->ScalarBar->GetScalarBarActor()->GetTitleTextProperty());
  this->TitleTextPropertyWidget->SetActor2D(
    this->ScalarBar->GetScalarBarActor());
  this->TitleTextPropertyWidget->Create(this->Application);
  this->TitleTextPropertyWidget->SetOnChangeCommand(onchangecommand.str());
  this->TitleTextPropertyWidget->SetTraceReferenceObject(this);
  this->TitleTextPropertyWidget->SetTraceReferenceCommand(
    "GetTitleTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarTitleLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarTitleEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -sticky nws %s",
               this->TitleTextPropertyWidget->GetWidgetName(),
               grid_settings);

  // Scalar bar : Label control

  this->ScalarBarLabelFormatFrame->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarLabelFormatFrame->Create(this->Application, "frame", "-bd 0");

  this->ScalarBarLabelFormatLabel->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatLabel->SetLabel("Labels:");
  this->ScalarBarLabelFormatLabel->Create(this->Application, label_settings);
  
  this->ScalarBarLabelFormatEntry->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatEntry->Create(this->Application, "");
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName()); 
  this->ScalarBarLabelFormatEntry->SetBalloonHelpString(
    "Set the labels format (printf() style).");

  this->SetScalarBarLabelFormat(
    this->ScalarBar->GetScalarBarActor()->GetLabelFormat());

  this->LabelTextPropertyWidget->SetParent(this->ScalarBarLabelFormatFrame);
  this->LabelTextPropertyWidget->SetTextProperty(
    this->ScalarBar->GetScalarBarActor()->GetLabelTextProperty());
  this->LabelTextPropertyWidget->SetActor2D(
    this->ScalarBar->GetScalarBarActor());
  this->LabelTextPropertyWidget->Create(this->Application);
  this->LabelTextPropertyWidget->SetOnChangeCommand(onchangecommand.str());
  this->LabelTextPropertyWidget->SetTraceReferenceObject(this);
  this->LabelTextPropertyWidget->SetTraceReferenceCommand(
    "GetLabelTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarLabelFormatLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -sticky nws %s",
               this->LabelTextPropertyWidget->GetWidgetName(),
               grid_settings);

  // Scalar bar : enable copy between text property widgets

  this->TitleTextPropertyWidget->ShowCopyOn();
  this->TitleTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the labels text properties to the title text properties.");
  ostrstream copy1;
  copy1 << "CopyValuesFrom " << this->LabelTextPropertyWidget->GetTclName() << ends;
  this->TitleTextPropertyWidget->GetCopyButton()->SetCommand(
    this->TitleTextPropertyWidget, copy1.str());
  copy1.rdbuf()->freeze(0);

  this->LabelTextPropertyWidget->ShowCopyOn();
  this->LabelTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the title text properties to the labels text properties.");
  ostrstream copy2;
  copy2 << "CopyValuesFrom " << this->TitleTextPropertyWidget->GetTclName() << ends;
  this->LabelTextPropertyWidget->GetCopyButton()->SetCommand(
    this->LabelTextPropertyWidget, copy2.str());
  copy2.rdbuf()->freeze(0);

  // Scalar bar: synchronize all those grids to have them aligned

  const char *widgets[2];
  widgets[0] = this->ScalarBarTitleFrame->GetWidgetName();
  widgets[1] = this->ScalarBarLabelFormatFrame->GetWidgetName();

  int weights[2];
  weights[0] = 0;
  weights[1] = 1;

  float factors[2];
  factors[0] = 1.3;
  factors[1] = 1.0;

  vtkKWTkUtilities::SynchroniseGridsColumnMinimumSize(
    this->Application->GetMainInterp(), 2, widgets, factors, weights);

  // Scalar bar : pack/grid

  this->Script("pack %s -side top -fill y -anchor w",
               this->ScalarBarCheck->GetWidgetName());
  
  this->Script("pack %s %s -side top -expand t -fill both -anchor w",
               this->ScalarBarTitleFrame->GetWidgetName(),
               this->ScalarBarLabelFormatFrame->GetWidgetName());

  onchangecommand.rdbuf()->freeze(0);

  // Back button

  this->BackButton->SetParent(this);
  this->BackButton->Create(this->Application, "-text {Back}");
  this->BackButton->SetCommand(this, "BackButtonCallback");

  // Pack

  this->Script("pack %s %s %s -side top -anchor n -fill x -padx 2 -pady 2", 
               this->ColorMapFrame->GetWidgetName(), 
               this->ScalarBarFrame->GetWidgetName(),
               this->BackButton->GetWidgetName());

  //  this->Script("pack %s -padx 4", this->BackButton->GetWidgetName());

  this->SetColorSchemeToRedBlue();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::CreateParallelTclObjects(vtkPVApplication *pvApp)
{
  char tclName[100];
  
  this->vtkKWWidget::SetApplication(pvApp);
  
  sprintf(tclName, "LookupTable%d", this->InstanceCount);
  this->SetLookupTableTclName(tclName);
  this->LookupTable = static_cast<vtkLookupTable*>(
    pvApp->MakeTclObject("vtkLookupTable", this->LookupTableTclName));
  
  this->ScalarBar = vtkScalarBarWidget::New();
  this->ScalarBar->SetInteractor(
    this->PVRenderView->GetPVWindow()->GetGenericInteractor());
  this->ScalarBar->GetScalarBarActor()->GetPositionCoordinate()
    ->SetValue(0.87, 0.25);
  this->ScalarBar->GetScalarBarActor()->SetWidth(0.13);
  this->ScalarBar->GetScalarBarActor()->SetHeight(0.5);

  this->ScalarBarObserver = vtkScalarBarWidgetObserver::New();
  this->ScalarBarObserver->PVColorMap = this;
  this->ScalarBar->AddObserver(vtkCommand::InteractionEvent, 
                               this->ScalarBarObserver);
  this->ScalarBar->AddObserver(vtkCommand::StartInteractionEvent, 
                               this->ScalarBarObserver);
  this->ScalarBar->AddObserver(vtkCommand::EndInteractionEvent, 
                               this->ScalarBarObserver);

  this->UpdateScalarBarTitle();

  this->ScalarBar->GetScalarBarActor()->SetLookupTable(this->LookupTable);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::BackButtonCallback()
{
  if (this->PVRenderView == NULL)
    {
    return;
    }

  this->PVRenderView->GetPVWindow()->GetCurrentPVData()->UpdateProperties();

  this->PVRenderView->GetPVWindow()->ShowCurrentSourceProperties();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetArrayName(const char* str)
{
  if ( this->ArrayName == NULL && str == NULL) 
    { 
    return;
    }
  if ( this->ArrayName && str && (!strcmp(this->ArrayName,str))) 
    { 
    return;
    }
  if (this->ArrayName)
    {
    delete [] this->ArrayName;
    this->ArrayName = NULL;
    }
  if (str)
    {
    this->ArrayName = new char[strlen(str)+1];
    strcpy(this->ArrayName,str);
    }
  if (str)
    {
    char *tmp;
    tmp = new char[strlen(str)+128];
    sprintf(tmp, "Parameter: %s", str);
    this->ArrayNameLabel->SetLabel(tmp);
    delete [] tmp;
    }
  this->ResetScalarRange();
  this->Modified();
}

//----------------------------------------------------------------------------
int vtkPVColorMap::MatchArrayName(const char* str)
{
  if (str == NULL || this->ArrayName == NULL)
    {
    return 0;
    }
  if (strcmp(str, this->ArrayName) == 0)
    {
    return 1;
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarTitleEntryCallback()
{
  this->SetScalarBarTitle(this->ScalarBarTitleEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarTitle(const char* name)
{
  if (this->ScalarBarTitle == NULL && name == NULL) 
    { 
    return;
    }

  if (this->ScalarBarTitle && name && (!strcmp(this->ScalarBarTitle, name))) 
    { 
    return;
    }

  if (this->ScalarBarTitle) 
    { 
    delete [] this->ScalarBarTitle; 
    this->ScalarBarTitle = NULL;
    }

  if (name)
    {
    this->ScalarBarTitle = new char[strlen(name) + 1];
    strcpy(this->ScalarBarTitle, name);

    char *str;
    str = new char [strlen(name) + 128];
    sprintf(str, "GetPVColorMap {%s}", name);
    this->SetTraceReferenceCommand(str);
    delete [] str;
    } 

  this->ScalarBarTitleEntry->SetValue(name);

  this->AddTraceEntry("$kw(%s) SetScalarBarTitle {%s}", 
                      this->GetTclName(), name);

  this->UpdateScalarBarTitle();

  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarLabelFormatEntryCallback()
{
  this->SetScalarBarLabelFormat(this->ScalarBarLabelFormatEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarLabelFormat(const char* name)
{
  if (this->ScalarBarLabelFormat == NULL && name == NULL) 
    { 
    return;
    }

  if (this->ScalarBarLabelFormat && 
      name && 
      (!strcmp(this->ScalarBarLabelFormat,name))) 
    { 
    return;
    }

  if (this->ScalarBarLabelFormat) 
    { 
    delete [] this->ScalarBarLabelFormat; 
    this->ScalarBarLabelFormat = NULL;
    }

  if (name)
    {
    this->ScalarBarLabelFormat = new char[strlen(name) + 1];
    strcpy(this->ScalarBarLabelFormat,name);
    } 

  this->ScalarBarLabelFormatEntry->SetValue(name);

  this->AddTraceEntry("$kw(%s) SetScalarBarLabelFormat {%s}", 
                      this->GetTclName(), name);

  if (this->ScalarBar != NULL && this->ScalarBarLabelFormat != NULL)
    {
    this->ScalarBar->GetScalarBarActor()->SetLabelFormat(
      this->ScalarBarLabelFormat);
    if (this->PVRenderView)
      {
      this->PVRenderView->EventuallyRender();
      }
    }

  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateLookupTable()
{
  vtkPVApplication *pvApp = this->GetPVApplication();


  // The a hue is arbitrary, make is consistent
  // so we do not get unexpected interpolated hues.
  if (this->StartHSV[1] == 0.0)
    {
    this->StartHSV[0] = this->EndHSV[0];
    }
  if (this->EndHSV[1] == 0.0)
    {
    this->EndHSV[0] = this->StartHSV[0];
    }

  pvApp->BroadcastScript("%s SetNumberOfTableValues %d", this->LookupTableTclName,
                         this->NumberOfColors);

  pvApp->BroadcastScript("%s SetHueRange %f %f", this->LookupTableTclName,
                         this->StartHSV[0], this->EndHSV[0]);
  pvApp->BroadcastScript("%s SetSaturationRange %f %f", this->LookupTableTclName,
                         this->StartHSV[1], this->EndHSV[1]);
  pvApp->BroadcastScript("%s SetValueRange %f %f", this->LookupTableTclName,
                         this->StartHSV[2], this->EndHSV[2]);
  pvApp->BroadcastScript("%s Build",
                         this->LookupTableTclName);

  if (this->MapWidth > 0 && this->MapHeight > 0)
    {
    this->UpdateMap(this->MapWidth, this->MapHeight);
    }

  this->GetPVRenderView()->EventuallyRender();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::NumberOfColorsScaleCallback()
{
  this->NumberOfColors = (int)(this->NumberOfColorsScale->GetValue());
  this->UpdateLookupTable();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToRedBlue()
{
  this->StartHSV[0] = 0.0;
  this->StartHSV[1] = 1.0;
  this->StartHSV[2] = 1.0;
  this->EndHSV[0] = 0.66667;
  this->EndHSV[1] = 1.0;
  this->EndHSV[2] = 1.0;

  this->StartColorButton->SetColor(1.0, 0.0, 0.0);
  this->EndColorButton->SetColor(0.0, 0.0, 1.0);

  this->UpdateLookupTable();
  this->AddTraceEntry("$kw(%s) SetColorSchemeToRedBlue", this->GetTclName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToBlueRed()
{
  this->StartHSV[0] = 0.66667;
  this->StartHSV[1] = 1.0;
  this->StartHSV[2] = 1.0;
  this->EndHSV[0] = 0.0;
  this->EndHSV[1] = 1.0;
  this->EndHSV[2] = 1.0;

  this->StartColorButton->SetColor(0.0, 0.0, 1.0);
  this->EndColorButton->SetColor(1.0, 0.0, 0.0);

  this->UpdateLookupTable();
  this->AddTraceEntry("$kw(%s) SetColorSchemeToBlueRed", this->GetTclName());
}



//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToGrayscale()
{
  this->StartHSV[0] = 0.0;
  this->StartHSV[1] = 0.0;
  this->StartHSV[2] = 0.0;
  this->EndHSV[0] = 0.0;
  this->EndHSV[1] = 0.0;
  this->EndHSV[2] = 1.0;

  this->StartColorButton->SetColor(0.0, 0.0, 0.0);
  this->EndColorButton->SetColor(1.0, 1.0, 1.0);

  this->UpdateLookupTable();
  this->AddTraceEntry("$kw(%s) SetColorSchemeToGrayscale", this->GetTclName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::StartColorButtonCallback(float r, float g, float b)
{
  float rgb[3];
  float hsv[3];

  // Convert RGB to HSV.
  rgb[0] = r;
  rgb[1] = g;
  rgb[2] = b;
  this->RGBToHSV(rgb, hsv);

  this->StartHSV[0] = hsv[0];
  this->StartHSV[1] = hsv[1];
  this->StartHSV[2] = hsv[2];

  this->UpdateLookupTable();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::EndColorButtonCallback(float r, float g, float b)
{
  float rgb[3];
  float hsv[3];

  // Convert RGB to HSV.
  rgb[0] = r;
  rgb[1] = g;
  rgb[2] = b;
  this->RGBToHSV(rgb, hsv);
  this->EndHSV[0] = hsv[0];
  this->EndHSV[1] = hsv[1];
  this->EndHSV[2] = hsv[2];

  this->UpdateLookupTable();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarCheckCallback()
{
  this->SetScalarBarVisibility(this->ScalarBarCheck->GetState());
  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarRange(float min, float max)
{
  this->SetScalarRangeInternal(min, max);
  this->AddTraceEntry("$kw(%s) SetScalarRange %f %f", this->GetTclName(),
                      min, max);
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarRangeInternal(float min, float max)
{
  vtkPVApplication *pvApp = this->GetPVApplication();

  this->ColorRangeMinEntry->SetValue(this->ScalarRange[0], 5);
  this->ColorRangeMaxEntry->SetValue(this->ScalarRange[1], 5);

  if (this->ScalarRange[0] == min && this->ScalarRange[1] == max)
    {
    return;
    }

  this->ScalarRange[0] = min;
  this->ScalarRange[1] = max;

  pvApp->BroadcastScript("%s SetTableRange %f %f", 
                         this->LookupTableTclName, min, max);
  //this->Script("%s Build", this->LookupTableTclName);
  //this->Script("%s Modified", this->LookupTableTclName);

  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetVectorComponent(int component, int numberOfComponents)
{
  vtkPVApplication *pvApp = this->GetPVApplication();

  if (this->VectorComponent == component && 
      this->NumberOfVectorComponents == numberOfComponents)
    {
    return;
    }

  this->VectorComponent = component;
  this->NumberOfVectorComponents = numberOfComponents;

  // Change the title of the scalar bar.
  this->UpdateScalarBarTitle();

  pvApp->BroadcastScript("%s SetVectorComponent %d", 
                         this->LookupTableTclName, component);

  this->Modified();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::ResetScalarRange()
{
  float range[2];
  float tmp[2];
  vtkPVSourceCollection *sourceList;
  vtkPVSource *pvs;
  vtkPVData *pvd;

  if (this->Application == NULL || this->PVRenderView == NULL)
    {
    vtkErrorMacro("Trying to reset scalar range without application and view.");
    return;
    }

  range[0] = VTK_LARGE_FLOAT;
  range[1] = -VTK_LARGE_FLOAT;

  // Compute global scalar range ...
  sourceList = this->PVRenderView->GetPVWindow()->GetSourceList("Sources");
  sourceList->InitTraversal();
  while ( (pvs = sourceList->GetNextPVSource()) )
    {
    pvd = pvs->GetPVOutput();
    // For point data ...
    pvd->GetArrayComponentRange(tmp, 1, this->ArrayName, 0);
    if (tmp[0] < range[0])
      {
      range[0] = tmp[0];
      }
    if (tmp[1] > range[1])
      {
      range[1] = tmp[1];
      }
    // For cell data ...
    pvd->GetArrayComponentRange(tmp, 0, this->ArrayName, 0);
    if (tmp[0] < range[0])
      {
      range[0] = tmp[0];
      }
    if (tmp[1] > range[1])
      {
      range[1] = tmp[1];
      }
    }

  if (range[1] < range[0])
    {
    range[0] = 0.0;
    range[1] = 1.0;
    }

  this->SetScalarRange(range[0], range[1]);

  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}


//----------------------------------------------------------------------------
vtkPVApplication* vtkPVColorMap::GetPVApplication()
{
  if (this->Application == NULL)
    {
    return NULL;
    }
  
  if (this->Application->IsA("vtkPVApplication"))
    {  
    return (vtkPVApplication*)(this->Application);
    }
  else
    {
    vtkErrorMacro("Bad typecast");
    return NULL;
    } 
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarVisibility(int val)
{
  vtkRenderer *ren;

  if (this->ScalarBarVisibility == val)
    {
    return;
    }
  this->ScalarBarVisibility = val;
  
  // Make sure the UI is up to date.
  if (val)
    {
    this->ScalarBarCheck->SetState(1);
    }
  else
    {
    this->ScalarBarCheck->SetState(0);
    }


  if (!this->GetPVRenderView())
    {
    return;
    }
  
  ren = this->GetPVRenderView()->GetRenderer();
  
  if (ren == NULL)
    {
    return;
    }
  
  // I am going to add and remove it from the renderer instead of using
  // visibility.  Composites should really have multiple props.
  
  if (ren)
    {
    if (val)
      {
      this->ScalarBar->SetEnabled(1);
      // This is here in case process 0 has not geometry.  
      // We have to explicitly build the color map.
      this->LookupTable->Build();
      this->LookupTable->Modified();
      }
    else
      {
      this->ScalarBar->SetEnabled(0);
      }
    }

  this->AddTraceEntry("$kw(%s) SetScalarBarVisibility %d", this->GetTclName(),
                      val);

  this->Modified();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SaveInTclScript(ofstream *file, int interactiveFlag,
                                    int vtkFlag)
{
  if (this->VisitedFlag)
    {
    return;
    }
  this->VisitedFlag = 1;

  *file << "vtkLookupTable " << this->LookupTableTclName << endl;
  *file << "\t" << this->LookupTableTclName << " SetNumberOfTableValues " 
        << this->NumberOfColors << endl;
  *file << "\t" << this->LookupTableTclName << " SetHueRange " 
        << this->StartHSV[0] << " " << this->EndHSV[0] << endl;
  *file << "\t" << this->LookupTableTclName << " SetSaturationRange " 
        << this->StartHSV[1] << " " << this->EndHSV[1] << endl;
  *file << "\t" << this->LookupTableTclName << " SetValueRange " 
        << this->StartHSV[2] << " " << this->EndHSV[2] << endl;
  *file << "\t" << this->LookupTableTclName << " SetTableRange "
        << this->ScalarRange[0] << " " << this->ScalarRange[1] << endl;
  *file << "\t" << this->LookupTableTclName << " SetVectorComponent " 
        << this->VectorComponent << endl;
  *file << "\t" << this->LookupTableTclName << " Build" << endl;

  if (this->ScalarBarVisibility)
    {
    char scalarBarTclName[128];
    sprintf(scalarBarTclName, "ScalarBar%d", this->InstanceCount);
    ostrstream actor;

    if (interactiveFlag && vtkFlag)
      {
      *file << "vtkScalarBarWidget " << scalarBarTclName << "\n";
      *file << "\t" << scalarBarTclName << " SetInteractor iren" << "\n";
      actor << "[" << scalarBarTclName << " GetScalarBarActor]" << ends;
      }
    else
      {
      *file << "vtkScalarBarActor " << scalarBarTclName << "\n";
      actor << scalarBarTclName << ends;
      }

    *file << "\t" << actor.str() << " SetLookupTable " 
          << this->LookupTableTclName << "\n";

    *file << "\t" << actor.str() << " SetOrientation "
          << this->ScalarBar->GetScalarBarActor()->GetOrientation() << "\n";

    *file << "\t" << actor.str() << " SetWidth " 
          << this->ScalarBar->GetScalarBarActor()->GetWidth() << "\n";

    *file << "\t" << actor.str() << " SetHeight " 
          << this->ScalarBar->GetScalarBarActor()->GetHeight() << "\n";

    const float *pos = 
     this->ScalarBar->GetScalarBarActor()->GetPositionCoordinate()->GetValue();
    *file << "\t[" << actor.str() << " GetPositionCoordinate] SetValue " 
          << pos[0] << " " << pos[1] << "\n";

    *file << "\t" << actor.str() << " SetTitle {" 
          << this->ScalarBar->GetScalarBarActor()->GetTitle() << "}\n";

    *file << "\t" << actor.str() << " SetLabelFormat {" 
          << this->ScalarBar->GetScalarBarActor()->GetLabelFormat() << "}\n";

    ostrstream ttprop, tlprop;
    ttprop << "[" << actor.str() << " GetTitleTextProperty]" << ends;
    this->TitleTextPropertyWidget->SaveInTclScript(file, ttprop.str());
    ttprop.rdbuf()->freeze(0);

    tlprop << "[" << actor.str() << " GetLabelTextProperty]" << ends;
    this->LabelTextPropertyWidget->SaveInTclScript(file, tlprop.str());
    tlprop.rdbuf()->freeze(0);

    if (interactiveFlag && vtkFlag)
      {
      *file << "\t" << scalarBarTclName << " EnabledOn\n";
      }
    else
      {
      *file << "Ren1 AddActor " << scalarBarTclName << endl;
      }

    actor.rdbuf()->freeze(0);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateScalarBarTitle()
{
  if (this->ScalarBar == NULL || this->ScalarBarTitle == NULL)
    {
    return;
    }

  if (this->NumberOfVectorComponents == 3)
    {
    if (this->VectorComponent == 0)
      {
      ostrstream ostr;
      ostr << this->ScalarBarTitle << " X" << ends;
      this->ScalarBar->GetScalarBarActor()->SetTitle(ostr.str());
      ostr.rdbuf()->freeze(0);
      }
    if (this->VectorComponent == 1)
      {
      ostrstream ostr;
      ostr << this->ScalarBarTitle << " Y" << ends;
      this->ScalarBar->GetScalarBarActor()->SetTitle(ostr.str());
      ostr.rdbuf()->freeze(0);
      }
    if (this->VectorComponent == 2)
      {
      ostrstream ostr;
      ostr << this->ScalarBarTitle << " Z" << ends;
      this->ScalarBar->GetScalarBarActor()->SetTitle(ostr.str());
      ostr.rdbuf()->freeze(0);
      }
    }
  else if (this->NumberOfVectorComponents > 1)
    {
    ostrstream ostr;
    ostr << this->ScalarBarTitle << " " << this->VectorComponent << ends;
    this->ScalarBar->GetScalarBarActor()->SetTitle(ostr.str());
    ostr.rdbuf()->freeze(0);
    }
  else
    {
    ostrstream ostr;
    ostr << this->ScalarBarTitle<< ends;
    this->ScalarBar->GetScalarBarActor()->SetTitle(ostr.str());
    ostr.rdbuf()->freeze(0);
    }
  if (this->PVRenderView)
    {
    this->PVRenderView->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ColorRangeEntryCallback()
{
  float min, max;

  min = this->ColorRangeMinEntry->GetValueAsFloat();
  max = this->ColorRangeMaxEntry->GetValueAsFloat();
  
  // Avoid the bad range error
  if (max <= min)
    {
    max = min + 0.00001;
    }

  this->SetScalarRange(min, max);
  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::RGBToHSV(float rgb[3], float hsv[3])
{
  float hue = 0;
  float sat = 0;
  float val = 0;
  float lx, ly, lz;

  if (rgb[0] <= 0.0 && rgb[1] <= 0.0 && rgb[2] <= 0.0)
    {
    hsv[0] = 0.0;
    hsv[1] = 0.0;
    hsv[2] = 0.0;
    return;
    }

  if (rgb[0] == rgb[1] && rgb[1] == rgb[2])
    {
    hsv[2] = rgb[0];
    hsv[1] = 0.0;
    hsv[0] = 0.0;
    return;
    }

  if (rgb[0] >= rgb[1] && rgb[1] >= rgb[2])
    { // case 0
    val = rgb[0];
    lz = rgb[1];
    lx = rgb[2];
    sat = 1.0 - (lx/val);
    hue = (0.0 + (1.0 - ((1.0 - (lz/val))/sat)))/6.0;
    }
  else if (rgb[1] >= rgb[0] && rgb[0] >= rgb[2])
    { // case 1
    ly = rgb[0];
    val = rgb[1];
    lx = rgb[2];
    sat = 1.0 - (lx/val);
    hue = (1.0 + ((1.0 - (ly/val))/sat))/6.0;
    }
  else if (rgb[1] >= rgb[2] && rgb[2] >= rgb[0])
    { // case 2
    lx = rgb[0];
    val = rgb[1];
    lz = rgb[2];
    sat = 1.0 - (lx/val);
    hue = (2.0 + (1.0 - ((1.0 - (lz/val))/sat)))/6.0;
    }
  else if (rgb[2] >= rgb[1] && rgb[1] >= rgb[0])
    { // case 3
    lx = rgb[0];
    ly = rgb[1];
    val = rgb[2];
    sat = 1.0 - (lx/val);
    hue = (3.0 + ((1.0 - (ly/val))/sat))/6.0;
    }
  else if (rgb[2] >= rgb[0] && rgb[0] >= rgb[1])
    { // case 4
    lz = rgb[0];
    lx = rgb[1];
    val = rgb[2];
    sat = 1.0 - (lx/val);
    hue = (4.0 + (1.0 - ((1.0 - (lz/val))/sat)))/6.0;
    }
  else if (rgb[0] >= rgb[2] && rgb[2] >= rgb[1])
    { // case 5
    val = rgb[0];
    lx = rgb[1];
    ly = rgb[2];
    sat = 1.0 - (lx/val);
    hue = (5.0 + ((1.0 - (ly/val))/sat))/6.0;
    }
  hsv[0] = hue;
  hsv[1] = sat;
  hsv[2] = val;
}


//----------------------------------------------------------------------------
void vtkPVColorMap::MapConfigureCallback(int width, int height)
{
  this->UpdateMap(width, height);
}


//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateMap(int width, int height)
{
  int size;
  int i, j;
  float *range;
  float val, step;
  unsigned char *rgba;  
  unsigned char *ptr;  

  size = width*height;
  if (this->MapDataSize < size)
    {
    if (this->MapData)
      {
      delete [] this->MapData;
      }
    this->MapData = new unsigned char[size*3];
    this->MapDataSize = size;
    }
  this->MapWidth = width;
  this->MapHeight = height;

  if (this->LookupTable == NULL)
    {
    return;
    }

  range = this->LookupTable->GetRange();
  step = (range[1]-range[0])/(float)(width);
  ptr = this->MapData;
  for (j = 0; j < height; ++j)
    {
    for (i = 0; i < width; ++i)
      {
      val = range[0] + ((float)(i)*step);
      rgba = this->LookupTable->MapValue(val);
      
      ptr[0] = rgba[0];
      ptr[1] = rgba[1];
      ptr[2] = rgba[2];
      ptr += 3;
      }
    }

  if (size > 0)
    {
    this->Map->SetImageData(this->MapData, width, height, 3);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarPosition1(float x, float y)
{
  vtkScalarBarActor* sact = this->ScalarBar->GetScalarBarActor();
  sact->GetPositionCoordinate()->SetValue(x, y);
  this->AddTraceEntry("$kw(%s) SetScalarBarPosition1 %f %f", 
                      this->GetTclName(), x, y);
  this->PVRenderView->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarPosition2(float x, float y)
{
  vtkScalarBarActor* sact = this->ScalarBar->GetScalarBarActor();
  sact->GetPosition2Coordinate()->SetValue(x,y);
  this->AddTraceEntry("$kw(%s) SetScalarBarPosition2 %f %f", 
                      this->GetTclName(), x, y);
  this->PVRenderView->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarOrientation(int o)
{
  vtkScalarBarActor* sact = this->ScalarBar->GetScalarBarActor();
  sact->SetOrientation(o);
  this->AddTraceEntry("$kw(%s) SetScalarBarOrientation %d", 
                      this->GetTclName(), o);
  this->PVRenderView->EventuallyRender();  
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ExecuteEvent(vtkObject* vtkNotUsed(wdg), 
                                 unsigned long event,  
                                 void* vtkNotUsed(calldata))
{
  switch ( event )
    {
    case vtkCommand::StartInteractionEvent:
      this->PVRenderView->GetPVWindow()->InteractionOn();
      break;
    case vtkCommand::EndInteractionEvent:
      this->PVRenderView->GetPVWindow()->InteractionOff();
      this->PVRenderView->EventuallyRender();
      vtkScalarBarActor* sact = this->ScalarBar->GetScalarBarActor();
      float *pos1 = sact->GetPositionCoordinate()->GetValue();
      float *pos2 = sact->GetPosition2Coordinate()->GetValue();
      this->AddTraceEntry("$kw(%s) SetScalarBarPosition1 %f %f", 
                          this->GetTclName(), pos1[0], pos1[1]);
      this->AddTraceEntry("$kw(%s) SetScalarBarPosition2 %f %f", 
                          this->GetTclName(), pos2[0], pos2[1]);
      this->AddTraceEntry("$kw(%s) SetScalarBarOrientation %d",
                          this->GetTclName(), sact->GetOrientation());
      break;
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "ScalarBarTitle: " 
     << (this->ScalarBarTitle ? this->ScalarBarTitle : "none" )
     << endl;
  os << indent << "ScalarBarLabelFormat: " 
     << (this->ScalarBarLabelFormat ? this->ScalarBarLabelFormat : "none" )
     << endl;
  os << indent << "ArrayName: " 
     << (this->ArrayName ? this->ArrayName : "none" )
     << endl;
  os << indent << "VectorComponent: " << this->VectorComponent << endl;
  os << indent << "LookupTableTclName: " 
     << (this->LookupTableTclName ? this->LookupTableTclName : "none" )
     << endl;
  os << indent << "ScalarBar: " << this->ScalarBar << endl;
  os << indent << "ScalarBarCheck: " << this->ScalarBarCheck << endl;
  os << indent << "TitleTextPropertyWidget: " << this->TitleTextPropertyWidget << endl;
  os << indent << "LabelTextPropertyWidget: " << this->LabelTextPropertyWidget << endl;
  
  os << indent << "ScalarBarVisibility: " << this->ScalarBarVisibility << endl;

  os << indent << "ScalarRange: " << this->ScalarRange[0] << ", "
     << this->ScalarRange[1] << endl;
  os << indent << "VisitedFlag: " << this->VisitedFlag << endl;
  os << indent << "ScalarBarCheck: " << this->ScalarBarCheck << endl;
}
