/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#pragma once

#include "cmConfigure.h" // IWYU pragma: keep

#include <cstddef>
#include <memory>
#include <string>

#include <cm3p/uv.h>

enum class cmCTestJobServerResult
{
  // Acquire/Release succeeded
  Success,
  // Acquire/Release failed with a non-recoverable error
  Fatal
};

// Interface for a jobserver client.
//
// The jobserver should acquire job slots in the background
class cmCTestJobServerClient
{
public:
  virtual ~cmCTestJobServerClient() = default;

  /**
   * @brief Returns the number of job slots available
   * @return Number of job slots available or -1 if the connection has been
   * invalidated
   */
  virtual ssize_t Ready() const = 0;

  /**
   * @brief Acquire marks n job slots as used.
   * @param n Number of job slots to acquire
   * @return Success or Fatal
   */
  virtual cmCTestJobServerResult Acquire(size_t n) = 0;

  /**
   * @brief Releases n job slots to the jobserver.
   * @param n Number of job slots to release
   * @return Success or Fatal
   */
  virtual cmCTestJobServerResult Release(size_t n) = 0;

  virtual void SolveDeadlock() = 0;

  /**
   * @brief Releases all job slots to the jobserver.
   */
  virtual void Clear() = 0;

  /**
   * @brief Stop receiving more job slots from the jobserver.
   */
  virtual void Stop() = 0;

  // Name of the jobserver client implementation
  virtual std::string_view Name() const = 0;

  // If available connect to the environment jobserver
  // If there is no jobserver available this returns a local client
  static std::unique_ptr<cmCTestJobServerClient> Connect(uv_loop_t* loop,
                                                         size_t maxJobs);
};

// A local jobserver client that tracks the number of processes running
class cmCTestJobServerLocal : public cmCTestJobServerClient
{
private:
  size_t MaxJobs;
  size_t UsedJobs;

public:
  cmCTestJobServerLocal(size_t maxJobs)
    : MaxJobs(maxJobs)
    , UsedJobs(0)
  {
  }
  ~cmCTestJobServerLocal() override = default;

  ssize_t Ready() const override;
  cmCTestJobServerResult Acquire(size_t n) override;
  cmCTestJobServerResult Release(size_t n) override;
  void ReleaseUnused() override{};
  void Clear() override{};
  std::string_view Name() const override { return std::string_view("local"); }
};

#ifdef _WIN32
#  include <windows.h>

// Windows jobserver client implementation
// https://www.gnu.org/software/make/manual/html_node/Windows-Jobserver.html
class cmCTestJobServerWindows : public cmCTestJobServerClient
{
private:
  size_t MaxJobs;
  HANDLE semephore;
  // Units allocated from the server
  size_t AcquiredJobs;
  // Units allocated to tests
  size_t UsedJobs;

public:
  cmCTestJobServerWindows(size_t maxJobs);
  ~cmCTestJobServerWindows() override;

  bool Connect(const std::string& auth);
  bool Connect();

  cmCTestJobServerResult Acquire(size_t n) override;
  cmCTestJobServerResult Release(size_t n) override;
  void Clear() override;
  std::string_view Name() const override
  {
    return std::string_view("windows");
  }
};
#else
#  include <vector>

// POSIX jobserver client implementation
// https://www.gnu.org/software/make/manual/html_node/POSIX-Jobserver.html
class cmCTestJobServerPosix : public cmCTestJobServerClient
{
private:
  // Either using 2 pipes (r,w) or a single pipe (fifo)
  union
  {
    struct
    {
      uv_pipe_t read_pipe;
      uv_pipe_t write_pipe;
    };
    uv_pipe_t fifo;
  };

  enum class ConnectionType
  {
    // read_pipe and write_pipe are valid
    Pipe,
    // fifo is valid
    Fifo
  };

  ConnectionType connection_type;

  // Get the writer handle depending on connect_type
  uv_pipe_t* GetWriter();

  // The maximum number of tokens that can possibly be allocated
  // min(upstream_max_jobs, allowed_local_jobs)
  size_t MaxJobs;

  // We must write back the same character we read, order doesn't matter
  std::vector<char> Tokens;
  // Tracks the number of tokens we have allocated to tests
  size_t UsedTokens;
  // 1 token is assumed to be available
  bool FreeTokenAvailable = false;
  bool valid = false;

  static void _on_read(uv_stream_t* handle, ssize_t nread,
                       const uv_buf_t* buf);

public:
  cmCTestJobServerPosix(size_t max_jobs);
  ~cmCTestJobServerPosix() override;

  /**
   * @brief Connect to jobserver using a pair of file descriptors
   * @return true if the connection was successful
   */
  bool Connect(uv_loop_t* loop, int rfd, int wfd);
  /**
   * @brief Connect to jobserver using a path to a fifo
   * @return true if the connection was successful
   */
  bool Connect(uv_loop_t* loop, const char* path);
  /**
   * @brief Connect to jobserver using the environment variables
   * @return true if the connection was successful
   */
  bool Connect(uv_loop_t* loop);

  ssize_t Ready() const override;
  void Stop() override;

  cmCTestJobServerResult Acquire(size_t n) override;
  cmCTestJobServerResult Release(size_t n) override;
  void SolveDeadlock() override;

  void Clear() override;
  std::string_view Name() const override { return std::string_view("posix"); }
};
#endif
