/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestJobServerClient.h"

#include <algorithm>
#include <iostream>
#include <string>

#include <cm/memory>
#include <cm/optional>

#include <cm3p/uv.h>

#include "cmSystemTools.h"

std::unique_ptr<cmCTestJobServerClient> cmCTestJobServerClient::Connect(
  uv_loop_t* loop, size_t maxJobs)
{
  bool success = false;

#ifdef _WIN32
  std::unique_ptr<cmCTestJobServerWindows> client =
    cm::make_unique<cmCTestJobServerWindows>(maxJobs);
  success = client->Connect();
#else
  std::unique_ptr<cmCTestJobServerPosix> client =
    cm::make_unique<cmCTestJobServerPosix>(maxJobs);
  success = client->Connect(loop);
#endif

  if (success) {
    return client;
  }

  return cm::make_unique<cmCTestJobServerLocal>(maxJobs);
}

ssize_t cmCTestJobServerLocal::Ready() const
{
  return this->MaxJobs - this->UsedJobs;
}

cmCTestJobServerResult cmCTestJobServerLocal::Acquire(size_t n)
{
  this->UsedJobs += n;
  return cmCTestJobServerResult::Success;
}

cmCTestJobServerResult cmCTestJobServerLocal::Release(size_t n)
{
  this->UsedJobs -= n;
  return cmCTestJobServerResult::Success;
}

#ifdef _WIN32
#  include <codecvt>
#  include <locale>

#  include <windows.h>

cmCTestJobServerWindows::cmCTestJobServerWindows(size_t maxJobs)
  : MaxJobs(maxJobs)
  , AcquiredJobs(0)
  , UsedJobs(0)
  , semephore(nullptr)
{
}

cmCTestJobServerWindows::~cmCTestJobServerWindows()
{
  // Release any acquired jobs.
  Clear();

  // Close the semaphore.
  if (this->semephore != nullptr) {
    CloseHandle(this->semephore);
  }
}

bool cmCTestJobServerWindows::Connect(const std::string& auth)
{
  std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>> converter;
  std::wstring wauth = converter.from_bytes(auth);

  this->semephore = OpenSemaphore(SEMAPHORE_ALL_ACCESS, FALSE, wauth.c_str());
  if (this->semephore == nullptr) {
    return false;
  }

  return true;
}

bool cmCTestJobServerWindows::Connect()
{
  // Read MAKEFLAGS from the environment.
  std::string makeflags;
  if (!cmSystemTools::GetEnv("MAKEFLAGS", makeflags)) {
    return false;
  }

  std::vector<std::string> args;
  cmSystemTools::ParseUnixCommandLine(makeflags.c_str(), args);

  // `-jN`
  cm::optional<size_t> maxJobs = cm::nullopt;
  // `--jobserver-auth=string`
  cm::optional<std::string> auth = cm::nullopt;

  for (const std::string& arg : args) {
    if (arg.compare(0, 2, "-j") == 0) {
      maxJobs = std::stoul(arg.substr(2));
    } else if (arg.compare(0, 17, "--jobserver-auth=") == 0) {
      auth = arg.substr(17);
    }
  }

  if (!maxJobs || !auth) {
    return false;
  }

  // The maximum number of jobs is the minimum of the upstream and local limits
  this->MaxJobs = std::min(this->MaxJobs, *maxJobs);
  return Connect(*auth);
}

cmCTestJobServerResult cmCTestJobServerWindows::Acquire(size_t n)
{
  size_t freeJobs = this->AcquiredJobs - this->UsedJobs;
  if (n <= freeJobs) {
    this->UsedJobs += n;
    return cmCTestJobServerResult::Success;
  }

  // We need to acquire more jobs.
  size_t neededJobs = n - freeJobs;
  for (size_t i = 0; i < neededJobs; i++) {
    switch (WaitForSingleObject(semephore, 0)) {
      case WAIT_OBJECT_0:
        this->AcquiredJobs++;
        break;
      case WAIT_TIMEOUT:
        return cmCTestJobServerResult::TryAgain;
      default:
        return cmCTestJobServerResult::Fatal;
    }
  }

  this->UsedJobs += n;
  return cmCTestJobServerResult::Success;
}

cmCTestJobServerResult cmCTestJobServerWindows::Release(size_t n)
{
  DWORD result = ReleaseSemaphore(semephore, n, nullptr);
  if (result == 0) {
    return cmCTestJobServerResult::Fatal;
  }

  this->AcquiredJobs -= n;
  this->UsedJobs -= n;
  return cmCTestJobServerResult::Success;
}

void cmCTestJobServerWindows::Clear()
{
  Release(this->AcquiredJobs);
  this->UsedJobs = 0;
  this->AcquiredJobs = 0;
}

#else
#  include <cerrno>
#  include <vector>

#  include <fcntl.h>
#  include <unistd.h>

cmCTestJobServerPosix::cmCTestJobServerPosix(size_t max_jobs)
  : MaxJobs(max_jobs)
  , Tokens(std::vector<char>())
  , UsedTokens(0)
{
}

cmCTestJobServerPosix::~cmCTestJobServerPosix()
{
  if (!valid) {
    return;
  }

  switch (this->connection_type) {
    case ConnectionType::Pipe:
      uv_close(reinterpret_cast<uv_handle_t*>(&read_pipe), nullptr);
      uv_close(reinterpret_cast<uv_handle_t*>(&write_pipe), nullptr);
      break;
    case ConnectionType::Fifo:
      uv_close(reinterpret_cast<uv_handle_t*>(&fifo), nullptr);
      break;
  }
}

uv_pipe_t* cmCTestJobServerPosix::GetWriter()
{
  switch (this->connection_type) {
    case ConnectionType::Pipe:
      return &this->write_pipe;
    case ConnectionType::Fifo:
      return &this->fifo;
  }
}

void cmCTestJobServerPosix::_on_read(uv_stream_t* handle, ssize_t nread,
                                     const uv_buf_t* buf)
{
  // Push the read character(s) onto the vector
  cmCTestJobServerPosix* client =
    static_cast<cmCTestJobServerPosix*>(handle->data);

  if (nread < 0) {
    client->valid = false;
    return;
  }

  client->Tokens.insert(client->Tokens.end(), buf->base, buf->base + nread);
}

bool cmCTestJobServerPosix::Connect(uv_loop_t* loop, int rfd, int wfd)
{
  this->connection_type = ConnectionType::Pipe;

  if (uv_pipe_init(loop, &read_pipe, 0) != 0) {
    return false;
  }

  if (uv_pipe_init(loop, &write_pipe, 0) != 0) {
    return false;
  }

  if (uv_pipe_open(&read_pipe, rfd) != 0) {
    return false;
  }

  if (uv_pipe_open(&write_pipe, wfd) != 0) {
    return false;
  }

  // Verify that the read side is readable, and the write side is writable
  if (!uv_is_readable(reinterpret_cast<uv_stream_t*>(&read_pipe)) ||
      !uv_is_writable(reinterpret_cast<uv_stream_t*>(&write_pipe))) {
    return false;
  }

  read_pipe.data = this;
  write_pipe.data = this;

  uv_read_start(
    (uv_stream_t*)&read_pipe,
    [](uv_handle_t* /* handle */, size_t /* suggested_size */, uv_buf_t* buf) {
      buf->base = new char[1];
      buf->len = 1;
    },
    _on_read);

  valid = true;
  return true;
}

bool cmCTestJobServerPosix::Connect(uv_loop_t* loop, const char* path)
{
  this->connection_type = ConnectionType::Fifo;

  if (uv_pipe_init(loop, &fifo, 0) != 0) {
    return false;
  }

  // Open for reading and writing
  int fd = open(path, O_RDWR);
  if (fd == -1) {
    return false;
  }

  if (uv_pipe_open(&fifo, fd) != 0) {
    return false;
  }

  fifo.data = this;

  uv_read_start(
    (uv_stream_t*)&fifo,
    [](uv_handle_t* /* handle */, size_t /* suggested_size */, uv_buf_t* buf) {
      buf->base = new char[1];
      buf->len = 1;
    },
    _on_read);

  valid = true;
  return true;
}

bool cmCTestJobServerPosix::Connect(uv_loop_t* loop)
{
  const cm::string_view jobserver_auth = "--jobserver-auth=";
  const cm::string_view jobserver_fds = "--jobserver-fds=";
  const cm::string_view fifo = "fifo:";

  std::string makeflags;
  if (!cmSystemTools::GetEnv("MAKEFLAGS", makeflags)) {
    return false;
  }

  // `-j[N]`
  cm::optional<size_t> maxJobs = cm::nullopt;
  // `--jobserver-auth=string` or `--jobserver-fds=string`.
  cm::optional<std::string> auth = cm::nullopt;

  std::vector<std::string> args;
  cmSystemTools::ParseUnixCommandLine(makeflags.c_str(), args);
  for (const std::string& arg : args) {
    if (arg.compare(0, 2, "-j") == 0) {
      try {
        maxJobs = std::stoul(arg.substr(2));
      } catch (...) {
      }
    } else if (arg.compare(0, jobserver_auth.length(), jobserver_auth) == 0) {
      auth = arg.substr(jobserver_auth.length());
    } else if (arg.compare(0, jobserver_fds.length(), jobserver_fds) == 0) {
      auth = arg.substr(jobserver_fds.length());
    }
  }

  if (!auth) {
    return false;
  }

  // The maximum number of jobs is the minimum of the upstream and local limits
  if (maxJobs) {
    this->MaxJobs = std::min(this->MaxJobs, *maxJobs);
  }

  // fifo:PATH
  if (auth->compare(0, fifo.length(), fifo) == 0) {
    return Connect(loop, auth->c_str() + fifo.length());
  }

  // Must be reader,writer
  size_t reader;
  size_t writer;
  try {
    size_t comma = auth->find(',');
    if (comma == std::string::npos) {
      return false;
    }

    reader = std::stoul(auth->substr(0, comma));
    writer = std::stoul(auth->substr(comma + 1));
  } catch (...) {
    return false;
  }

  return Connect(loop, reader, writer);
}

ssize_t cmCTestJobServerPosix::Ready() const
{
  if (!valid) {
    return -1;
  }

  return Tokens.size() + FreeTokenAvailable;
}

void cmCTestJobServerPosix::Stop()
{
  if (!valid) {
    return;
  }

  switch (this->connection_type) {
    case ConnectionType::Pipe:
      uv_read_stop(reinterpret_cast<uv_stream_t*>(&read_pipe));
      break;
    case ConnectionType::Fifo:
      uv_read_stop(reinterpret_cast<uv_stream_t*>(&fifo));
      break;
  }
}

cmCTestJobServerResult cmCTestJobServerPosix::Acquire(size_t n)
{
  // Prioritize acquiring the free job slot
  if (FreeTokenAvailable) {
    FreeTokenAvailable = false;
    n--;
  }

  // Mark tokens as used
  this->UsedTokens += n;
  return cmCTestJobServerResult::Success;
}

// A small wrapper around uv_write_t to make it easier to clean up
struct write_context_t
{
  uv_write_t req;
  uv_buf_t buf;
  // Keep a pointer to the client to invalidate it if the write fails
  cmCTestJobServerPosix* client;

  ~write_context_t() { delete[] buf.base; }
};

cmCTestJobServerResult cmCTestJobServerPosix::Release(size_t n)
{
  if (!valid) {
    return cmCTestJobServerResult::Fatal;
  }

  // Prioritize releasing the free job slot
  if (FreeTokenAvailable) {
    FreeTokenAvailable = false;
    n--;
  }

  // Release tokens
  if (UsedTokens < n) {
    return cmCTestJobServerResult::Fatal;
  }

  write_context_t* context = new write_context_t;
  context->buf = uv_buf_init(new char[n], n);
  context->client = this;

  // Write the tokens back to the jobserver
  uv_write(&context->req, (uv_stream_t*)GetWriter(), &context->buf, 1,
           [](uv_write_t* req, int status) {
             write_context_t* context =
               reinterpret_cast<write_context_t*>(req);
             cmCTestJobServerPosix* client = context->client;

             if (status < 0) {
               client->valid = false;
               return;
             }

             // Release tokens
             client->UsedTokens -= context->buf.len;

             // Free the buffer
             delete context;
           });

  return cmCTestJobServerResult::Success;
}

void cmCTestJobServerPosix::SolveDeadlock()
{
  // Release all unused tokens

  unsigned int milliseconds = (cmSystemTools::RandomSeed() % 5 + 1) * 1000;
}

void cmCTestJobServerPosix::Clear()
{
  cmCTestJobServerPosix::Release(this->UsedTokens);
}
#endif
