#include <stdio.h>
#include <stdlib.h>
#include <string.h>

const int MAX_MESSAGE_LENGTH = 1024;
const char* usage = "Usage: %s [--present|--absent] <output_file>\n";

// Extracts --jobserver-auth=<string> from MAKEFLAGS
// The returned pointer points to the start of <string>
// Returns NULL if MAKEFLAGS is not set or does not contain --jobserver-auth
char* jobserver_auth(char* message)
{
  const char* jobserver_auth = "--jobserver-auth=";

  char* makeflags = getenv("MAKEFLAGS");
  if (makeflags == NULL) {
    strncpy(message, "MAKEFLAGS not set", MAX_MESSAGE_LENGTH);
    return NULL;
  }

  char* jobserver = strstr(makeflags, jobserver_auth);
  if (jobserver == NULL) {
    strncpy(message, "MAKEFLAGS does not contain --jobserver-auth", MAX_MESSAGE_LENGTH);
    return NULL;
  }

  return jobserver + strlen(jobserver_auth);
}

#ifdef _WIN32
#include <windows.h>

int windows_semaphore(const char* semaphore, char* message)
{
  // Open the semaphore
  HANDLE hSemaphore = OpenSemaphoreA(SEMAPHORE_ALL_ACCESS, FALSE, semaphore);

  if (hSemaphore == NULL) {
    strncpy(message, "Error opening semaphore", MAX_MESSAGE_LENGTH);
    return 1;
  }

  strncpy(message, "Success", MAX_MESSAGE_LENGTH);
  return 0;
}
#else
#include <fcntl.h>
#include <errno.h>

int test_fd(int read_fd, int write_fd, char* message)
{
  // Detect if the file descriptors are valid
  int read_good = fcntl(read_fd, F_GETFD) != -1;
  int read_error = errno;

  int write_good = fcntl(write_fd, F_GETFD) != -1;
  int write_error = errno;

  if (!read_good || !write_good) {
    snprintf(message, MAX_MESSAGE_LENGTH, "Error opening file descriptors: %d (%s), %d (%s)\n", read_fd, strerror(read_error), write_fd, strerror(write_error));
    return 1;
  }

  snprintf(message, MAX_MESSAGE_LENGTH, "Success\n");
  return 0;
}

int posix(const char* jobserver, char* message)
{
  int read_fd, write_fd;

  // First try to parse as "R,W" file descriptors
  if (sscanf(jobserver, "%d,%d", &read_fd, &write_fd) == 2) {
    return test_fd(read_fd, write_fd, message);
  }

  // Then try to parse as "fifo:PATH"
  if (strncmp(jobserver, "fifo:", 5) == 0) {
    const char* path = jobserver + 5;
    read_fd = open(path, O_RDONLY);
    write_fd = open(path, O_WRONLY);
    return test_fd(read_fd, write_fd, message);
  }

  // We don't understand the format
  snprintf(message, MAX_MESSAGE_LENGTH, "Unrecognized jobserver format: %s\n", jobserver);
  return 1;
}
#endif

// Takes 2 arguments:
// Either --present or --absent to indicate we expect the jobserver to be "present and valid", or "absent or invalid"  
//
// if `--present` is passed, the exit code will be 0 if the jobserver is present, 1 if it is absent
// if `--absent` is passed, the exit code will be 0 if the jobserver is absent, 1 if it is present
// in either case, if there is some fatal error (e.g the output file cannot be opened), the exit code will be 2
int main(int argc, char** argv)
{
  char message[MAX_MESSAGE_LENGTH];

  if (argc != 3) {
    fprintf(stderr, usage, argv[0]);
    return 2;
  }

  int expecting_present = strcmp(argv[1], "--present") == 0;
  int expecting_absent = strcmp(argv[1], "--absent") == 0;
  if (!expecting_present && !expecting_absent) {
    fprintf(stderr, usage, argv[0]);
    return 2;
  }

  char* output_file = argv[2];
  FILE* fp = fopen(output_file, "w");  
  if (fp == NULL) {
    fprintf(stderr, "Error opening output file: %s\n", output_file);
    return 2;
  }

  char* jobserver = jobserver_auth(message);
  if (jobserver == NULL) {
    if (expecting_absent) {
      fprintf(stdout, "Success\n");
      return 0;
    }

    fprintf(stderr, "%s\n", message);
    return 1;
  }

  int result;
#ifdef _WIN32
  result = windows_semaphore(jobserver, message);
#else
  result = posix(jobserver, message);
#endif

  if (result == 0 && expecting_present) {
    fprintf(stdout, "Success\n");
    return 0;
  }

  if (result == 1 && expecting_absent) {
    fprintf(stdout, "Success\n");
    return 0;
  }

  fprintf(stderr, "%s\n", message);
  return 1;
}