/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkOSPRayRendererNode.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkOSPRayRendererNode - links vtkRenderers to OSPRay
// .SECTION Description
// Translates vtkRenderer state into OSPRay rendering calls

#ifndef vtkOSPRayRendererNode_h
#define vtkOSPRayRendererNode_h

#include "vtkRenderingOSPRayModule.h" // For export macro
#include "vtkRendererNode.h"
#include <vector> // for ivars
#include <string>

class vtkRenderer;
class vtkInformationIntegerKey;
class vtkInformationStringKey;
// ospray forward decs so that someone does not need to include ospray.h
namespace osp {
struct Model;
struct Renderer;
struct Light;
struct Texture2D;
struct FrameBuffer;
}
typedef osp::Model *OSPModel;
typedef osp::Renderer *OSPRenderer;
typedef osp::Light *OSPLight;
typedef osp::FrameBuffer *OSPFrameBuffer;
typedef osp::Texture2D* OSPTexture2D;
typedef osp::FrameBuffer* OSPFrameBuffer;

class VTKRENDERINGOSPRAY_EXPORT vtkOSPRayRendererNode :
  public vtkRendererNode
{
public:
  static vtkOSPRayRendererNode* New();
  vtkTypeMacro(vtkOSPRayRendererNode, vtkRendererNode);
  void PrintSelf(ostream& os, vtkIndent indent);

  //Description:
  //Builds myself.
  virtual void Build(bool prepass);

  //Description:
  //Traverse graph in ospray's prefered order and render
  virtual void Render(bool prepass);

  //Description:
  //Put my results into the correct place in the provided pixel buffer.
  virtual void WriteLayer(unsigned char *buffer, float *zbuffer,
                          int buffx, int buffy, int layer);

  // virtual void Clear();

  //state beyond rendering core...

  //Description:
  //When present on renderer, controls the number of primary rays
  //shot per pixel
  //default is 1
  static vtkInformationIntegerKey* SAMPLES_PER_PIXEL();

  //Description:
  //Convenience method to set/get SAMPLES_PER_PIXEL on a vtkRenderer.
  static void SetSamplesPerPixel(int, vtkRenderer *renderer);
  static int GetSamplesPerPixel(vtkRenderer *renderer);

  //Description:
  //When present on renderer, controls the number of ambient occlusion
  //samples shot per hit.
  //default is 4
  static vtkInformationIntegerKey* AMBIENT_SAMPLES();
  //Description:
  //Convenience method to set/get AMBIENT_SAMPLES on a vtkRenderer.
  static void SetAmbientSamples(int, vtkRenderer *renderer);
  static int GetAmbientSamples(vtkRenderer *renderer);

  //Description:
  //When present on renderer, controls the number of ambient occlusion
  //samples shot per hit.
  //default is 4
  static vtkInformationIntegerKey* PATHTRACING();
  //Description:
  //Convenience method to set/get AMBIENT_SAMPLES on a vtkRenderer.
  static void SetPathTracing(bool, vtkRenderer *renderer);
  static bool GetPathTracing(vtkRenderer *renderer);

  //Description:
  //used to make the renderer add ospray's content onto GL rendered
  //content on the window
  static vtkInformationIntegerKey* COMPOSITE_ON_GL();
  //Description:
  //Convenience method to set/get COMPOSITE_ON_GL on a vtkRenderer.
  static void SetCompositeOnGL(int, vtkRenderer *renderer);
  static int GetCompositeOnGL(vtkRenderer *renderer);

  /// LEDIAEV
  //Description:
  //When present on renderer, controls the number of maximum bounces of the light.
  //default is 5
  static vtkInformationIntegerKey * PT_MAX_DEPTH();
  static void SetPTMaxDepth (int val, vtkRenderer *renderer);
  static int GetPTMaxDepth (vtkRenderer *renderer);

  static vtkInformationIntegerKey * PT_SHOW_ENV_BACKGROUND();
  static void SetShowEnvironmentBackground (int val, vtkRenderer *renderer);
  static bool GetShowEnvironmentBackground (vtkRenderer *renderer);

  static vtkInformationStringKey * ENV_IMAGE_FILE();
  static void SetEnvImageFile (std::string val, vtkRenderer *renderer);
  static const char * GetEnvImageFile (vtkRenderer *renderer);


  // Description:
  // Methods for other nodes to access
  OSPModel GetOModel() { return this->OModel; }
  OSPModel SetOModel(OSPModel model) { this->OModel = model; }
  OSPRenderer GetORenderer() { return this->ORenderer; }
  void AddLight(OSPLight light) {
    this->Lights.push_back(light); }

  // Description:
  // Get the last rendered ColorBuffer
  virtual unsigned char *GetBuffer() {
    return this->Buffer; }

  // Description:
  // Get the last rendered ZBuffer
  virtual float *GetZBuffer() {
    return this->ZBuffer; }

  // if you want to traverse your children in a specific order
  // or way override this method
  virtual void Traverse(int operation);

protected:
  vtkOSPRayRendererNode();
  ~vtkOSPRayRendererNode();

  void UpdateOSPRayRenderer();

  //internal structures
  unsigned char *Buffer;
  float *ZBuffer;

  OSPModel OModel;
  OSPRenderer ORenderer;
  OSPFrameBuffer OFrameBuffer;
  int ImageX, ImageY;
  std::vector<OSPLight> Lights;
  std::string RendererStr;
  int NumActors;
  bool ComputeDepth;
  bool Accumulate;
  bool CompositeOnGL;
  unsigned long CameraTime;
  bool SceneDirty;

private:
  vtkOSPRayRendererNode(const vtkOSPRayRendererNode&) VTK_DELETE_FUNCTION;
  void operator=(const vtkOSPRayRendererNode&) VTK_DELETE_FUNCTION;
};


// Include file for reading directory contents.
#include <dirent.h>
// Adding texture loading from ospray.
namespace ospcommon
{
#define OSPCOMMON_INTERFACE

  /*! Convenience class for handling file names and paths. */
  class FileName
  {
  public:

    /*! create an empty filename */
    OSPCOMMON_INTERFACE FileName();

    /*! create a valid filenam
    e from a string */
    OSPCOMMON_INTERFACE FileName(const char* filename);

    /*! create a valid filename from a string */
    OSPCOMMON_INTERFACE FileName(const std::string& filename);

    /*! returns path to home folder */
    OSPCOMMON_INTERFACE static FileName homeFolder();

    /*! returns path to executable */
    OSPCOMMON_INTERFACE static FileName executableFolder();

    /*! auto convert into a string */
    OSPCOMMON_INTERFACE operator std::string() const { return filename; }

    /*! returns a string of the filename */
    OSPCOMMON_INTERFACE const std::string str() const { return filename; }

    /*! returns a c-string of the filename */
    OSPCOMMON_INTERFACE const char* c_str() const { return filename.c_str(); }

    /*! returns the path of a filename */
    OSPCOMMON_INTERFACE FileName path() const;

    /*! returns the file of a filename  */
    OSPCOMMON_INTERFACE std::string base() const;

    /*! returns the base of a filename without extension */
    OSPCOMMON_INTERFACE std::string name() const;

    /*! returns the file extension */
    OSPCOMMON_INTERFACE std::string ext() const;

    /*! drops the file extension */
    OSPCOMMON_INTERFACE FileName dropExt() const;

    /*! replaces the file extension */
    OSPCOMMON_INTERFACE FileName setExt(const std::string& ext = "") const;

    /*! adds file extension */
    OSPCOMMON_INTERFACE FileName addExt(const std::string& ext = "") const;

    /*! concatenates two filenames to this/other */
    OSPCOMMON_INTERFACE FileName operator+( const FileName& other ) const;

    /*! concatenates two filenames to this/other */
    OSPCOMMON_INTERFACE FileName operator+( const std::string& other ) const;

    /*! removes the base from a filename (if possible) */
    OSPCOMMON_INTERFACE FileName operator-( const FileName& base ) const;

    /*! == operator */
    OSPCOMMON_INTERFACE friend bool operator==(const FileName& a, const FileName& b);

    /*! != operator */
    OSPCOMMON_INTERFACE friend bool operator!=(const FileName& a, const FileName& b);

    /*! output operator */
    OSPCOMMON_INTERFACE friend std::ostream& operator<<(std::ostream& cout, const FileName& filename);

  private:
    std::string filename;
  };
}

#include <vector>
#include <map>

template<typename T, int N, int ALIGN=0> struct vec_t
{
  typedef T scalar_t;
  typedef T Scalar;
};
template<typename T>
struct vec_t<T,2> {
  typedef T scalar_t;
  typedef T Scalar;

  inline vec_t() {};
  inline vec_t(scalar_t s) : x(s), y(s) {};
  inline vec_t(scalar_t x, scalar_t y) : x(x), y(y) {};
  inline vec_t(const vec_t<T,2> &o) : x(o.x), y(o.y) {}

  template<typename OT>
  explicit inline vec_t(const vec_t<OT,2> &o) : x(o.x), y(o.y) {}

  /*! return result of reduce_add() across all components */
  inline scalar_t sum() const { return x+y; }
  /*! return result of reduce_mul() across all components */
  inline scalar_t product() const { return x*y; }

  T x, y;
};
typedef vec_t<int32_t,2>  vec2i;

namespace ospray {

  namespace miniSG {
    using namespace ospcommon;

  struct Texture2D {// : public RefCount {
    Texture2D();

    int channels; //Number of color channels per pixel
    int depth;    //Bytes per color channel
    bool prefereLinear; //A linear texel format is preferred over sRGB
    int width;    //Pixels per row
    int height;   //Pixels per column
    void *data;   //Pointer to binary texture data
  };

  Texture2D *loadTexture(const std::string &path, const std::string &fileName, const bool prefereLinear = false);

  }
}

#endif
