/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkOSPRayRendererNode.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkOSPRayRendererNode.h"

#include "vtkCamera.h"
#include "vtkCollectionIterator.h"
#include "vtkInformation.h"
#include "vtkInformationIntegerKey.h"
#include "vtkInformationStringKey.h"
#include "vtkObjectFactory.h"
#include "vtkOSPRayActorNode.h"
#include "vtkOSPRayCameraNode.h"
#include "vtkOSPRayLightNode.h"
#include "vtkOSPRayVolumeNode.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkViewNodeCollection.h"
#include "vtkTexture.h"
#include "vtkImageData.h"
#include "vtkPointData.h"
#include "vtkDataArray.h"

#include "ospray/ospray.h"
#include "ospray/version.h"

#include <cmath>

// Adding texture loading from ospray.
#include <unistd.h>
namespace ospcommon
{
#ifdef _WIN32
const char path_sep = '\\';
#else
const char path_sep = '/';
#endif

std::string getExecutableFileName()
{
    char pid[32]; sprintf(pid, "/proc/%d/exe", getpid());
    char buf[1024];
    int bytes = readlink(pid, buf, sizeof(buf)-1);
    if (bytes != -1) buf[bytes] = '\0';
    return std::string(buf);
}

/*! create an empty filename */
FileName::FileName () {}

/*! create a valid filename from a string */
FileName::FileName (const char* in) {
    filename = in;
    for (size_t i=0; i<filename.size(); i++)
        if (filename[i] == '\\' || filename[i] == '/')
            filename[i] = path_sep;
    while (!filename.empty() && filename[filename.size()-1] == path_sep)
        filename.resize(filename.size()-1);
}

/*! create a valid filename from a string */
FileName::FileName (const std::string& in) {
    filename = in;
    for (size_t i=0; i<filename.size(); i++)
        if (filename[i] == '\\' || filename[i] == '/')
            filename[i] = path_sep;
    while (!filename.empty() && filename[filename.size()-1] == path_sep)
        filename.resize(filename.size()-1);
}

/*! returns path to home folder */
FileName FileName::homeFolder()
{
#ifdef _WIN32
    const char* home = getenv("UserProfile");
#else
    const char* home = getenv("HOME");
#endif
    if (home) return home;
    return "";
}

/*! returns path to executable */
FileName FileName::executableFolder() {
    return FileName(getExecutableFileName()).path();
}

/*! returns the path */
FileName FileName::path() const {
    size_t pos = filename.find_last_of(path_sep);
    if (pos == std::string::npos) return FileName();
    return filename.substr(0,pos);
}

/*! returns the basename */
std::string FileName::base() const {
    size_t pos = filename.find_last_of(path_sep);
    if (pos == std::string::npos) return filename;
    return filename.substr(pos+1);
}

/*! returns the extension */
std::string FileName::ext() const {
    size_t pos = filename.find_last_of('.');
    if (pos == std::string::npos) return "";
    return filename.substr(pos+1);
}

/*! returns the extension */
FileName FileName::dropExt() const {
    size_t pos = filename.find_last_of('.');
    if (pos == std::string::npos) return filename;
    return filename.substr(0,pos);
}

/*! returns the basename without extension */
std::string FileName::name() const {
    size_t start = filename.find_last_of(path_sep);
    if (start == std::string::npos) start = 0; else start++;
    size_t end = filename.find_last_of('.');
    if (end == std::string::npos || end < start) end = filename.size();
    return filename.substr(start, end - start);
}

/*! replaces the extension */
FileName FileName::setExt(const std::string& ext) const {
    size_t start = filename.find_last_of(path_sep);
    if (start == std::string::npos) start = 0; else start++;
    size_t end = filename.find_last_of('.');
    if (end == std::string::npos || end < start) return FileName(filename+ext);
    return FileName(filename.substr(0,end)+ext);
}

/*! adds the extension */
FileName FileName::addExt(const std::string& ext) const {
    return FileName(filename+ext);
}

/*! concatenates two filenames to this/other */
FileName FileName::operator +( const FileName& other ) const {
    if (filename == "") return FileName(other);
    else return FileName(filename + path_sep + other.filename);
}

/*! concatenates two filenames to this/other */
FileName FileName::operator +( const std::string& other ) const {
    return operator+(FileName(other));
}

/*! removes the base from a filename (if possible) */
FileName FileName::operator -( const FileName& base ) const {
    size_t pos = filename.find_first_of(base);
    if (pos == std::string::npos) return *this;
    return FileName(filename.substr(pos+1));
}

/*! == operator */
bool operator== (const FileName& a, const FileName& b) {
    return a.filename == b.filename;
}

/*! != operator */
bool operator!= (const FileName& a, const FileName& b) {
    return a.filename != b.filename;
}

/*! output operator */
std::ostream& operator<<(std::ostream& cout, const FileName& filename) {
    return cout << filename.filename;
}
}

namespace ospray {

  namespace miniSG {
Texture2D::Texture2D()
    : channels(0)
    , depth(0)
    , width(0)
    , height(0)
    , data(NULL)
{}

Texture2D *loadTexture(const std::string &path, const std::string &fileNameBase, const bool prefereLinear)
{
    const FileName fileName = path+"/"+fileNameBase;

    static std::map<std::string,Texture2D*> textureCache;
    if (textureCache.find(fileName.str()) != textureCache.end())
        return textureCache[fileName.str()];

    Texture2D *tex = NULL;
    const std::string ext = fileName.ext();
    if (ext == "ppm") {
        try {
            int rc, peekchar;

            // open file
            FILE *file = fopen(fileName.str().c_str(),"rb");
            const int LINESZ=10000;
            char lineBuf[LINESZ+1];

            if (!file)
                throw std::runtime_error("#osp:miniSG: could not open texture file '"+fileName.str()+"'.");

            // read format specifier:
            int format=0;
            fscanf(file,"P%i\n",&format);
            if (format != 6)
                throw std::runtime_error("#osp:miniSG: can currently load only binary P6 subformats for PPM texture files. "
                                         "Please report this bug at ospray.github.io.");

            // skip all comment lines
            peekchar = getc(file);
            while (peekchar == '#') {
                fgets(lineBuf,LINESZ,file);
                peekchar = getc(file);
            } ungetc(peekchar,file);

            // read width and height from first non-comment line
            int width=-1,height=-1;
            rc = fscanf(file,"%i %i\n",&width,&height);
            if (rc != 2)
                throw std::runtime_error("#osp:miniSG: could not parse width and height in P6 PPM file '"+fileName.str()+"'. "
                                                                                                                         "Please report this bug at ospray.github.io, and include named file to reproduce the error.");

            // skip all comment lines
            peekchar = getc(file);
            while (peekchar == '#') {
                fgets(lineBuf,LINESZ,file);
                peekchar = getc(file);
            } ungetc(peekchar,file);

            // read maxval
            int maxVal=-1;
            rc = fscanf(file,"%i",&maxVal);
            peekchar = getc(file);

            if (rc != 1)
                throw std::runtime_error("#osp:miniSG: could not parse maxval in P6 PPM file '"+fileName.str()+"'. "
                                                                                                               "Please report this bug at ospray.github.io, and include named file to reproduce the error.");
            if (maxVal != 255)
                throw std::runtime_error("#osp:miniSG: could not parse P6 PPM file '"+fileName.str()+"': currently supporting only maxVal=255 formats."
                                                                                                     "Please report this bug at ospray.github.io, and include named file to reproduce the error.");

            tex = new Texture2D;
            tex->width    = width;
            tex->height   = height;
            tex->channels = 3;
            tex->depth    = 1;
            tex->prefereLinear = prefereLinear;
            tex->data     = new unsigned char[width*height*3];
            fread(tex->data,width*height*3,1,file);
            // flip in y, because OSPRay's textures have the origin at the lower left corner
            unsigned char *texels = (unsigned char *)tex->data;
            for (size_t y=0; y < height/2; y++)
                for (size_t x=0; x < width*3; x++)
                    std::swap(texels[y*width*3+x], texels[(height-1-y)*width*3+x]);
        } catch(std::runtime_error e) {
            std::cerr << e.what() << std::endl;
        }
    } else if (ext == "pfm") {
        try {
            // Note: the PFM file specification does not support comments thus we don't skip any
            // http://netpbm.sourceforge.net/doc/pfm.html
            int rc = 0;
            FILE *file = fopen(fileName.str().c_str(), "rb");
            const int LINESZ = 10000;
            char lineBuf[LINESZ + 1];
            if (!file) {
                throw std::runtime_error("#osp:miniSG: could not open texture file '"+fileName.str()+"'.");
            }
            // read format specifier:
            // PF: color floating point image
            // Pf: grayscae floating point image
            char format[2] = {0};
            fscanf(file, "%c%c\n", &format[0], &format[1]);
            if (format[0] != 'P' || (format[1] != 'F' && format[1] != 'f')){
                throw std::runtime_error("#osp:miniSG: invalid pfm texture file, header is not PF or Pf");
            }
            int numChannels = 3;
            if (format[1] == 'f') {
                numChannels = 1;
            }

            // read width and height
            int width = -1;
            int height = -1;
            rc = fscanf(file, "%i %i\n", &width, &height);
            if (rc != 2) {
                throw std::runtime_error("#osp:miniSG: could not parse width and height in PF PFM file '"+fileName.str()+"'. "
                                                                                                                         "Please report this bug at ospray.github.io, and include named file to reproduce the error.");
            }

            // read scale factor/endiannes
            float scaleEndian = 0.0;
            rc = fscanf(file, "%f\n", &scaleEndian);

            if (rc != 1) {
                throw std::runtime_error("#osp:miniSG: could not parse scale factor/endianness in PF PFM file '"+fileName.str()+"'. "
                                                                                                                                "Please report this bug at ospray.github.io, and include named file to reproduce the error.");
            }
            if (scaleEndian == 0.0) {
                throw std::runtime_error("#osp:miniSG: scale factor/endianness in PF PFM file can not be 0");
            }
            if (scaleEndian > 0.0) {
                throw std::runtime_error("#osp:miniSG: could not parse PF PFM file '"+fileName.str()+"': currently supporting only little endian formats"
                                                                                                     "Please report this bug at ospray.github.io, and include named file to reproduce the error.");
            }
            float scaleFactor = std::abs(scaleEndian);

            tex = new Texture2D;
            tex->width    = width;
            tex->height   = height;
            tex->channels = numChannels;
            tex->depth    = sizeof(float);
            tex->prefereLinear = prefereLinear;
            tex->data     = new unsigned char[width * height * numChannels * sizeof(float)];
            fread(tex->data, sizeof(float), width * height * numChannels, file);
            // flip in y, because OSPRay's textures have the origin at the lower left corner
            float *texels = (float *)tex->data;
            for (size_t y = 0; y < height / 2; ++y) {
                for (size_t x = 0; x < width * numChannels; ++x) {
                    // Scale the pixels by the scale factor
                    texels[y * width * numChannels + x] = texels[y * width * numChannels + x] * scaleFactor;
                    texels[(height - 1 - y) * width * numChannels + x] = texels[(height - 1 - y) * width * numChannels + x] * scaleFactor;
                    std::swap(texels[y * width * numChannels + x], texels[(height - 1 - y) * width * numChannels + x]);
                }
            }
        } catch(std::runtime_error e) {
            std::cerr << e.what() << std::endl;
        }
    } else {
#ifdef USE_IMAGEMAGICK
        Magick::Image image(fileName.str().c_str());
        tex = new Texture2D;
        tex->width    = image.columns();
        tex->height   = image.rows();
        tex->channels = image.matte() ? 4 : 3;
        tex->depth    = 4;
        tex->prefereLinear = prefereLinear;
        float rcpMaxRGB = 1.0f/float(MaxRGB);
        const Magick::PixelPacket* pixels = image.getConstPixels(0,0,tex->width,tex->height);
        if (!pixels) {
            std::cerr << "#osp:minisg: failed to load texture '"+fileName.str()+"'" << std::endl;
            delete tex;
            tex = NULL;
        } else {
            tex->data = new float[tex->width*tex->height*tex->channels];
            // convert pixels and flip image (because OSPRay's textures have the origin at the lower left corner)
            for (size_t y=0; y<tex->height; y++) {
                for (size_t x=0; x<tex->width; x++) {
                    const Magick::PixelPacket &pixel = pixels[y*tex->width+x];
                    float *dst = &((float*)tex->data)[(x+(tex->height-1-y)*tex->width)*tex->channels];
                    *dst++ = pixel.red * rcpMaxRGB;
                    *dst++ = pixel.green * rcpMaxRGB;
                    *dst++ = pixel.blue * rcpMaxRGB;
                    if (tex->channels == 4)
                        *dst++ = pixel.opacity * rcpMaxRGB;
                }
            }
        }
#endif
     }
    textureCache[fileName.str()] = tex;
    return tex;
}

  OSPTexture2D createTexture2D(ospray::miniSG::Texture2D *msgTex)
  {
    if(msgTex == NULL) {
        static int numWarnings = 0;
        if (++numWarnings < 10)
            cout << "WARNING: material does not have Textures (only warning for the first 10 times)!" << endl;
        return NULL;
    }
    static std::map<ospray::miniSG::Texture2D*, OSPTexture2D> alreadyCreatedTextures;
    if (alreadyCreatedTextures.find(msgTex) != alreadyCreatedTextures.end())
        return alreadyCreatedTextures[msgTex];

    //TODO: We need to come up with a better way to handle different possible pixel layouts
    OSPTextureFormat type = OSP_TEXTURE_R8;

    if (msgTex->depth == 1) {
        if( msgTex->channels == 1 ) type = OSP_TEXTURE_R8;
        if( msgTex->channels == 3 )
            type = msgTex->prefereLinear ? OSP_TEXTURE_RGB8 : OSP_TEXTURE_SRGB;
        if( msgTex->channels == 4 )
            type = msgTex->prefereLinear ? OSP_TEXTURE_RGBA8 : OSP_TEXTURE_SRGBA;
    } else if (msgTex->depth == 4) {
        if( msgTex->channels == 1 ) type = OSP_TEXTURE_R32F;
        if( msgTex->channels == 3 ) type = OSP_TEXTURE_RGB32F;
        if( msgTex->channels == 4 ) type = OSP_TEXTURE_RGBA32F;
    }
    vec2i texSize(msgTex->width, msgTex->height);
    OSPTexture2D ospTex = ospNewTexture2D( (osp::vec2i&)texSize,
                                           type,
                                           msgTex->data,
                                           0);

    alreadyCreatedTextures[msgTex] = ospTex;

    ospCommit(ospTex);
    //g_tex = ospTex; // remember last texture for debugging

    return ospTex;
  }
  } //namespace miniSG

  namespace opengl {

    //code borrowed from ospray::modules::opengl to facilitate updating
    //and linking
    inline osp::vec3f operator*(const osp::vec3f &a, const osp::vec3f &b)
    {
      return (osp::vec3f){a.x*b.x, a.y*b.y, a.z*b.z};
    }
    inline osp::vec3f operator*(const osp::vec3f &a, float b)
    {
      return (osp::vec3f){a.x*b, a.y*b, a.z*b};
    }
    inline osp::vec3f operator/(const osp::vec3f &a, float b)
    {
      return (osp::vec3f){a.x/b, a.y/b, a.z/b};
    }
    inline osp::vec3f operator*(float b, const osp::vec3f &a)
    {
      return (osp::vec3f){a.x*b, a.y*b, a.z*b};
    }
    inline osp::vec3f operator*=(osp::vec3f a, float b)
    {
      return a = (osp::vec3f){a.x*b, a.y*b, a.z*b};
    }
    inline osp::vec3f operator-(const osp::vec3f& a, const osp::vec3f& b)
    {
      return (osp::vec3f){a.x-b.x, a.y-b.y, a.z-b.z};
    }
    inline osp::vec3f operator+(const osp::vec3f& a, const osp::vec3f& b)
    {
      return (osp::vec3f){a.x+b.x, a.y+b.y, a.z+b.z};
    }
    inline osp::vec3f cross(const osp::vec3f &a, const osp::vec3f &b)
    {
      return (osp::vec3f){a.y*b.z-a.z*b.y,
          a.z*b.x-a.x*b.z,
          a.x*b.y-a.y*b.x};
    }

    inline float dot(const osp::vec3f &a, const osp::vec3f &b)
    {
      return a.x*b.x+a.y*b.y+a.z*b.z;
    }
    inline osp::vec3f normalize(const osp::vec3f &v)
    {
      return v/sqrtf(dot(v,v));
    }

    /*! \brief Compute and return OpenGL depth values from the depth component of the given
      OSPRay framebuffer, using parameters of the current OpenGL context and assuming a
      perspective projection.

      This function automatically determines the parameters of the OpenGL perspective
      projection and camera direction / up vectors. It assumes these values match those
      provided to OSPRay (fovy, aspect, camera direction / up vectors). It then maps the
      OSPRay depth buffer and transforms it to OpenGL depth values according to the OpenGL
      perspective projection.

      The OSPRay frame buffer object must have been constructed with the OSP_FB_DEPTH flag.
    */
    OSPTexture2D getOSPDepthTextureFromOpenGLPerspective(const double &fovy,
                                                         const double &aspect,
                                                         const double &zNear,
                                                         const double &zFar,
                                                         const osp::vec3f &_cameraDir,
                                                         const osp::vec3f &_cameraUp,
                                                         const float *glDepthBuffer,
                                                         float *ospDepthBuffer,
                                                         const size_t &glDepthBufferWidth,
                                                         const size_t &glDepthBufferHeight)
    {
      osp::vec3f cameraDir = (osp::vec3f&)_cameraDir;
      osp::vec3f cameraUp = (osp::vec3f&)_cameraUp;
      // this should later be done in ISPC...

      // transform OpenGL depth to linear depth
      for (size_t i=0; i<glDepthBufferWidth*glDepthBufferHeight; i++)
        {
        const double z_n = 2.0 * glDepthBuffer[i] - 1.0;
        ospDepthBuffer[i] = 2.0 * zNear * zFar / (zFar + zNear - z_n * (zFar - zNear));
        if (isnan(ospDepthBuffer[i]))
          {
          ospDepthBuffer[i] = FLT_MAX;
          }
        }

      // transform from orthogonal Z depth to ray distance t
      osp::vec3f dir_du = normalize(cross(cameraDir, cameraUp));
      osp::vec3f dir_dv = normalize(cross(dir_du, cameraDir));

      const float imagePlaneSizeY = 2.f * tanf(fovy/2.f * M_PI/180.f);
      const float imagePlaneSizeX = imagePlaneSizeY * aspect;

      dir_du *= imagePlaneSizeX;
      dir_dv *= imagePlaneSizeY;

      const osp::vec3f dir_00 = cameraDir - .5f * dir_du - .5f * dir_dv;

      for (size_t j=0; j<glDepthBufferHeight; j++)
        {
        for (size_t i=0; i<glDepthBufferWidth; i++)
          {
          const osp::vec3f dir_ij = normalize(dir_00 +
                                              float(i)/float(glDepthBufferWidth-1) * dir_du +
                                              float(j)/float(glDepthBufferHeight-1) * dir_dv);

          const float t = ospDepthBuffer[j*glDepthBufferWidth+i] / dot(cameraDir, dir_ij);
          ospDepthBuffer[j*glDepthBufferWidth+i] = t;
          }
        }

      // nearest texture filtering required for depth textures -- we don't want interpolation of depth values...
      osp::vec2i texSize = {static_cast<int>(glDepthBufferWidth),
                            static_cast<int>(glDepthBufferHeight)};
      OSPTexture2D depthTexture = ospNewTexture2D((osp::vec2i&)texSize,
                                                  OSP_TEXTURE_R32F, ospDepthBuffer,
                                                  OSP_TEXTURE_FILTER_NEAREST);

      return depthTexture;
    }
  }
}

vtkInformationKeyMacro(vtkOSPRayRendererNode, SAMPLES_PER_PIXEL, Integer);
vtkInformationKeyMacro(vtkOSPRayRendererNode, AMBIENT_SAMPLES, Integer);
vtkInformationKeyMacro(vtkOSPRayRendererNode, PATHTRACING, Integer);
vtkInformationKeyMacro(vtkOSPRayRendererNode, COMPOSITE_ON_GL, Integer);

/// LEDIAEV
// Maximum bounces for path tracing.
vtkInformationKeyMacro(vtkOSPRayRendererNode, PT_MAX_DEPTH, Integer);

vtkInformationKeyMacro(vtkOSPRayRendererNode, PT_SHOW_ENV_BACKGROUND, Integer);

// Environment light (chosen file or preset number).
//vtkInformationKeyMacro(vtkOSPRayRendererNode, ENV_IMAGE_CHOICE, Integer);
vtkInformationKeyMacro(vtkOSPRayRendererNode, ENV_IMAGE_FILE, String);

//============================================================================
vtkStandardNewMacro(vtkOSPRayRendererNode);

//----------------------------------------------------------------------------
vtkOSPRayRendererNode::vtkOSPRayRendererNode()
{
  this->Buffer = NULL;
  this->ZBuffer = NULL;
  this->OModel = NULL;
  this->ORenderer = NULL;
  this->NumActors = 0;
  this->ComputeDepth = false;
  this->RendererStr = "";
  this->OFrameBuffer = nullptr;
  this->ImageX = this->ImageY = -1;
  this->Accumulate = true;
  this->CompositeOnGL = false;
  this->SceneDirty = true;
}

//----------------------------------------------------------------------------
vtkOSPRayRendererNode::~vtkOSPRayRendererNode()
{
  if (this->Buffer)
    delete[] this->Buffer;
  if (this->ZBuffer)
    delete[] this->ZBuffer;
  if (this->OModel)
    ospRelease((OSPModel)this->OModel);
  if (this->ORenderer)
    ospRelease((OSPRenderer)this->ORenderer);
  if (this->OFrameBuffer)
    ospRelease(OFrameBuffer);
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::SetSamplesPerPixel(int value, vtkRenderer *renderer)
{
  if (!renderer)
    {
    return;
    }
  vtkInformation *info = renderer->GetInformation();
  info->Set(vtkOSPRayRendererNode::SAMPLES_PER_PIXEL(), value);
}

//----------------------------------------------------------------------------
int vtkOSPRayRendererNode::GetSamplesPerPixel(vtkRenderer *renderer)
{
  if (!renderer)
    {
    return 1;
    }
  vtkInformation *info = renderer->GetInformation();
  if (info && info->Has(vtkOSPRayRendererNode::SAMPLES_PER_PIXEL()))
    {
    return (info->Get(vtkOSPRayRendererNode::SAMPLES_PER_PIXEL()));
    }
  return 1;
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::SetAmbientSamples(int value, vtkRenderer *renderer)
{
  if (!renderer)
    {
    return;
    }
  vtkInformation *info = renderer->GetInformation();
  info->Set(vtkOSPRayRendererNode::AMBIENT_SAMPLES(), value);
}

//----------------------------------------------------------------------------
int vtkOSPRayRendererNode::GetAmbientSamples(vtkRenderer *renderer)
{
  if (!renderer)
    {
    return 0;
    }
  vtkInformation *info = renderer->GetInformation();
  if (info && info->Has(vtkOSPRayRendererNode::AMBIENT_SAMPLES()))
    {
    return (info->Get(vtkOSPRayRendererNode::AMBIENT_SAMPLES()));
    }
  return 0;
}

void vtkOSPRayRendererNode::UpdateOSPRayRenderer()
{
  vtkRenderer *ren = vtkRenderer::SafeDownCast(this->Renderable);
  if (!ren)
    return;
  vtkActorCollection *actorList = ren->GetActors();
  actorList->InitTraversal();

  int numActors = actorList->GetNumberOfItems();
  for(int i=0; i<numActors; i++) {
    vtkActor *a = actorList->GetNextActor();
    a->Modified();
  }
  ren->Modified();
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::SetCompositeOnGL(int value, vtkRenderer *renderer)
{
  if (!renderer)
    {
    return;
    }
  vtkInformation *info = renderer->GetInformation();
  info->Set(vtkOSPRayRendererNode::COMPOSITE_ON_GL(), value);
}

//----------------------------------------------------------------------------
int vtkOSPRayRendererNode::GetCompositeOnGL(vtkRenderer *renderer)
{
  if (!renderer)
    {
    return 0;
    }
  vtkInformation *info = renderer->GetInformation();
  if (info && info->Has(vtkOSPRayRendererNode::COMPOSITE_ON_GL()))
    {
    return (info->Get(vtkOSPRayRendererNode::COMPOSITE_ON_GL()));
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::SetPathTracing(bool value, vtkRenderer *renderer)
{
  if (!renderer)
    {
    return;
    }
  vtkInformation *info = renderer->GetInformation();
  bool oldVal = info->Get(vtkOSPRayRendererNode::PATHTRACING());
  info->Set(vtkOSPRayRendererNode::PATHTRACING(), value);
  if (oldVal != value)
    renderer->Modified();
  // if (value)
  // {
  //   if (RendererStr == "pathtracer")
  //     {
  //     return;
  //     }
  //   else
  //     {
  //     RendererStr = "pathtracer"
  //     UpdateOSPRayRenderer();
  //     }
  // }
  // else
  // {
  //   if (RendererStr == "scivis")
  //     {
  //     return;
  //     }
  //   else
  //     {
  //     RendererStr = "scivis";
  //     UpdateOSPRayRenderer();
  //     }
  // }
}

//----------------------------------------------------------------------------
bool vtkOSPRayRendererNode::GetPathTracing(vtkRenderer *renderer)
{
  if (!renderer)
    {
    return 0;
    }
  vtkInformation *info = renderer->GetInformation();
  if (info && info->Has(vtkOSPRayRendererNode::PATHTRACING()))
    {
    return (info->Get(vtkOSPRayRendererNode::PATHTRACING()));
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

void vtkOSPRayRendererNode::Traverse(int operation)
{
  // do not override other passes
  if (operation != render)
    {
    this->Superclass::Traverse(operation);
    return;
    }

  this->Apply(operation,true);

  OSPRenderer oRenderer = (osp::Renderer*)this->ORenderer;
  this->SceneDirty = false;

//TODO: HACK for fast forward
  float lightScale = vtkOSPRayLightNode::GetLightScale();
  if (this->GetPathTracing(static_cast<vtkRenderer*>(this->Renderable)))
    vtkOSPRayLightNode::SetLightScale(0.45);
  else if (this->GetAmbientSamples(static_cast<vtkRenderer*>(this->Renderable)))
    vtkOSPRayLightNode::SetLightScale(0.2);
  else if (static_cast<vtkRenderer*>(this->Renderable)->GetUseShadows())
    vtkOSPRayLightNode::SetLightScale(1.6);

  //camera
  //TODO: this repeated traversal to find things of particular types
  //is bad, find something smarter
  vtkViewNodeCollection *nodes = this->GetChildren();
  vtkCollectionIterator *it = nodes->NewIterator();
  it->InitTraversal();
  while (!it->IsDoneWithTraversal())
    {
    vtkOSPRayCameraNode *child =
      vtkOSPRayCameraNode::SafeDownCast(it->GetCurrentObject());
    if (child)
      {
      child->Traverse(operation);
      unsigned long int newCameraTime = child->GetRenderable()->GetMTime();
      if (this->CameraTime < newCameraTime)
      {
        this->SceneDirty = true;
      }
      this->CameraTime = newCameraTime;
      break;
      }
    it->GoToNextItem();
    }

  //lights
  this->Lights.clear();
  it->InitTraversal();
  while (!it->IsDoneWithTraversal())
    {
    vtkOSPRayLightNode *child =
      vtkOSPRayLightNode::SafeDownCast(it->GetCurrentObject());
    if (child)
      {
      child->Traverse(operation);
      }
    it->GoToNextItem();
    }

    if (static_cast<vtkRenderer*>(this->Renderable)->GetUseShadows())
      {
        std::cout << "adding ambient light\n";
      OSPLight ospAmbient = ospNewLight(oRenderer, "AmbientLight");
      ospSetString(ospAmbient, "name", "ambient_test");
      ospSet1f(ospAmbient, "intensity", 0.2f*vtkOSPRayLightNode::GetLightScale()*vtkMath::Pi());
      ospCommit(ospAmbient);
      Lights.push_back(ospAmbient);
      }
    if (this->GetPathTracing(static_cast<vtkRenderer*>(this->Renderable)))
      {
      //quad light
      OSPLight ospQuad = ospNewLight(oRenderer, "QuadLight");
      ospSetString(ospQuad, "name", "quad_test");
      ospSet3f(ospQuad, "position", 1.f, 3.5f, 0.f);
      ospSet3f(ospQuad, "edge1", 0.f, 0.f, 0.3f);
      ospSet3f(ospQuad, "edge2", 2.f, 0.f, 0.f);
      ospSet3f(ospQuad, "color", .5f, 1.f, .5f);
      ospSet1f(ospQuad, "intensity", 45.f*vtkOSPRayLightNode::GetLightScale()*vtkMath::Pi());
      ospCommit(ospQuad);
      Lights.push_back(ospQuad);
      //HDRI light
      // cout << "#ospModelViewer: Adding a hard coded hdrilight for test." << endl;
      // OSPLight ospHdri = ospNewLight(ospRenderer, "hdri");
      // ospSetString(ospHdri, "name", "hdri_test");
      // ospSet3f(ospHdri, "up", 0.f, 0.f, 1.f);
      // ospSet3f(ospHdri, "dir", 0.f, 1.f, 0.0f);
      // ospSet1f(ospHdri, "intensity", 10.f);
      // ospSetObject(ospHdri, "map", g_tex);
      // ospCommit(ospHdri);
      // lights.push_back(ospHdri);

       // /home/lmlediae/git/ospray/common/FileName.h

        /// LEDIAEV
      vtkRenderer * vtkRen = static_cast<vtkRenderer*>(this->Renderable);
            static bool once = false; //TODO: DEBUG REMOVE ME
      if (!once)
      {
        this->SetEnvImageFile(std::string("/bertha/lightProbes/rnl_probe.pfm"), vtkRen);
        this->SetShowEnvironmentBackground(false, vtkRen);
        once = true;
      }
      bool ShowEnvBackground = false;

      static OSPLight hdriLight;
      static OSPRenderer oldRenderer;

      if (!hdriLight || oldRenderer != this->ORenderer)
      {

SetPTMaxDepth(2, vtkRen);
        int max_depth = GetPTMaxDepth(vtkRen);
        ospSet1i(oRenderer, "maxDepth", max_depth);
        std::cout << "Max Depth = " << max_depth << "\n";


        const char * envFile = GetEnvImageFile(vtkRen);// Set during ParaView UI event (file dialog).

        std::string chosen_env_image_file = "";
        if (envFile) {
            chosen_env_image_file = envFile;
        }

        //std::cout << "chosen_env_image_file = " << chosen_env_image_file << "\n";

        if (chosen_env_image_file.size() > 0) {
            ospcommon::FileName imageFile(chosen_env_image_file.c_str());//("/home/lmlediae/Desktop/env_images/pfm/abstract_studio.pfm");
            ospray::miniSG::Texture2D *lightMap = ospray::miniSG::loadTexture(imageFile.path(), imageFile.base());
            if (lightMap == NULL){
                std::cout << "Failed to load hdri-light texture '" << imageFile << "'" << std::endl;
            } else {
                printf("adding hdri light\n");
                OSPLight ospHdri = ospNewLight(oRenderer, "hdri");
                ospSetString(ospHdri, "name", "hdri_test");
                ospSet3f(ospHdri, "up", 0.f, 1.f, 0.f);
                ospSet3f(ospHdri, "dir", 1.f, 0.f, 0.0f);


                //std::cout << "Light scale = " << light_scale << "\n";

                OSPTexture2D ospLightMap = createTexture2D(lightMap);
                ospSetObject(ospHdri, "map", ospLightMap);
                ospCommit(ospHdri);
                // AddLight(ospHdri);// Do we need to add the light before every render?
                hdriLight = ospHdri;
                oldRenderer = this->ORenderer;

              ShowEnvBackground = GetShowEnvironmentBackground(vtkRen);
              if (ShowEnvBackground)
                std::cout << "Displaying environment light as background.\n";
            }
        }
      }
      if (hdriLight)
      {
                        double light_scale = vtkOSPRayLightNode::GetLightScale();
                ospSet1f(hdriLight, "intensity", light_scale*2);
                ospCommit(hdriLight);
        AddLight(hdriLight);
      }
    }
    vtkOSPRayLightNode::SetLightScale(lightScale);

  OSPData lightArray = ospNewData(this->Lights.size(), OSP_OBJECT,
    (this->Lights.size()?&this->Lights[0]:NULL), 0);
  ospSetData(oRenderer, "lights", lightArray);

  //actors
  OSPModel oModel=NULL;
  it->InitTraversal();
  //since we have to spatially sort everything
  //let's see if we can avoid that in the common case when
  //the objects have not changed. Note we also cache in actornodes
  //to reuse already created ospray meshes
  unsigned int recent = 0;
  int numAct = 0; //catches removed actors
  while (!it->IsDoneWithTraversal())
    {
    vtkOSPRayActorNode *child =
      vtkOSPRayActorNode::SafeDownCast(it->GetCurrentObject());
    vtkOSPRayVolumeNode *vchild =
      vtkOSPRayVolumeNode::SafeDownCast(it->GetCurrentObject());
    if (child)
      {
      numAct++;
      recent = std::max(recent,(unsigned int)child->GetMTime());
      }
    if (vchild)
      {
      numAct++;
      recent = std::max(recent,(unsigned int)vchild->GetMTime());
      }

    it->GoToNextItem();
    }

  if (!this->OModel ||
      (recent > this->RenderTime) ||
      (numAct != this->NumActors))
    {
    this->NumActors = numAct;
    ospRelease((OSPModel)this->OModel);
    oModel = ospNewModel();
    this->OModel = oModel;
    it->InitTraversal();
    while (!it->IsDoneWithTraversal())
      {
      vtkOSPRayActorNode *child =
        vtkOSPRayActorNode::SafeDownCast(it->GetCurrentObject());
      if (child)
        {
        child->Traverse(operation);
        }
      vtkOSPRayVolumeNode *vchild =
        vtkOSPRayVolumeNode::SafeDownCast(it->GetCurrentObject());
      if (vchild)
        {
        vchild->Traverse(operation);
        }
      it->GoToNextItem();
      }
    this->RenderTime = recent;
    ospSetObject(oRenderer,"model", oModel);
    ospCommit(oModel);
    }
  else
    {
    oModel = (OSPModel)this->OModel;
    }
  it->Delete();

  this->Apply(operation,false);
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::Build(bool prepass)
{
  if (prepass)
    {
    vtkRenderer *aren = vtkRenderer::SafeDownCast(this->Renderable);
    // make sure we have a camera
    if ( !aren->IsActiveCameraCreated() )
      {
      aren->ResetCamera();
      }
    }
  this->Superclass::Build(prepass);
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::Render(bool prepass)
{
  vtkRenderer *ren = vtkRenderer::SafeDownCast(this->GetRenderable());
  if (!ren)
    {
    return;
    }

  if (prepass)
    {
    OSPRenderer oRenderer = NULL;
    bool updatedRenderer = false;
    std::string newRendererStr = this->RendererStr;
    #if  OSPRAY_VERSION_MAJOR == 0 && OSPRAY_VERSION_MINOR < 9
      newRendererStr = "obj";
    #else
      if (this->GetPathTracing(static_cast<vtkRenderer*>(this->Renderable)))
        {
        newRendererStr = "pathtracer";
        }
      else
        {
        newRendererStr = "scivis";
        }
    #endif
    if (!this->ORenderer || newRendererStr != RendererStr)
      {
      ospRelease((osp::Renderer*)this->ORenderer);
      oRenderer = (osp::Renderer*)ospNewRenderer(newRendererStr.c_str());
      this->ORenderer = oRenderer;
      UpdateOSPRayRenderer();
      updatedRenderer = true;
      }
    else
      {
      oRenderer = (osp::Renderer*)this->ORenderer;
      }
    RendererStr = newRendererStr;

    vtkRenderer *ren = vtkRenderer::SafeDownCast(this->GetRenderable());
    int *tmp = ren->GetSize();
    this->Size[0] = tmp[0];
    this->Size[1] = tmp[1];
    if (ren->GetUseShadows())
      {
      ospSet1i(oRenderer,"shadowsEnabled",1);
      }
    else
      {
      ospSet1i(oRenderer,"shadowsEnabled",0);
      }
    ospSet1i(oRenderer,"aoSamples",
      this->GetAmbientSamples(static_cast<vtkRenderer*>(this->Renderable)));
    ospSet1f(oRenderer,"aoWeight", 1.2f);
    ospSet1i(oRenderer,"spp",
    this->GetSamplesPerPixel(static_cast<vtkRenderer*>(this->Renderable)));
    this->CompositeOnGL =
    this->GetCompositeOnGL(static_cast<vtkRenderer*>(this->Renderable));


if (this->ImageX != this->Size[0] || this->ImageY != this->Size[1] || updatedRenderer)
{
        /// LEDIAEV
      vtkRenderer * vtkRen = static_cast<vtkRenderer*>(this->Renderable);
      bool ShowEnvBackground = false;
      if (vtkRen)  {
        ShowEnvBackground = this->GetShowEnvironmentBackground(vtkRen);
      }

      if (ShowEnvBackground) {
        std::cout << "using env bcg\n";
      }

        bool hasTx = ren->GetTexturedBackground();
        bool hasGradient = ren->GetGradientBackground();

        //std::cout << "hasTx = " << (hasTx ? "true" : "false") << "\n";
        //std::cout << "hasGradient = " << (hasGradient ? "true" : "false") << "\n";

      if (ShowEnvBackground) {
        ospSetObject(oRenderer, "backplate", NULL);
      } else {
        if (hasTx) {// image background.
            //std::cout << "has texture background\n";
            vtkTexture *tx = ren->GetBackgroundTexture();
            if (tx) {
                vtkImageData *d = tx->GetInput();
                if (d) {
                    vtkPointData *points = d->GetPointData();
                    int *dims = d->GetDimensions();
                    int w = dims[0];
                    int h = dims[1];
                    vtkDataArray *a = points->GetArray(0);
                    int n = a->GetNumberOfTuples();
                    // Images usually have 1, 3, or 4 components.
                    int comps = a->GetNumberOfComponents();
                    if (comps > 0 && n > 0) {
                        ospray::miniSG::Texture2D *backplate = new ospray::miniSG::Texture2D;
                        backplate->width = w;
                        backplate->height = h;
                        backplate->channels = 3;
                        backplate->depth = 1;
                        backplate->prefereLinear = true;
                        unsigned char *bp_data = new unsigned char[(backplate->width)*(backplate->height)*(backplate->channels)];
                        double t[comps];
                        for (int i=0; i<n; i++) {
                            a->GetTuple(i, t);
                            bp_data[3*i + 0] = (unsigned char)(t[0]);
                            bp_data[3*i + 1] = (unsigned char)(t[1]);
                            bp_data[3*i + 2] = (unsigned char)(t[2]);
                        }
                        backplate->data = bp_data;
                        OSPTexture2D ospBackplate = createTexture2D(backplate);
                        ospSetObject(oRenderer, "backplate", ospBackplate);
                        delete[] bp_data;
                        delete backplate;
                    }
                } else {
                    //std::cout << "no color data???\n";
                    ospSetObject(oRenderer, "backplate", NULL);
                }
            } else {
                //std::cout << "no texture pointer???\n";
                ospSetObject(oRenderer, "backplate", NULL);
            }
        } else if (hasGradient) {// Two toned background.
            double *bg = ren->GetBackground();
            double *bg2 = ren->GetBackground2();
            ospray::miniSG::Texture2D *backplate = new ospray::miniSG::Texture2D;
            backplate->width = 1;
            backplate->height = 100;// A sufficient height for a smooth gradient.
            backplate->channels = 3;
            backplate->depth = 1;
            backplate->prefereLinear = true;
            unsigned char *bp_data = new unsigned char[(backplate->width)*(backplate->height)*(backplate->channels)];
            for (int i=0; i<backplate->height; i++) {
                double f = ((double)i)/backplate->height;
                bp_data[3*i+0] = (unsigned char)((bg[0] + f*(bg2[0] - bg[0]))*255);
                bp_data[3*i+1] = (unsigned char)((bg[1] + f*(bg2[1] - bg[1]))*255);
                bp_data[3*i+2] = (unsigned char)((bg[2] + f*(bg2[2] - bg[2]))*255);
            }
            backplate->data = bp_data;
            OSPTexture2D ospBackplate = createTexture2D(backplate);
            ospSetObject(oRenderer, "backplate", ospBackplate);
            delete[] bp_data;
            delete backplate;
        } else {// solid color background.
            // std::cout << "does not have texture background\n";
            double *bg = ren->GetBackground();
            //ospSet3f(oRenderer,"bgColor", bg[0], bg[1], bg[2]);
            //ospSetObject(oRenderer, "backplate", NULL);
            ospray::miniSG::Texture2D *backplate = new ospray::miniSG::Texture2D;
            backplate->width = 1;
            backplate->height = 1;
            backplate->channels = 3;
            backplate->depth = 1;
            backplate->prefereLinear = true;
            unsigned char *bp_data = new unsigned char[(backplate->width)*(backplate->height)*(backplate->channels)];
            bp_data[0] = (unsigned char)(bg[0]*255);
            bp_data[1] = (unsigned char)(bg[1]*255);
            bp_data[2] = (unsigned char)(bg[2]*255);;
            backplate->data = bp_data;
            OSPTexture2D ospBackplate = createTexture2D(backplate);
            ospSetObject(oRenderer, "backplate", ospBackplate);
            delete[] bp_data;
            delete backplate;
            /*
            if (vtkRen->chosen_backplate_image > 0) {
                if (vtkRen->chosen_backplate_image == 1) {
                    ospcommon::FileName imageFile(vtkRen->chosen_backplate_image_file.c_str());
                    ospray::miniSG::Texture2D *backplate = ospray::miniSG::loadTexture(imageFile.path(), imageFile.base());
                    if (backplate == NULL){
                        std::cout << "Failed to load backplate file '" << imageFile << "'" << std::endl;
                        ospSetObject(oRenderer, "backplate", NULL);
                    } else {
                        OSPTexture2D ospBackplate = createTexture2D(backplate);
                        ospSetObject(oRenderer, "backplate", ospBackplate);
                    }
                } else if (vtkRen->chosen_backplate_image == 2) {


                } else {//color preset.

                }

            } else {
                ospSetObject(oRenderer, "backplate", NULL);
            }*/
        }// End background texture
      }
    }

    double *bg = ren->GetBackground();
    ospSet3f(oRenderer,"bgColor", bg[0], bg[1], bg[2]);
    }
  else
    {
    OSPRenderer oRenderer = (osp::Renderer*)this->ORenderer;
    ospCommit(oRenderer);

    osp::vec2i isize = {this->Size[0], this->Size[1]};
    if (this->ImageX != this->Size[0] || this->ImageY != this->Size[1])
    {
       this->ImageX = this->Size[0];
       this->ImageY = this->Size[1];
       osp::vec2i isize = {this->Size[0], this->Size[1]};
       OFrameBuffer = ospNewFrameBuffer(isize,
      #if OSPRAY_VERSION_MAJOR < 1 && OSPRAY_VERSION_MINOR < 10 && OSPRAY_VERSION_PATCH < 2
        OSP_RGBA_I8
      #else
        OSP_FB_RGBA8
      #endif
       , OSP_FB_COLOR | (ComputeDepth ? OSP_FB_DEPTH : 0) | (Accumulate ? OSP_FB_ACCUM : 0));
              ospSet1f(OFrameBuffer, "gamma", 1.0f);
      ospSet1f(OFrameBuffer, "gamma", 1.0f);
      ospCommit(OFrameBuffer);
      ospFrameBufferClear(OFrameBuffer, OSP_FB_COLOR|(ComputeDepth ? OSP_FB_DEPTH : 0)|(Accumulate ? OSP_FB_ACCUM : 0));
      if (this->Buffer)
        delete[] this->Buffer;
      this->Buffer = new unsigned char[this->Size[0]*this->Size[1]*4];
      if (this->ZBuffer)
        delete[] this->ZBuffer;
      this->ZBuffer = new float[this->Size[0]*this->Size[1]];
    }
    else if (Accumulate && SceneDirty)
    {
      ospFrameBufferClear(OFrameBuffer, OSP_FB_COLOR|(ComputeDepth ? OSP_FB_DEPTH : 0)|(Accumulate ? OSP_FB_ACCUM : 0));
      SceneDirty = true;
    }

    vtkCamera *cam = vtkRenderer::SafeDownCast(this->Renderable)->GetActiveCamera();

    ospSet1i(oRenderer, "backgroundEnabled",ren->GetErase());
    if (!ren->GetErase())  //Assume that if we don't want to overwrite, that we want to composite with GL
    {
      static OSPTexture2D glDepthTex=NULL;
      if (glDepthTex)
        ospRelease(glDepthTex);
      vtkRenderWindow *rwin =
      vtkRenderWindow::SafeDownCast(ren->GetVTKWindow());
      int viewportX, viewportY;
      int viewportWidth, viewportHeight;
      ren->GetTiledSizeAndOrigin(&viewportWidth,&viewportHeight,
        &viewportX,&viewportY);
      rwin->GetZbufferData(
        viewportX,  viewportY,
        viewportX+viewportWidth-1,
        viewportY+viewportHeight-1,
        this->GetZBuffer());

      double zNear, zFar;
      double fovy, aspect;
      fovy = cam->GetViewAngle();
      aspect = double(viewportWidth)/double(viewportHeight);
      cam->GetClippingRange(zNear,zFar);
      double camUp[3];
      double camDir[3];
      cam->GetViewUp(camUp);
      cam->GetFocalPoint(camDir);
      osp::vec3f  cameraUp = {camUp[0], camUp[1], camUp[2]};
      osp::vec3f  cameraDir = {camDir[0], camDir[1], camDir[2]};
      double cameraPos[3];
      cam->GetPosition(cameraPos);
      cameraDir.x -= cameraPos[0];
      cameraDir.y -= cameraPos[1];
      cameraDir.z -= cameraPos[2];
      cameraDir = ospray::opengl::normalize(cameraDir);

      static float *ospDepthBuffer=nullptr;
      static int ospDepthBufferWidth=0;
      static int ospDepthBufferHeight=0;
      float* glDepthBuffer = this->GetZBuffer();
      if (ospDepthBufferWidth != viewportWidth || ospDepthBufferHeight != viewportHeight)
       {
        if (ospDepthBuffer)
          delete[] ospDepthBuffer;
        ospDepthBuffer = new float[viewportWidth * viewportHeight];
      }
      ospDepthBufferWidth = viewportWidth;
      ospDepthBufferHeight = viewportHeight;
      glDepthTex
      = ospray::opengl::getOSPDepthTextureFromOpenGLPerspective(fovy, aspect, zNear, zFar,
        (osp::vec3f&)cameraDir, (osp::vec3f&)cameraUp,
        this->GetZBuffer(), ospDepthBuffer, viewportWidth, viewportHeight);

      ospSetObject(oRenderer, "maxDepthTexture", glDepthTex);
    }
    else
      ospSetObject(oRenderer, "maxDepthTexture", 0);
    ospCommit(oRenderer);
    ospRenderFrame(OFrameBuffer, oRenderer,
      OSP_FB_COLOR|(ComputeDepth ? OSP_FB_DEPTH : 0)|(Accumulate ? OSP_FB_ACCUM : 0));
    // for (int i =0; i< 10; i++)
    // {
    //       ospRenderFrame(OFrameBuffer, oRenderer,
    //   OSP_FB_COLOR|(ComputeDepth ? OSP_FB_DEPTH : 0)|(Accumulate ? OSP_FB_ACCUM : 0));
    //     }

    const void* rgba = ospMapFrameBuffer(OFrameBuffer, OSP_FB_COLOR);
    memcpy((void*)this->Buffer, rgba, this->Size[0]*this->Size[1]*sizeof(char)*4);
    ospUnmapFrameBuffer(rgba, OFrameBuffer);

    if (ComputeDepth)
      {
      double *clipValues = cam->GetClippingRange();
      double clipMin = clipValues[0];
      double clipMax = clipValues[1];
      double clipDiv = 1.0 / (clipMax - clipMin);

      const void *Z = ospMapFrameBuffer(OFrameBuffer, OSP_FB_DEPTH);
      float *s = (float *)Z;
      float *d = this->ZBuffer;
      for (int i = 0; i < (this->Size[0]*this->Size[1]); i++, s++, d++)
        {
        *d = (*s<clipMin? 1.0 : (*s - clipMin) * clipDiv);
        }
      ospUnmapFrameBuffer(Z, OFrameBuffer);
      }
    }
}

//----------------------------------------------------------------------------
void vtkOSPRayRendererNode::WriteLayer(unsigned char *buffer, float *Z,
                                       int buffx, int buffy, int layer)
{
  if (layer == 0)
    {
    for (int j = 0; j < buffy && j < this->Size[1]; j++)
      {
      unsigned char *iptr = this->Buffer + j*this->Size[0]*4;
      float *zptr = this->ZBuffer + j*this->Size[0];
      unsigned char *optr = buffer + j*buffx*4;
      float *ozptr = Z +  j*buffx;
      for (int i = 0; i < buffx && i < this->Size[0]; i++)
        {
        *optr++ = *iptr++;
        *optr++ = *iptr++;
        *optr++ = *iptr++;
        *optr++ = *iptr++;
        *ozptr++ = *zptr;
        zptr++;
        }
      }
    }
  else
    {
    //TODO: blending needs to be optional
    for (int j = 0; j < buffy && j < this->Size[1]; j++)
      {
      unsigned char *iptr = this->Buffer + j*this->Size[0]*4;
      float *zptr = this->ZBuffer + j*this->Size[0];
      unsigned char *optr = buffer + j*buffx*4;
      float *ozptr = Z +  j*buffx;
      for (int i = 0; i < buffx && i < this->Size[0]; i++)
        {
        if (*zptr<1.0)
          {
          if (this->CompositeOnGL)
            {
            //ospray is cooperating with GL (osprayvolumemapper)
            unsigned char a = (*(iptr+2));
            float A = (float)a/255;
            for (int h = 0; h<3; h++)
              {
              *optr = (unsigned char)(((float)*iptr)*(1-A) + ((float)*optr)*(A));
              optr++; iptr++;
              }
            optr++;
            iptr++;
            }
          else
            {
            //ospray owns all layers in window
            *optr++ = *iptr++;
            *optr++ = *iptr++;
            *optr++ = *iptr++;
            *optr++ = *iptr++;
            }
          *ozptr = *zptr;
          }
        else
          {
          optr+=4;
          iptr+=4;
          }
        ozptr++;
        zptr++;
        }
      }
    }
}

//----------------------------------------------------------------------------
// void vtkOSPRayRendererNode::Clear()
// {
//   std::cerr << __PRETTY_FUNCTION__ << std::endl;
//   if (Accumulate)
//     {
//     ospFrameBufferClear(OFrameBuffer, OSP_FB_COLOR|(ComputeDepth ? OSP_FB_DEPTH : 0)|(Accumulate ? OSP_FB_ACCUM : 0));
//     }
// }



void vtkOSPRayRendererNode::SetShowEnvironmentBackground (int val, vtkRenderer *renderer)
{
  if (!renderer)
  {
    return;
  }
  vtkInformation *info = renderer->GetInformation();
  int val2 = 0;
  if (val > 0.5) {
    val2 = 1;
  }
  info->Set(vtkOSPRayRendererNode::PT_SHOW_ENV_BACKGROUND(), val2);
}

bool vtkOSPRayRendererNode::GetShowEnvironmentBackground(vtkRenderer *renderer)
{
  if (!renderer)
  {
    return false;
  }
  vtkInformation *info = renderer->GetInformation();
  if (info && info->Has(vtkOSPRayRendererNode::PT_SHOW_ENV_BACKGROUND()))
  {
    return ((info->Get(vtkOSPRayRendererNode::PT_SHOW_ENV_BACKGROUND())) > 0.5);
  }
  return false;
}

void vtkOSPRayRendererNode::SetPTMaxDepth (int val, vtkRenderer *renderer)
{
    if (!renderer)
    {
        return;
    }
    vtkInformation *info = renderer->GetInformation();
    info->Set(vtkOSPRayRendererNode::PT_MAX_DEPTH(), val);
}

int vtkOSPRayRendererNode::GetPTMaxDepth(vtkRenderer *renderer)
{
    if (!renderer)
    {
        return 5;
    }
    vtkInformation *info = renderer->GetInformation();
    if (info && info->Has(vtkOSPRayRendererNode::PT_MAX_DEPTH()))
    {
        return (info->Get(vtkOSPRayRendererNode::PT_MAX_DEPTH()));
    }
    return 5;
}

void vtkOSPRayRendererNode::SetEnvImageFile (std::string val, vtkRenderer *renderer)
{
    if (!renderer)
    {
        return;
    }
    vtkInformation *info = renderer->GetInformation();

    // copy val?
    if (info) {
        info->Set(vtkOSPRayRendererNode::ENV_IMAGE_FILE(), val.c_str());
    }
}

const char * vtkOSPRayRendererNode::GetEnvImageFile (vtkRenderer *renderer)
{
    if (!renderer)
    {
        return NULL;
    }
    vtkInformation *info = renderer->GetInformation();
    if (info && info->Has(vtkOSPRayRendererNode::ENV_IMAGE_FILE()))
    {
        return (info->Get(vtkOSPRayRendererNode::ENV_IMAGE_FILE()));
    }
    return NULL;
}

std::string trim_string (std::string & s)
{
    std::string whitespaces(" \t\f\v\n\r");

    if (s.size() > 0) {
        int pos = s.find_first_not_of(whitespaces);
        if (pos != std::string::npos && pos > 0) {
            s = s.substr(pos, s.size() - pos);
        }
    }

    if (s.size() > 0) {
        int pos = s.find_last_not_of(whitespaces);
        if (pos != std::string::npos && pos < s.size()-1) {
            s = s.substr(0, pos + 1);
        }
    }

    return s;
}

void parse_file_type (std::string f, std::string & f_name, std::string & f_type)
{
    f_name = f;
    f_type = "";
    int pos = f.find_last_of('.');
    if (pos != std::string::npos && pos != f.size()-1) {
        int f_type_length = f.size() - pos - 1;
        int f_name_length = pos;
        if (f_type_length > 0 && f_name_length > 0) {
            f_type = f.substr(pos + 1, f_type_length);
            f_type = trim_string(f_type);
            f_name = f.substr(0, f_name_length);
        }
    }
}