/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkOSPRayAMRVolumeMapper.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkOSPRayAMRVolumeMapper.h"

#include "vtkAMRResampleFilter.h"
#include "vtkBoundingBox.h"
#include "vtkCamera.h"
#include "vtkCompositeDataPipeline.h"
#include "vtkDataSet.h"
#include "vtkExecutive.h"
#include "vtkObjectFactory.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMath.h"
#include "vtkOverlappingAMR.h"
#include "vtkMatrix4x4.h"
#include "vtkMultiBlockDataSet.h"
#include "vtkMultiThreader.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkSmartVolumeMapper.h"
#include "vtkUniformGrid.h"
#include "vtkSmartPointer.h"
#include "vtkUniformGridAMRDataIterator.h"
#include "vtkAMRBox.h"
#include "vtkAMRInformation.h"
#include "vtkBoundingBox.h"
#include "vtkColorTransferFunction.h"
#include "vtkPiecewiseFunction.h"

#include "vtkTimerLog.h"
#include "vtkNew.h"
#include "vtkPointData.h"
#include "vtkDataArray.h"
#include "vtkCellData.h"

#include "vtkOSPRay/vtkOSPRayManager.h"
#include "vtkOSPRay/vtkOSPRayRenderer.h"
#include "vtkOSPRay/vtkOSPRayVolumeRayCastMapper.h"
#include "vtkOSPRay/vtkOSPRayCamera.h"
#include "ChomboVolume.h"
// #include "../amr/Sumerian.h"
#include "vtkVolumeProperty.h"
#include "ospray/ospray.h"
#include "modules/opengl/util.h"
#ifdef VTK_OPENGL2
#include <vtk_glew.h>
#include "vtkOpenGLHelper.h"
#else
#include "vtkOpenGLExtensionManager.h"
#include "vtkgl.h"
#include "vtkOpenGL.h"
#endif
// #include "common/box.h"
// #include "common/vec.h"

     vtkStandardNewMacro( vtkOSPRayAMRVolumeMapper );

// Construct a vtkOSPRayAMRVolumeMapper
//----------------------------------------------------------------------------
     vtkOSPRayAMRVolumeMapper::vtkOSPRayAMRVolumeMapper()
     {
      std::cerr << "vtkOSPRayOSPRayAMRVolumeMapper!\n";
      this->InternalMapper = vtkSmartVolumeMapper::New();
      this->Resampler = vtkAMRResampleFilter::New();
      this->HasMetaData = false;
      this->Resampler->SetDemandDrivenMode(0);
      this->Grid = NULL;
      this->NumberOfSamples[0] = 128;
      this->NumberOfSamples[1] = 128;
      this->NumberOfSamples[2] = 128;
  this->RequestedResamplingMode = 0; // Frustrum Mode
  this->FreezeFocalPoint = false;
  this->LastFocalPointPosition[0] =
  this->LastFocalPointPosition[1] =
  this->LastFocalPointPosition[2] = 0.0;
  // Set the camera position to focal point distance to
  // something that would indicate an initial update is needed
  this->LastPostionFPDistance = -1.0;
  this->ResamplerUpdateTolerance = 10e-8;
  this->GridNeedsToBeUpdated = true;
  this->UseDefaultThreading = false;
  vtkMath::UninitializeBounds(this->Bounds);
  vtkOSPRayManager::Singleton(); // init ospray
  ospRen=vtkOSPRayRenderer::New();
  // ospMapper =vtkOSPRayVolumeRayCastMapper::New();
  transferFunction = ospNewTransferFunction("piecewise_linear");

  camera = vtkOSPRayCamera::New();
  ospModel = ospNewModel();
  ospCommit(ospModel);
  ospLoadModule("amr");

  range[0] = FLT_MAX;
  range[1] = -FLT_MAX;

  this->NumSamples = 1;
  this->SamplingRate = 0.4;

  this->CacheKey = -1.0;
}

//----------------------------------------------------------------------------
vtkOSPRayAMRVolumeMapper::~vtkOSPRayAMRVolumeMapper()
{
  this->InternalMapper->Delete();
  this->InternalMapper = NULL;
  this->Resampler->Delete();
  this->Resampler = NULL;
  if (this->Grid)
  {
    this->Grid->Delete();
    this->Grid = NULL;
  }
}

//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetInputData(vtkImageData* vtkNotUsed(genericInput))
{
  vtkErrorMacro("Mapper expects a hierarchical dataset as input" );
  this->Resampler->SetInputConnection(0, 0);
}

//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetInputData(vtkDataSet* vtkNotUsed(genericInput))
{
  vtkErrorMacro("Mapper expects a hierarchical dataset as input" );
  this->Resampler->SetInputConnection(0, 0);
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetInputData(vtkOverlappingAMR *hdata)
{
  this->SetInputDataInternal(0,hdata);
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetInputConnection (int port, vtkAlgorithmOutput *input)
{
  if ((this->Resampler->GetNumberOfInputConnections(0) > 0)
    && (this->Resampler->GetInputConnection(port,0) == input))
  {
    return;
  }
  this->Resampler->SetInputConnection(port, input);
  this->vtkVolumeMapper::SetInputConnection(port, input);
  if (this->Grid)
  {
    this->Grid->Delete();
    this->Grid = NULL;
  }
}
//----------------------------------------------------------------------------
double *vtkOSPRayAMRVolumeMapper::GetBounds()
{
  vtkOverlappingAMR*hdata;
  hdata =
  vtkOverlappingAMR::SafeDownCast
  (this->Resampler->GetInputDataObject(0,0));
  if (!hdata)
  {
    vtkMath::UninitializeBounds(this->Bounds);
  }
  else
  {
    hdata->GetBounds(this->Bounds);
  }
  return this->Bounds;
}
//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::FillInputPortInformation(
  int vtkNotUsed(port), vtkInformation* info)
{
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkOverlappingAMR");
  return 1;
}

//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SelectScalarArray(int arrayNum)
{
  this->InternalMapper->SelectScalarArray(arrayNum);
}

//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SelectScalarArray(const char *arrayName)
{
  this->InternalMapper->SelectScalarArray(arrayName);
}

//----------------------------------------------------------------------------
const char *vtkOSPRayAMRVolumeMapper::GetScalarModeAsString()
{
  return this->InternalMapper->GetScalarModeAsString();
}

//----------------------------------------------------------------------------
char *vtkOSPRayAMRVolumeMapper::GetArrayName()
{
  return this->InternalMapper->GetArrayName();
}

//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::GetArrayId()
{
  return this->InternalMapper->GetArrayId();
}

//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::GetArrayAccessMode()
{
  return this->InternalMapper->GetArrayAccessMode();
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetScalarMode(int mode)
{
  this->vtkVolumeMapper::SetScalarMode(mode);
  // for the internal mapper we need to convert all cell based
  // modes to point based since this is what the resample filter is doing
  int newMode = mode;
  if (mode == VTK_SCALAR_MODE_USE_CELL_DATA)
  {
    newMode = VTK_SCALAR_MODE_USE_POINT_DATA;
  }
  else if (mode == VTK_SCALAR_MODE_USE_CELL_FIELD_DATA)
  {
    newMode = VTK_SCALAR_MODE_USE_POINT_FIELD_DATA;
  }

  this->InternalMapper->SetScalarMode(newMode);
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetBlendMode(int mode)
{
  this->InternalMapper->SetBlendMode(mode);
}
//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::GetBlendMode()
{
  return this->InternalMapper->GetBlendMode();
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetCropping(int mode)
{
  this->InternalMapper->SetCropping(mode);
}
//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::GetCropping()
{
  return this->InternalMapper->GetCropping();
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetCroppingRegionFlags(int mode)
{
  this->InternalMapper->SetCroppingRegionFlags(mode);
}
//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::GetCroppingRegionFlags()
{
  return this->InternalMapper->GetCroppingRegionFlags();
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetCroppingRegionPlanes(double arg1, double arg2,
 double arg3, double arg4,
 double arg5, double arg6)
{
  this->InternalMapper->SetCroppingRegionPlanes(arg1, arg2, arg3,
    arg4, arg5, arg6);
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::GetCroppingRegionPlanes(double *planes)
{
  this->InternalMapper->GetCroppingRegionPlanes(planes);
}
//----------------------------------------------------------------------------
double *vtkOSPRayAMRVolumeMapper::GetCroppingRegionPlanes()
{
  return this->InternalMapper->GetCroppingRegionPlanes();
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetRequestedRenderMode(int mode)
{
  this->InternalMapper->SetRequestedRenderMode(mode);
}
//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::GetRequestedRenderMode()
{
  return this->InternalMapper->GetRequestedRenderMode();
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::SetInterpolationMode(int mode)
{
  this->InternalMapper->SetInterpolationMode(mode);
}
//----------------------------------------------------------------------------
int vtkOSPRayAMRVolumeMapper::GetInterpolationMode()
{
  return this->InternalMapper->GetInterpolationMode();
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::ReleaseGraphicsResources(vtkWindow *window)
{
  this->InternalMapper->ReleaseGraphicsResources(window);
}

struct BrickInfo {
  ospcommon::box3i box;
  int   level;
  float dt;

  ospcommon::vec3i size() const { return box.size() + ospcommon::vec3i(1); }
};

//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::Render(vtkRenderer *ren, vtkVolume *vol)
{
  // std::cerr << __PRETTY_FUNCTION__ << std::endl;
  vtkOverlappingAMR*amr=
  vtkOverlappingAMR::SafeDownCast
  (this->Resampler->GetInputDataObject(0,0));
  if (!amr)
  {
    std::cerr << "couldn't get amr data";
    return;
  }
  OSPRenderer orenderer = vtkOSPRayManager::Singleton()->OSPRayVolumeRenderer;

  vtkVolumeProperty *volProperty = vol->GetProperty();
  ospRen->SetSamples(this->NumSamples);
  camera->DeepCopy(ren->GetActiveCamera());
  ospRen->SetActiveCamera(camera);
  camera->Modified();
  ospRen->SetRenderWindow(ren->GetRenderWindow());
  ospRen->SetBackground(ren->GetBackground()[0],ren->GetBackground()[1],ren->GetBackground()[2]);
  ospRen->SetBackgroundEnabled(1);

  vtkOSPRayAMRVolumeCacheEntry* cacheEntry = _cache[CacheKey];
  ospVolume = 0;

  bool volDirty = false;
  if (cacheEntry)
  {
    ospVolume = cacheEntry->Volume;
  }
  if (!ospVolume)
  {
    if (amr->GetMTime() > this->BuildTime)
    {
      volDirty=true;
      std::cerr << "Building AMR\n";
      this->BuildTime.Modified();

      int lastLevel=0;
      std::vector<OSPData>   brickDataArray;
      std::vector<BrickInfo> brickInfoArray;
      size_t totalDataSize = 0;

      const vtkAMRInformation* amrInfo = amr->GetAMRInfo();
      vtkSmartPointer<vtkUniformGridAMRDataIterator> iter;
      iter.TakeReference(vtkUniformGridAMRDataIterator::SafeDownCast(
        amr->NewIterator()));
      for (iter->InitTraversal(); !iter->IsDoneWithTraversal();
        iter->GoToNextItem())
      {

        unsigned int level = iter->GetCurrentLevel();
          assert(level >= lastLevel);  //ospray requires level info be ordered lowest to highest
          lastLevel = level;
          unsigned int index = iter->GetCurrentIndex();

        // note: this iteration "naturally" goes from datasets at lower levels to
        // those at higher levels.
          vtkImageData* data = vtkImageData::SafeDownCast(iter->GetCurrentDataObject());
          assert(data != NULL);
          float* dataPtr;
          int dim[3];

          const vtkAMRBox& box = amrInfo->GetAMRBox(level, index);
          const int* lo=box.GetLoCorner();
          const int* hi=box.GetHiCorner();
          ospcommon::vec3i lo_v= {lo[0],lo[1],lo[2]};
          ospcommon::vec3i hi_v={hi[0],hi[1],hi[2]};
          dim[0] = hi[0]-lo[0]+1;
          dim[1] = hi[1]-lo[1]+1;
          dim[2] = hi[2]-lo[2]+1;

          vtkDataArray* cellArray = data->GetCellData()->GetArray(this->GetArrayName());  //TODO: unhardcode array name

          if (!cellArray)
            std::cerr << "could not get data!\n";

          if (cellArray->GetDataType() != VTK_FLOAT)
          {
                if (cellArray->GetDataType() == VTK_DOUBLE)
                {
                  float* fdata=new float[dim[0]*dim[1]*size_t(dim[2])];
                  double* dptr;
                  dptr = (double*)cellArray->WriteVoidPointer(0, cellArray->GetSize());
                  for(size_t i=0;i<dim[0]*dim[1]*size_t(dim[2]);i++)
                  {
                    fdata[i]=dptr[i];
                  }
                  dataPtr=fdata;

                  int numTuples = cellArray->GetNumberOfTuples();
                } 
                else
                {
                 std::cerr << "Data was not fload and not double!\n";
                 assert(0);
               }
             }
             else
             {
              dataPtr=(float*)cellArray->WriteVoidPointer(0, cellArray->GetSize());
            }

            double* crange = cellArray->GetRange();
            if (range[0] > crange[0])
              range[0] = crange[0];
            if (range[1] < crange[1])
              range[1] = crange[1];

            totalDataSize += sizeof(float)*size_t(dim[0]*dim[1])*dim[2];
            OSPData odata = ospNewData(dim[0]*dim[1]*dim[2],OSP_FLOAT,dataPtr, OSP_DATA_SHARED_BUFFER);
            brickDataArray.push_back(odata);

            BrickInfo bi;
            ospcommon::box3i obox = {lo_v,hi_v};
            bi.box = obox;
            bi.dt = pow(2.0,-double(level))*1.0f;
            bi.level = level;
            totalDataSize += sizeof(BrickInfo);

            brickInfoArray.push_back(bi);
        }


        ospVolume = ospNewVolume("chombo_volume");
        assert(ospVolume);
        assert(brickDataArray.size() == brickInfoArray.size());
    ospSet1f(ospVolume, "samplingRate", this->SamplingRate);  //TODO: gui option

    double origin[3];
    vol->GetOrigin(origin);
    double *bds = this->GetBounds();
    origin[0] = bds[0];
    origin[1] = bds[2];
    origin[2] = bds[4];
    std::cout << "gridOrigin: " << origin[0] << " " << origin[1] << " " << origin[2];
    std::cout << "gridBounds: " << bds[1] << " " << bds[3] << " " << bds[5];

    double spacing[3];
    amr->GetAMRInfo()->GetSpacing(0,spacing);
    std::cout << "gridSpacing: " << spacing[0] << " " << spacing[1] << " " << spacing[2];
    ospSet3f(ospVolume, "gridOrigin", origin[0], origin[1], origin[2]);
    ospSet3f(ospVolume, "gridSpacing", spacing[0],spacing[1],spacing[2]);

    OSPData brickDataData = ospNewData(brickDataArray.size(),OSP_OBJECT,&brickDataArray[0],0);
    ospSetData(ospVolume,"brickData",brickDataData);
    OSPData brickInfoData = ospNewData(brickInfoArray.size()*sizeof(brickInfoArray[0]),OSP_RAW,&brickInfoArray[0],0);
    ospSetData(ospVolume,"brickInfo",brickInfoData);

  //TODO: multiplying hardcoded slicing plane by spacing as amr renderer does not respect spacing
    ospSet3f(ospVolume, "volumeClippingBoxLower", -10000,-10000,(bds[5]-bds[4])/2.0f/float(spacing[0]));
    ospSet3f(ospVolume, "volumeClippingBoxUpper", 10000,10000,10000);

    std::cout << "totalDataSize: " << totalDataSize << std::endl;


  }


  if (CacheKey >= 0.0)
  {
    _cache[CacheKey] = new vtkOSPRayAMRVolumeCacheEntry;
    _cache[CacheKey]->Volume = ospVolume;
    _cache[CacheKey]->SamplingRate = this->SamplingRate;
    cacheEntry = _cache[CacheKey];
  }
  }
  else
  {
    if (cacheEntry->SamplingRate != this->SamplingRate)
    {
      volDirty=true;
      cacheEntry->SamplingRate = this->SamplingRate;
    }
    else
    {
      volDirty = false;
    }
  }
  if ((vol->GetProperty()->GetMTime() > PropertyTime) || volDirty || 
    (cacheEntry && (vol->GetProperty()->GetMTime() > cacheEntry->PropertyTime)))
  {
    std::cerr << "Building TF\n";
    PropertyTime.Modified();
    // 
    // transfer function
    //
    std::cerr << "scalar range " << range[0] << " " << range[1] << std::endl;
    std::cerr << "creating tf with range " << range[0] << " " << range[1] << std::endl;

    ospCommit(transferFunction);
    int NumColors = 128;
    std::vector<float> TFVals, TFOVals;
    vtkColorTransferFunction* colorTF = volProperty->GetRGBTransferFunction(0);
    vtkPiecewiseFunction *scalarTF = volProperty->GetScalarOpacity(0);
    int numNodes = colorTF->GetSize();
    double* tfData = colorTF->GetDataPointer();
    double* tfRangeD = colorTF->GetRange();
    osp::vec2f tfRange = {tfRangeD[0], tfRangeD[1]};
    std::cerr << "vtkTF range " << tfRange.x << " " << tfRange.y << std::endl;
   // delete[] tfRangeD;

    TFVals.resize(NumColors*3);
    TFOVals.resize(NumColors);
    scalarTF->GetTable(tfRangeD[0],tfRangeD[1], NumColors, &TFOVals[0]);
    colorTF->GetTable(tfRangeD[0],tfRangeD[1], NumColors, &TFVals[0]);

    OSPData colorData = ospNewData(NumColors, OSP_FLOAT3, &TFVals[0]);// TODO: memory leak?  does ospray manage this>
    ospSetData(transferFunction, "colors", colorData);
    OSPData tfAlphaData = ospNewData(NumColors, OSP_FLOAT, &TFOVals[0]);
    ospSetData(transferFunction, "opacities", tfAlphaData);
    ospSetVec2f(transferFunction, "valueRange", tfRange);
    ospCommit(transferFunction);
    ospSetObject(ospVolume,"transferFunction",transferFunction);

    ospCommit(ospVolume);

    if (cacheEntry)
    {
      cacheEntry->PropertyTime = PropertyTime;
    }


  }
  ospRen->PreRender();

  static int oldSamplingRate = -1;
  if (oldSamplingRate != this->SamplingRate)
  {
    oldSamplingRate = this->SamplingRate;
    volDirty=true;
  }

  if (volDirty)
  {
    ospSet1f(ospVolume, "samplingRate", this->SamplingRate);
    ospCommit(ospVolume);
  }
  ospModel = ospNewModel();
  ospAddVolume(ospModel,ospVolume);
  ospCommit(ospModel);


  ospSetObject(orenderer, "model", ospModel);
  ospCommit(orenderer);

  vtkOSPRayManager::Singleton()->OSPRayVolumeModel = ospModel;
  ospRen->SetHasVolume(true);
  ospRen->SetClearAccumFlag();
  ospRen->SetComputeDepth(false);

  {
    static OSPTexture2D glDepthTex=NULL;
    OSPRenderer orenderer = ospRen->GetOSPRayManager()->OSPRayVolumeRenderer;
    if (glDepthTex)
      ospRelease(glDepthTex);
    glDepthTex = ospray::opengl::getOSPDepthTextureFromOpenGLPerspective();
    ospSetObject(orenderer, "maxDepthTexture", glDepthTex);
    ospCommit(orenderer);
  }
  glDisable(GL_DEPTH_TEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
  glDisable(GL_TEXTURE_2D);
  glColor3f(1,1,1);
  ospRen->LayerRender();
  glEnable(GL_DEPTH_TEST);

  ospRen->RemoveVolume(vol);

}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::UpdateResampler(vtkRenderer *ren, vtkOverlappingAMR *amr)
{
  // Set the bias of the resample filter to be the projection direction
  double bvec[3];
  vtkCamera *cam = ren->GetActiveCamera();
  double d, d2, fp[3], pd, gb[6];
  d = cam->GetDistance();
  cam->GetFocalPoint(fp);

  if (this->Grid)
  {
    // Compare distances with the grid's bounds
    this->Grid->GetBounds(gb);
    vtkBoundingBox bbox(gb);
    double maxL = bbox.GetMaxLength();
    // If the grid's max length is 0 then we need to update
    if (maxL > 0.0)
    {
      pd = fabs(d - this->LastPostionFPDistance) / this->LastPostionFPDistance;
      if ((this->LastPostionFPDistance > 0.0) && (pd <= this->ResamplerUpdateTolerance))
      {
        // Lets see if the focal point has not moved enough to cause an update
        d2 = vtkMath::Distance2BetweenPoints(fp, this->LastFocalPointPosition)/(maxL*maxL);
        if (d2 <= (this->ResamplerUpdateTolerance * this->ResamplerUpdateTolerance))
        {
          // nothing needs to be updated
          return;
        }
        else
        {
//          int oops = 1;
        }
      }
    }
  }
  cam->GetDirectionOfProjection(bvec);
  this->Resampler->SetBiasVector(bvec);
  this->Resampler->SetUseBiasVector(true);
  this->LastPostionFPDistance = d;
  this->LastFocalPointPosition[0] = fp[0];
  this->LastFocalPointPosition[1] = fp[1];
  this->LastFocalPointPosition[2] = fp[2];

  if (this->RequestedResamplingMode == 0)
  {
    this->UpdateResamplerFrustrumMethod(ren, amr);
  }
  else
  {
    // This is the focal point approach where we
    // center the grid on the focal point and set its length
    // to be the distance between the camera and its focal point
    double p[3];
    p[0] = fp[0] - d;
    p[1] = fp[1] - d;
    p[2] = fp[2] - d;
    // Now set the min/max of the resample filter
    this->Resampler->SetMin(p);
    p[0] = fp[0] + d;
    p[1] = fp[1] + d;
    p[2] = fp[2] + d;
    this->Resampler->SetMax(p);
    this->Resampler->SetNumberOfSamples(this->NumberOfSamples);
  }
  // The grid may have changed
  this->GridNeedsToBeUpdated = true;
}

//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::UpdateResamplerFrustrumMethod(vtkRenderer *ren,
 vtkOverlappingAMR *amr)
{
  double bounds[6];
  // If we have been passed a valid amr then assume this is the proper
  // meta data to use
  if (amr)
  {
    amr->GetBounds(bounds);
  }
  else
  {
    // Make sure the bounds are up to date
    this->GetBounds(bounds);
  }

  double computed_bounds[6];
  if (vtkOSPRayAMRVolumeMapper::ComputeResamplerBoundsFrustumMethod(
    ren->GetActiveCamera(), ren, bounds, computed_bounds))
  {
    vtkBoundingBox bbox(computed_bounds);
    // Now set the min/max of the resample filter
    this->Resampler->SetMin( const_cast< double* >(bbox.GetMinPoint()) );
    this->Resampler->SetMax( const_cast< double* >(bbox.GetMaxPoint()) );
    this->Resampler->SetNumberOfSamples(this->NumberOfSamples);
  }
}

//----------------------------------------------------------------------------
bool vtkOSPRayAMRVolumeMapper::ComputeResamplerBoundsFrustumMethod(
  vtkCamera* camera, vtkRenderer* renderer,
  const double bounds[6], double out_bounds[6])
{
  vtkMath::UninitializeBounds(out_bounds);

  // First we need to create a bouding box that represents the visible region
  // of the camera in World Coordinates

  // In order to produce as tight of bounding box as possible we need to determine
  // the z range in view coordinates of the data and then project that part
  // of the view volume back into world coordinates

  // We would just use the renderer's WorldToView and ViewToWorld methods but those
  // implementations are not efficient for example ViewToWorld would do 8
  // matrix inverse ops when all we really need to do is one


  // Get the camera transformation
  vtkMatrix4x4 *matrix = camera->GetCompositeProjectionTransformMatrix(
    renderer->GetTiledAspectRatio(), 0, 1);

  int i, j, k;
  double pnt[4], tpnt[4];
  vtkBoundingBox bbox;
  pnt[3] = 1.0;
  for (i = 0; i < 2; i++)
  {
    pnt[0] = bounds[i];
    for (j = 2; j < 4; j++)
    {
      pnt[1] = bounds[j];
      for (k = 4; k < 6; k++)
      {
        pnt[2] = bounds[k];
        matrix->MultiplyPoint(pnt, tpnt);
        if (tpnt[3])
        {
          bbox.AddPoint(tpnt[0]/tpnt[3],
            tpnt[1]/tpnt[3], tpnt[2]/tpnt[3]);
        }
        else
        {
          vtkGenericWarningMacro("UpdateResampler: Found an Ideal Point going to VC!");
        }
      }
    }
  }

  double zRange[2];
  if (bbox.IsValid())
  {
    zRange[0] = bbox.GetMinPoint()[2];
    zRange[1] = bbox.GetMaxPoint()[2];
    // Normalize the z values to make sure they are between -1 and 1
    for (i = 0; i < 2; i++)
    {
      if (zRange[i] < -1.0)
      {
        zRange[i] = -1.0;
      }
      else if (zRange[i] > 1.0)
      {
        zRange[i] = 1.0;
      }
    }
  }
  else
  {
    // Since we could not find a valid bounding box assume that the
    // zrange is -1 to 1
    zRange[0] = -1.0;
    zRange[1] = 1.0;
  }

  // Now that we have the z range of the data in View Coordinates lets
  // convert that part of the View Volume back into World Coordinates
  double mat[16];
  //Need the inverse
  vtkMatrix4x4::Invert(*matrix->Element, mat);

  bbox.Reset();
  // Compute the bounding box
  for (i = -1; i < 2; i+=2)
  {
    pnt[0] = i;
    for (j = -1; j < 2; j+=2)
    {
      pnt[1] = j;
      for (k = 0; k < 2; k++)
      {
        pnt[2] = zRange[k];
        vtkMatrix4x4::MultiplyPoint(mat,pnt,tpnt);
        if (tpnt[3])
        {
          bbox.AddPoint(tpnt[0]/tpnt[3],
            tpnt[1]/tpnt[3], tpnt[2]/tpnt[3]);
        }
        else
        {
          vtkGenericWarningMacro("UpdateResampler: Found an Ideal Point going to WC!");
        }
      }
    }
  }

  // Check to see if the box is valid
  if (!bbox.IsValid())
  {
    return false; // There is nothing we can do
  }
  bbox.GetBounds(out_bounds);
  return true;
}

//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::UpdateGrid()
{
  // This is for debugging
#define PRINTSTATS 0
#if PRINTSTATS
  vtkNew<vtkTimerLog> timer;
  int gridDim[3];
  double gridOrigin[3];
  timer->StartTimer();
#endif
  this->Resampler->Update();
#if PRINTSTATS
  timer->StopTimer();
  std::cerr << "Resample Time:" << timer->GetElapsedTime() << " ";
  std::cerr << "New Bounds: [" << bbox.GetMinPoint()[0]
  << ", " << bbox.GetMaxPoint()[0] << "], ["
  << bbox.GetMinPoint()[1]
  << ", " << bbox.GetMaxPoint()[1] << "], ["
  << bbox.GetMinPoint()[2]
  << ", " << bbox.GetMaxPoint()[2] << "\n";
#endif
  vtkMultiBlockDataSet *mb = this->Resampler->GetOutput();
  if (!mb)
  {
    return;
  }
  unsigned int nb = mb->GetNumberOfBlocks();
  if (!nb)
  {
    // No new grid was created
    return;
  }
  if (nb != 1)
  {
    vtkErrorMacro("UpdateGrid: Resampler created more than 1 Grid!");
  }
  if (this->Grid)
  {
    this->Grid->Delete();
  }
  this->Grid = vtkUniformGrid::SafeDownCast(mb->GetBlock(0));
  this->Grid->Register(0);
  this->GridNeedsToBeUpdated = false;
#if PRINTSTATS
  this->Grid->GetDimensions(gridDim);
  this->Grid->GetOrigin(gridOrigin);
  std::cerr << "Grid Dimenions: (" << gridDim[0] << ", " << gridDim[1] << ", "
  << gridDim[2]
  << ") Origin:(" << gridOrigin[0] << ", "<< gridOrigin[1] << ", "
  << gridOrigin[2] << ")\n";
#endif
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::ProcessUpdateExtentRequest(vtkRenderer *vtkNotUsed(ren),
  vtkInformation*info,
  vtkInformationVector **inputVector,
  vtkInformationVector *outputVector)
{
  this->Resampler->RequestUpdateExtent(info, inputVector, outputVector);
}
//----------------------------------------------------------------------------
void vtkOSPRayAMRVolumeMapper::ProcessInformationRequest(vtkRenderer *ren,
 vtkInformation*info,
 vtkInformationVector **inputVector,
 vtkInformationVector *outputVector)
{
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  vtkInformation *input = inputVector[0]->GetInformationObject( 0 );
  if (!(input &&  input->Has(vtkCompositeDataPipeline::COMPOSITE_DATA_META_DATA())))
  {
    this->HasMetaData = false;
    this->Resampler->SetDemandDrivenMode(0);
    return;
  }

  if (!this->HasMetaData)
  {
    this->HasMetaData = true;
    this->Resampler->SetDemandDrivenMode(1);
  }
  vtkOverlappingAMR *amrMetaData =
  vtkOverlappingAMR::SafeDownCast(
    input->Get(vtkCompositeDataPipeline::COMPOSITE_DATA_META_DATA()) );

  this->UpdateResampler(ren, amrMetaData);
  this->Resampler->RequestInformation(info, inputVector, outputVector);
}
//----------------------------------------------------------------------------gg

// Print the vtkOSPRayAMRVolumeMapper
void vtkOSPRayAMRVolumeMapper::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "ScalarMode: " << this->GetScalarModeAsString() << endl;

  if ( this->ScalarMode == VTK_SCALAR_MODE_USE_POINT_FIELD_DATA ||
   this->ScalarMode == VTK_SCALAR_MODE_USE_CELL_FIELD_DATA )
  {
    if (this->ArrayAccessMode == VTK_GET_ARRAY_BY_ID)
    {
      os << indent << "ArrayId: " << this->ArrayId << endl;
    }
    else
    {
      os << indent << "ArrayName: " << this->ArrayName << endl;
    }
  }
  os << indent << "UseDefaultThreading:" << this->UseDefaultThreading << "\n";
  os << indent << "ResampledUpdateTolerance: " <<
  this->ResamplerUpdateTolerance << "\n";
  os << indent << "NumberOfSamples: ";
  for( int i=0; i < 3; ++i )
  {
    os << this->NumberOfSamples[ i ] << " ";
  }
  os << std::endl;
  os << indent << "RequestedResamplingMode: " <<
  this->RequestedResamplingMode << "\n";
  os << indent << "FreezeFocalPoint: " << this->FreezeFocalPoint << "\n";
}
//----------------------------------------------------------------------------
