//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataSetReader.h>
#include <adis/xgc/StorageXGC.h>

#include <vtkm/cont/CellSetExtrude.h>

#ifdef ADIS_USE_MPI
#include <mpi.h>
#endif

#include <string>
#include <unordered_map>

int main(int argc, char **argv)
{
  if (argc != 3)
  {
    std::cout << "./xgc <name of the json file> <path of data source folder>\n";
    return 0;
  }

  int rank = 0;
#ifdef ADIS_USE_MPI
  MPI_Init(&argc, &argv);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
#endif

  std::string dataModelFile = std::string(argv[1]);
  adis::io::DataSetReader reader(dataModelFile);

  std::unordered_map<std::string, std::string> paths;
  paths["mesh"] = std::string(argv[2]) + "/";
  paths["3d"] = std::string(argv[2]) + "/";
  paths["bfield"] = std::string(argv[2]) + "/";

  auto metaData = reader.ReadMetaData(paths);

  adis::metadata::MetaData selections;
  adis::metadata::Index idx(2);
  selections.Set(adis::keys::STEP_SELECTION(), idx);

  using FieldInfoType =
    adis::metadata::Vector<adis::metadata::FieldInformation>;
  FieldInfoType fieldSelection;
  fieldSelection.Data.push_back(
    adis::metadata::FieldInformation(
      "pot0", vtkm::cont::Field::Association::POINTS));
  fieldSelection.Data.push_back(
    adis::metadata::FieldInformation(
      "dpot", vtkm::cont::Field::Association::POINTS));
  fieldSelection.Data.push_back(
    adis::metadata::FieldInformation(
      "node_data1", vtkm::cont::Field::Association::POINTS));
  selections.Set(adis::keys::FIELDS(), fieldSelection);

  vtkm::cont::PartitionedDataSet output =
    reader.ReadDataSet(paths, selections);

  if (output.GetNumberOfPartitions() == 0)
  {
#ifdef ADIS_USE_MPI
    MPI_Finalize();
#endif
    return 0;
  }
  vtkm::cont::DataSet ds = output.GetPartition(0);

  // CellSet checks
  auto& cellSet = ds.GetCellSet();
  if (!cellSet.IsType<vtkm::cont::CellSetExtrude>())
  {
    throw std::runtime_error("CellSet for XGC should be type CellSetExtrude");
  }
  auto& csXGC = cellSet.Cast<vtkm::cont::CellSetExtrude>();
  auto numPlanes = csXGC.GetNumberOfPlanes();
  if (numPlanes != 4)
  {
    throw std::runtime_error("There should be 4 planes");
  }
  auto numPointsPerPlane = csXGC.GetNumberOfPointsPerPlane();
  if (numPointsPerPlane != 20694)
  {
    throw std::runtime_error("There should be 20,694 points per plane in this dataset.");
  }
  if (numPointsPerPlane * numPlanes != cellSet.GetNumberOfPoints())
  {
    throw std::runtime_error("The total number of points in the cell set is incorrect");
  }

  // CoordinateSystem checks
  vtkm::cont::CoordinateSystem coordSys = ds.GetCoordinateSystem(0);
  if (ds.GetNumberOfCoordinateSystems() < 1)
  {
    throw std::runtime_error("XGC should have a coordinate system");
  }
  if (numPointsPerPlane * numPlanes != coordSys.GetNumberOfPoints())
  {
    throw std::runtime_error("The total number of points in the coordinate system is incorrect");
  }

  // Fields checks
  const auto& arrHandle
    = ds.GetField("pot0").GetData().Cast<vtkm::cont::ArrayHandle<
      double, vtkm::cont::internal::StorageTagXGCPlane> >();
  auto rangeArray = vtkm::cont::ArrayRangeCompute(arrHandle);
  auto rangePortal = rangeArray.ReadPortal();
  if (rangePortal.Get(0).Min < 1.67863 || rangePortal.Get(0).Min > 1.67865)
  {
    throw std::runtime_error("pot0 min is incorrect");
  }
  if (rangePortal.Get(0).Max < 90.7623 || rangePortal.Get(0).Max > 90.7625)
  {
    throw std::runtime_error("pot0 max is incorrect");
  }

  const auto& dpotAH
    = ds.GetField("dpot").GetData().Cast<vtkm::cont::ArrayHandle<
    double, vtkm::cont::internal::StorageTagXGCPlane> >();
  rangeArray = vtkm::cont::ArrayRangeCompute(dpotAH);
  rangePortal = rangeArray.ReadPortal();
  if (rangePortal.Get(0).Min < -15.1787 || rangePortal.Get(0).Min > -15.1785)
  {
    throw std::runtime_error("dpot min is incorrect");
  }
  if (rangePortal.Get(0).Max < 15.5977 || rangePortal.Get(0).Max > 15.5979)
  {
    throw std::runtime_error("dpot max is incorrect");
  }

  const auto& nd1
    = ds.GetField("node_data1").GetData().Cast<vtkm::cont::ArrayHandle<
      double, vtkm::cont::internal::StorageTagXGCPlane> >();
  rangeArray = vtkm::cont::ArrayRangeCompute(nd1);
  rangePortal = rangeArray.ReadPortal();
  if (rangePortal.Get(0).Min != 0)
  {
    throw std::runtime_error("node_data min is incorrect");
  }
  if (rangePortal.Get(0).Max < 0.22297 || rangePortal.Get(0).Max > 0.22299)
  {
    throw std::runtime_error("node_data max is incorrect");
  }

#ifdef ADIS_USE_MPI
  MPI_Finalize();
#endif
  return 0;
}
