//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataSetReader.h>
#include <adis/predefined/SupportedDataModels.h>
#include <adis/predefined/InternalMetadataSource.h>
#include <adis/predefined/DataModelFactory.h>

#include <adis_rapidjson.h>
#include ADIS_RAPIDJSON(rapidjson/error/en.h)
#include ADIS_RAPIDJSON(rapidjson/filereadstream.h)
#include ADIS_RAPIDJSON(rapidjson/schema.h)
#include ADIS_RAPIDJSON(rapidjson/stringbuffer.h)
#include ADIS_RAPIDJSON(rapidjson/prettywriter.h)

#include "SchemaValidator.h"

#ifdef ADIS_USE_MPI
#include <mpi.h>
#endif

// matches order of DataModelTypes
std::vector<std::string> bpFiles =
{
  "cartesian-attr.bp",
  "rectilinear-attr.bp",
  "tris-explicit-attr.bp",
  "tris-blocks-attr.bp",
  "xgc-attr.bp"
};

void testReader(const std::string& dir, const std::string& dataModelStr)
{
  auto dataModel = adis::predefined::ConvertDataModelToEnum(dataModelStr);
  std::string filename = dir + "/" + bpFiles[static_cast<size_t>(dataModel)];
  auto rc = adis::io::DataSetReader::CheckForDataModelAttribute(filename);
  if (dataModel != adis::predefined::DataModelTypes::UNSUPPORTED && !rc)
  {
    throw std::runtime_error("Data model " + dataModelStr + " does not appear to be a supported "
        "predefined data model");
  }

  std::unordered_map<std::string, std::string> paths;
  if (dataModel == adis::predefined::DataModelTypes::XGC)
  {
    paths["3d"] = dir + "/";
    paths["mesh"] = dir + "/";
  }
  else
  {
    paths["source"] = filename;
  }
  adis::io::DataSetReader reader(filename,
    adis::io::DataSetReader::DataModelInput::BPFile);
  auto metadata = reader.ReadMetaData(paths);

  reader.PrepareNextStep(paths);
  adis::metadata::MetaData selections;
  auto output = reader.ReadStep(paths, selections);
}

void validateSchema(const std::string& dir, const std::string& dataModelStr, const std::string& schemaFile)
{
  auto dataModel = adis::predefined::ConvertDataModelToEnum(dataModelStr);
  std::string filename = dir + "/" + bpFiles[static_cast<size_t>(dataModel)];
  std::shared_ptr<adis::predefined::InternalMetadataSource> metadataSource(
      new adis::predefined::InternalMetadataSource(filename));
  auto dm =
    adis::predefined::DataModelFactory::GetInstance().CreateDataModel(metadataSource);
  auto& doc = dm->GetDOM(true);
  rapidjson::StringBuffer buffer;
  rapidjson::Writer<rapidjson::StringBuffer> writer(buffer);
  doc.Accept(writer);

  // run through validator
  ADISSchemaValidator validator(schemaFile);
  if (!validator.SetDocument(buffer.GetString()))
  {
    throw std::runtime_error("Converting JSON string to document failed");
  }

  if (!validator.Validate())
  {
    throw std::runtime_error("Validation failed");
  }
}

int main(int argc, char** argv)
{
#ifdef ADIS_USE_MPI
  MPI_Init(&argc, &argv);
#endif

  if(argc != 4)
  {
    std::cerr << "Usage: " << argv[0] << "predefined-model-name /path/to/dataroot adis-schema.json\n";
    return 1;
  }

  std::string dataModelStr = argv[1];
  auto dataModel = adis::predefined::ConvertDataModelToEnum(dataModelStr);
  switch (dataModel)
  {
    case adis::predefined::DataModelTypes::UNIFORM:
      testReader(argv[2], dataModelStr);
      validateSchema(argv[2], dataModelStr, argv[3]);
      break;
    case adis::predefined::DataModelTypes::RECTILINEAR:
      testReader(argv[2], dataModelStr);
      validateSchema(argv[2], dataModelStr, argv[3]);
      break;
    case adis::predefined::DataModelTypes::UNSTRUCTURED:
      testReader(argv[2], dataModelStr);
      validateSchema(argv[2], dataModelStr, argv[3]);
      break;
    case adis::predefined::DataModelTypes::UNSTRUCTURED_SINGLE:
      testReader(argv[2], dataModelStr);
      validateSchema(argv[2], dataModelStr, argv[3]);
      break;
    case adis::predefined::DataModelTypes::XGC:
      testReader(argv[2], dataModelStr);
      validateSchema(argv[2], dataModelStr, argv[3]);
      break;
    default:
      throw std::runtime_error("Not a supported data model");
  }

#ifdef ADIS_USE_MPI
  MPI_Finalize();
#endif
  return 0;
}
