//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataSetReader.h>
#include <adis/xgc/StorageXGC.h>

#include <vtkm/cont/CellSetExtrude.h>
#include <vtkm/cont/ColorTable.h>
#include <vtkm/cont/Initialize.h>
#include <vtkm/filter/CleanGrid.h>
#ifdef USE_VTKM_RENDERING
#include <vtkm/rendering/Camera.h>
#include <vtkm/rendering/Scene.h>
#include <vtkm/rendering/MapperPoint.h>
#include <vtkm/rendering/CanvasRayTracer.h>
#include <vtkm/rendering/View3D.h>
#endif
#include <vtkm/io/writer/VTKDataSetWriter.h>

#ifdef ADIS_USE_MPI
#include <mpi.h>
#endif

#include <string>
#include <unordered_map>
#include <vector>

int main(int argc, char **argv)
{
  if (argc < 3)
  {
    // output method is 1 for writing VTK files (default), 0 to use VTK-m rendering
    std::cerr << "Usage ./xgc <name of the json file> <path of data source folder> <optional output method>\n";
    std::cerr << "Example 1: ./xgc ~/adis/examples/xgc/xgc-reader.json ~/adis/tests/data 0\n\tThis outputs the data in .png format using VTK-m rendering.\n";
    std::cerr << "Example 2: ./xgc ~/adis/examples/xgc/xgc-reader.json ~/adis/tests/data 1\n\tThis outputs the data in .vtk format.\n";
    return 1;
  }

#ifdef ADIS_USE_MPI
  MPI_Init(&argc, &argv);
#endif

  int retVal = 0;
  bool outputVTK = true;
  if (argc == 4)
  {
    outputVTK = std::atoi(argv[3]) == 1;
  }

  std::string dataModelFile = std::string(argv[1]);
  std::cout << "data-model: " << dataModelFile << "\n";

  adis::io::DataSetReader reader(dataModelFile);

  std::unordered_map<std::string, std::string> paths;

  // This would be so much easier with C++17 filesystem!
  std::string data_dir = std::string(argv[2]);
  if (data_dir.back() != '/') {
    data_dir += '/';
  }
  paths["mesh"] = data_dir;
  paths["3d"] = data_dir;
  paths["bfield"] = data_dir;

  auto metaData = reader.ReadMetaData(paths);
  auto& nBlocks =
    metaData.Get<adis::metadata::Size>(adis::keys::NUMBER_OF_BLOCKS());
  std::cout << "num blocks " << nBlocks.NumberOfItems << std::endl;

  adis::metadata::MetaData selections;
  adis::metadata::Index idx(2);
  selections.Set(adis::keys::STEP_SELECTION(), idx);

  using FieldInfoType =
    adis::metadata::Vector<adis::metadata::FieldInformation>;
  FieldInfoType fieldSelection;
  fieldSelection.Data.push_back(
    adis::metadata::FieldInformation(
      "pot0", vtkm::cont::Field::Association::POINTS));
  fieldSelection.Data.push_back(
    adis::metadata::FieldInformation(
      "potm0", vtkm::cont::Field::Association::POINTS));
  fieldSelection.Data.push_back(
    adis::metadata::FieldInformation(
      "dpot", vtkm::cont::Field::Association::POINTS));
  fieldSelection.Data.push_back(
    adis::metadata::FieldInformation(
      "node_data1", vtkm::cont::Field::Association::POINTS));

  selections.Set(adis::keys::FIELDS(), fieldSelection);
  adis::metadata::Vector<size_t> blockSelection;
  blockSelection.Data.push_back(0);
  blockSelection.Data.push_back(1);
  selections.Set(adis::keys::BLOCK_SELECTION(), blockSelection);

  vtkm::cont::PartitionedDataSet output =
    reader.ReadDataSet(paths, selections);

  if (output.GetNumberOfPartitions() == 0)
  {
#ifdef ADIS_USE_MPI
    MPI_Finalize();
#endif
    return 0;
  }
  vtkm::cont::DataSet ds = output.GetPartition(0);
  //ds.PrintSummary(std::cout);

  // check that rz was read correctly into the coordinate system
  std::cout << "--Checking Coordinate System--" << std::endl;
  std::cout << "num coord systems: " << ds.GetNumberOfCoordinateSystems() << std::endl;
  vtkm::cont::CoordinateSystem coordSys = ds.GetCoordinateSystem(0);
  std::cout << "coord sys # of points: " << coordSys.GetNumberOfPoints() << std::endl;
  auto range = coordSys.GetRangeAsArrayHandle();
  auto csRangePortal = range.ReadPortal();
  std::cout << "Min: " << csRangePortal.Get(0).Min << ", Max: " <<
    csRangePortal.Get(0).Max << std::endl;
  std::cout << "Min: " << csRangePortal.Get(1).Min << ", Max: " <<
    csRangePortal.Get(1).Max << std::endl;
  std::cout << "Min: " << csRangePortal.Get(2).Min << ", Max: " <<
    csRangePortal.Get(2).Max << std::endl;

  std::cout << "--Checking CellSet--" << std::endl;
  auto& cellSet = ds.GetCellSet();
  //std::cout << "Number of planes: " << cellSet.GetNumberOfPlanes() << std::endl;
  std::cout << "Total Cells: " << cellSet.GetNumberOfCells() << std::endl;
  std::cout << "Total Points: " << cellSet.GetNumberOfPoints() << std::endl;
  vtkm::Id ptids[6];
  if (cellSet.IsType<vtkm::cont::CellSetExtrude>())
  {
    auto& csXGC = cellSet.Cast<vtkm::cont::CellSetExtrude>();
    csXGC.GetCellPointIds(0, ptids);
    std::cout << "Cell 0 Point Ids: " << ptids[0] << ", " << ptids[1] << ", " <<
      ptids[2] << ", " << ptids[3] << ", " << ptids[4] << ", " << ptids[5] << std::endl;
  }

  std::cout << "--Checking Fields--" << std::endl;
  vtkm::IdComponent nFields = ds.GetNumberOfFields();
  std::cout << "nFields: " << nFields << std::endl;

  if (ds.HasField("pot0"))
  {
    const auto& arrHandle
      = ds.GetField("pot0").GetData().Cast<vtkm::cont::ArrayHandle<
      double, vtkm::cont::internal::StorageTagXGCPlane> >();
    vtkm::cont::ArrayHandle<vtkm::Range> rangeArray =
      vtkm::cont::ArrayRangeCompute(arrHandle);
    auto rangePortal = rangeArray.ReadPortal();
    std::cout << "pot0 Min: " << rangePortal.Get(0).Min << ", Max: " <<
      rangePortal.Get(0).Max << std::endl;

    auto pot0Portal = arrHandle.ReadPortal();
    std::cout << pot0Portal.GetNumberOfValues() << std::endl;
  }

  if (ds.HasField("potm0"))
  {
    const auto& potmAH
      = ds.GetField("potm0").GetData().Cast<vtkm::cont::ArrayHandle<
      double, vtkm::cont::internal::StorageTagXGCPlane> >();
    auto rangeArray = vtkm::cont::ArrayRangeCompute(potmAH);
    auto rangePortal = rangeArray.ReadPortal();
    std::cout << "potm0 Min: " << rangePortal.Get(0).Min << ", Max: " <<
      rangePortal.Get(0).Max << std::endl;
  }

  if (ds.HasField("dpot"))
  {
    const auto& dpotAH
      = ds.GetField("dpot").GetData().Cast<vtkm::cont::ArrayHandle<
      double, vtkm::cont::internal::StorageTagXGCPlane> >();
    auto rangeArray = vtkm::cont::ArrayRangeCompute(dpotAH);
    auto rangePortal = rangeArray.ReadPortal();
    std::cout << "dpot Min: " << rangePortal.Get(0).Min << ", Max: " <<
      rangePortal.Get(0).Max << std::endl;
  }

  if (ds.HasField("node_data1"))
  {
    const auto& nd1
      = ds.GetField("node_data1").GetData().Cast<vtkm::cont::ArrayHandle<
      double, vtkm::cont::internal::StorageTagXGCPlane> >();
    auto rangeArray = vtkm::cont::ArrayRangeCompute(nd1);
    auto rangePortal = rangeArray.ReadPortal();
    std::cout << "node_data1 Min: " << rangePortal.Get(0).Min << ", Max: " <<
      rangePortal.Get(0).Max << std::endl;
  }

  vtkm::filter::CleanGrid clean;
  auto outputDataPDS = clean.Execute(output);
  if (outputVTK)
  {
    auto& outputData = outputDataPDS.GetPartition(0);
    std::cout << "writing output in vtk format...\n";
    vtkm::io::VTKDataSetWriter writer("xgc-output.vtk");
    writer.WriteDataSet(outputData);
  }
#ifdef USE_VTKM_RENDERING
  else
  {
    std::cout << "rendering image using vtk-m...\n";
    // Create a mapper, canvas and view that will be used to render the scene
    vtkm::rendering::Scene scene;
    vtkm::rendering::MapperPoint mapper;
    vtkm::rendering::CanvasRayTracer canvas(1024, 1024);
    vtkm::rendering::Color bg(0.2f, 0.2f, 0.2f, 1.0f);
    vtkm::cont::ColorTable colorTable(vtkm::cont::ColorTable::Preset::COOL_TO_WARM);

    // Render an image of the output
    std::cout << "Rendering image\n";
    for (vtkm::Id i = 0; i < outputDataPDS.GetNumberOfPartitions(); ++i)
    {
      auto& outputData = outputDataPDS.GetPartition(i);
      std::cout << "Adding partition " << i << std::endl;
      scene.AddActor(vtkm::rendering::Actor(outputData.GetCellSet(),
            outputData.GetCoordinateSystem(),
            outputData.GetField("dpot"),
            colorTable));
    }
    vtkm::rendering::View3D view(scene, mapper, canvas);
    auto& camera = view.GetCamera();
    camera.SetViewUp(vtkm::Vec3f_64{0.5, -0.2, 0.8});
    camera.SetPosition(vtkm::Vec3f_32{-9, 5, 7});
    vtkm::Float32 zoom = 0.5;
    camera.Zoom(zoom);
    view.Paint();
    std::string filename = "xgc.png";
    view.SaveAs(filename);
  }
#endif
#ifdef ADIS_USE_MPI
  MPI_Finalize();
#endif
  return retVal;
}
