//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataSetReader.h>
#include <vtkm/cont/CellSetExtrude.h>
#include <vtkm/cont/ColorTable.h>
#include <vtkm/filter/CleanGrid.h>

#ifdef USE_VTKM_RENDERING
#include <vtkm/rendering/Camera.h>
#include <vtkm/rendering/Scene.h>
#include <vtkm/rendering/MapperPoint.h>
#include <vtkm/rendering/CanvasRayTracer.h>
#include <vtkm/rendering/View3D.h>
#endif

#ifdef ADIS_USE_MPI
#include <mpi.h>
#endif

#include <string>
#include <unordered_map>
#include <vector>

using FieldInfoType =
  adis::metadata::Vector<adis::metadata::FieldInformation>;

int main(int argc, char** argv)
{
  if (argc != 3)
  {
      std::cout << "./xgc <name of the json file> <path of data source folder>"
                << "\n";
      return 0;
  }

#ifdef ADIS_USE_MPI
  MPI_Init(&argc, &argv);
#endif

  std::string dataModelFile = std::string(argv[1]);
  adis::io::DataSetReader reader(dataModelFile);
  std::cout << "XGC Reader started\n";

  std::unordered_map<std::string, std::string> paths;
  // Data we want to read from file
  paths["mesh"] = std::string(argv[2]);
  paths["bfield"] = std::string(argv[2]);
  paths["oneddiag"] = std::string(argv[2]);

  // Data to read using SST
  paths["3d"] = std::string(argv[2]);
  adis::DataSourceParams params;
  params["engine_type"] = "SST";
  reader.SetDataSourceParameters("3d", params);

  size_t step = 0;
  while (true)
  {
    reader.PrepareNextStep(paths);
    auto metaData = reader.ReadMetaData(paths);
    adis::metadata::MetaData selections;

    FieldInfoType fieldSelection;
    fieldSelection.Data.push_back(
      adis::metadata::FieldInformation(
        "pot0", vtkm::cont::Field::Association::POINTS));
    fieldSelection.Data.push_back(
      adis::metadata::FieldInformation(
        "potm0", vtkm::cont::Field::Association::POINTS));
    fieldSelection.Data.push_back(
      adis::metadata::FieldInformation(
        "dpot", vtkm::cont::Field::Association::POINTS));
    fieldSelection.Data.push_back(
      adis::metadata::FieldInformation(
        "node_data1", vtkm::cont::Field::Association::POINTS));
    selections.Set(adis::keys::FIELDS(), fieldSelection);

    auto output = reader.ReadStep(paths, selections);

    if (output.GetNumberOfPartitions() == 0)
    {
      continue;
    }
    vtkm::filter::CleanGrid clean;
    auto outputDataPDS = clean.Execute(output);

#ifdef USE_VTKM_RENDERING
    // Create a mapper, canvas and view that will be used to render the scene
    vtkm::rendering::Scene scene;
    vtkm::rendering::MapperPoint mapper;
    vtkm::rendering::CanvasRayTracer canvas(1024, 1024);
    vtkm::rendering::Color bg(0.2f, 0.2f, 0.2f, 1.0f);
    vtkm::cont::ColorTable colorTable(vtkm::cont::ColorTable::Preset::COOL_TO_WARM);

    // Render an image of the output
    std::cout << "Rendering image\n";
    for (vtkm::Id i = 0; i < outputDataPDS.GetNumberOfPartitions(); ++i)
    {
      auto& outputData = outputDataPDS.GetPartition(i);
      scene.AddActor(vtkm::rendering::Actor(outputData.GetCellSet(),
            outputData.GetCoordinateSystem(),
            outputData.GetField("dpot"),
            colorTable));
    }
    vtkm::rendering::View3D view(scene, mapper, canvas);
    auto& camera = view.GetCamera();
    camera.SetViewUp(vtkm::Vec3f_64{0.5, -0.2, 0.8});
    camera.SetPosition(vtkm::Vec3f_32{-9, 5, 7});
    vtkm::Float32 zoom = 0.5;
    camera.Zoom(zoom);
    view.Paint();
    std::string filename = "xgc-step-" + std::to_string(step++) + ".ppm";
    view.SaveAs(filename);
    std::cout << __FILE__ << ":" << __LINE__ << ": File " << filename <<
      " written, going to next iteration of infinite loop." << std::endl;
#endif
  }

#ifdef ADIS_USE_MPI
  MPI_Finalize();
#endif
  return 0;
}
