//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#ifndef adis_datamodel_InternalMetadataSource_H
#define adis_datamodel_InternalMetadataSource_H

#include <adis/DataSource.h>
#include <adis/predefined/SupportedDataModels.h>

#include <memory>
#include <string>
#include <vector>

namespace adis
{
namespace predefined
{

/// \brief InternalMetadataSource is a DataSource that has attributes containing
/// info that can be used to generate a data model (instead of providing ADIS
/// with a user-created data model).
class InternalMetadataSource
{
public:
  /// Constructor. filename is a path to the file containing the attribute
  /// information
  InternalMetadataSource(const std::string& filename);

  ~InternalMetadataSource();

  /// Get the name of the data model for ADIS to generate
  std::string GetDataModelName(const std::string& attrName = "ADIS_Data_Model");

  /// Same as GetDataModelName, except the string is converted to an DataModelTypes enum.
  DataModelTypes GetDataModelType(const std::string& attrName = "ADIS_Data_Model");

  /// Gets the cell type. Not used by all data models
  std::string GetDataModelCellType(const std::string& attrName = "ADIS_Cell_Type");

  /// Reads the attribute specified by attrName
  template <typename AttributeType>
  std::vector<AttributeType> GetAttribute(const std::string& attrName);

  /// Gets the type of the attribute specified by attrName
  std::string GetAttributeType(const std::string& attrName);

private:
  std::shared_ptr<adis::io::DataSource> Source = nullptr;
};

template <typename AttributeType>
std::vector<AttributeType> InternalMetadataSource::GetAttribute(
  const std::string& attrName)
{
  auto attr = this->Source->ReadAttribute<AttributeType>(attrName);
  return attr;
}

}
}

#endif
