//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/predefined/DataModelFactory.h>
#include <adis/predefined/InternalMetadataSource.h>
#include <adis/predefined/SupportedDataModels.h>

namespace adis
{
namespace predefined
{

DataModelFactory* DataModelFactory::Instance = nullptr;
bool DataModelFactory::Destroyed = false;

DataModelFactory::~DataModelFactory()
{
  Instance = nullptr;
  Destroyed = true;
}

DataModelFactory& DataModelFactory::GetInstance()
{
  if (!Instance)
  {
    if (Destroyed)
    {
      throw std::runtime_error("Dead reference to DataModelFactory singleton");
    }
    else
    {
      CreateInstance();
    }
  }
  return *Instance;
}

void DataModelFactory::CreateInstance()
{
  static DataModelFactory theInstance;
  Instance = &theInstance;
}

bool DataModelFactory::RegisterDataModel(DataModelTypes modelId, CreateDataModelCallback createFn)
{
  return this->Callbacks.insert(CallbackMap::value_type(modelId, createFn)).second;
}

bool DataModelFactory::UnregisterDataModel(DataModelTypes modelId)
{
  return this->Callbacks.erase(modelId) == 1;
}

std::shared_ptr<PredefinedDataModel> DataModelFactory::CreateDataModel(
  std::shared_ptr<InternalMetadataSource> source)
{
  auto modelId = source->GetDataModelType();
  auto it = this->Callbacks.find(modelId);
  if (it == this->Callbacks.end())
  {
    throw std::runtime_error("Unknown data model ID provided to ADIS for selecting predefined data model");
  }
  return (it->second)(source);
}

}
}
