//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <vector>
#include <map>
#include <iostream>
#include <numeric>
#include <typeinfo>

#include <vtkm/CellShape.h>
#include <vtkm/cont/ArrayHandle.h>

#include <adios2.h>

#include <vtkm/io/reader/VTKDataSetReader.h>
#include <vtkm/io/writer/VTKDataSetWriter.h>
#include <vtkm/cont/DataSetBuilderExplicit.h>
#include <vtkm/cont/DataSetFieldAdd.h>
#include <vtkm/cont/Field.h>
#include <vtkm/cont/CoordinateSystem.hxx>
#include <vtkm/filter/CleanGrid.h>

#include <adis/xgc/PointsXGC.h>
#include <adis/xgc/PointsXGC.hxx>
#include <adis/xgc/ArrayHandleXGCPointCoordinates.h>
#include <vtkm/cont/CellSetExtrude.h>
#include <vtkm/filter/PolicyExtrude.h>

#include <vtkNew.h>
#include <vtkUnstructuredGrid.h>
#include <vtkDataSetWriter.h>

#include <vtkmlib/ArrayConverters.h>
#include <vtkmlib/UnstructuredGridConverter.h>
#include <vtkmlib/PolyDataConverter.h>

#define adisTemplateMacro(call)                     \
    switch (type[0])                                \
    {                                               \
    case 'c':                                       \
    {                                               \
        using adis_TT = char;                       \
        return call;                                \
        break;                                      \
    }                                               \
    case 'f':                                       \
    {                                               \
        using adis_TT = float;                      \
        return call;                                \
        break;                                      \
    }                                               \
    case 'd':                                       \
    {                                               \
        using adis_TT = double;                     \
        return call;                                \
        break;                                      \
    }                                               \
    case 'i':                                       \
    {                                               \
        using adis_TT = vtkm::Int32;                \
        return call;                                \
        break;                                      \
    }                                               \
    case 'l':                                       \
        if (type == "long long int")                \
        {                                           \
            using adis_TT = vtkm::Id;               \
            return call;                            \
        }                                           \
        else if (type == "long int")                \
        {                                           \
            using adis_TT = long int;               \
            return call;                            \
        }                                           \
        break;                                      \
    case 's':                                       \
        if (type == "short")                        \
        {                                           \
            using adis_TT = long int;               \
            return call;                            \
        }                                           \
        else if (type == "signed char")             \
        {                                           \
            using adis_TT = signed char;            \
            return call;                            \
        }                                           \
        break;                                      \
    case 'u':                                       \
        if (type == "unsigned char")                \
        {                                           \
            using adis_TT = unsigned char;          \
            return call;                            \
        }                                           \
        else if (type == "unsigned int")            \
        {                                           \
            using adis_TT = unsigned int;           \
            return call;                            \
        }                                           \
        else if (type == "unsigned long int")       \
        {                                           \
            using adis_TT = unsigned long int;      \
            return call;                            \
        }                                           \
        else if (type == "unsigned long long int")  \
        {                                           \
            using adis_TT = unsigned long long int; \
            return call;                            \
        }                                           \
        break;                                      \
    }

template <typename VariableType, typename VecType>
vtkm::cont::VariantArrayHandle AllocateArrayHandle(
    size_t bufSize, VariableType *&buffer)
{
    vtkm::cont::internal::Storage<
        VecType, vtkm::cont::StorageTagBasic>
        storage;
    storage.Allocate(bufSize);
    buffer = reinterpret_cast<VariableType *>(storage.GetArray());
    return vtkm::cont::ArrayHandle<VecType>(std::move(storage));
}

template <typename VariableType>
vtkm::cont::VariantArrayHandle ReadVariableInternal(
    adios2::Engine &bpReader,
    adios2::Variable<VariableType> &varADIOS2,
    size_t blockId,
    size_t cellDim,
    bool flatenRead)
{
    auto blocksInfo = bpReader.BlocksInfo(varADIOS2, 0);
    const auto &shape = blocksInfo[blockId].Count;

    std::cout << "shape: " << shape.size() << "\n";
    size_t valueNum = 1;
    for (auto n : shape)
    {
        valueNum *= n;
    }

    if (flatenRead)
    {
        std::cout << "flattern read for " << varADIOS2.Name() << ", elem num is " << valueNum << std::endl;
    }

    //std::cout << "value num: " << valueNum << std::endl;

    vtkm::cont::VariantArrayHandle retVal;
    VariableType *buffer = nullptr;
    //all use same size

    if (cellDim == 1 || flatenRead)
    {
        //for one array in flatten pattern, the cellNum should be the first parameter
        retVal = AllocateArrayHandle<VariableType, VariableType>(valueNum, buffer);
    }
    else if (cellDim == 2)
    {
        retVal = AllocateArrayHandle<VariableType, vtkm::Vec<VariableType, 2>>(shape[0], buffer);
    }
    else if (cellDim == 3)
    {

        retVal = AllocateArrayHandle<VariableType, vtkm::Vec<VariableType, 3>>(shape[0], buffer);
    }
    else
    {
        throw std::runtime_error("unsuported the celldim " + std::to_string(cellDim));
    }

    //std::cout << " allocate array handle ok " << std::endl;

    bpReader.Get(varADIOS2, buffer, adios2::Mode::Sync);
    //difference between Sync and Deffered
    //bpReader.Get(varADIOS2, testbuff.data(), adios2::Mode::Sync);

    //vtkm::cont::ArrayHandle<VariableType> arrayHandle(std::move(storage));

    //retVal = arrayHandle;

    return retVal;
}

template <typename VariableType>
std::vector<vtkm::cont::VariantArrayHandle> ReadVariableBlocksInternal(
    adios2::IO &adiosIO,
    adios2::Engine &bpReader,
    const std::string &varName,
    size_t cellDim,
    bool flatenRead)
{

    auto varADIOS2 =
        adiosIO.InquireVariable<VariableType>(varName);

    //if varADIOS2 is nullpointer, not found
    if (!varADIOS2)
    {
        std::cout << __PRETTY_FUNCTION__ << std::endl;
        throw std::runtime_error(varName + "is not in type " + typeid(VariableType).name());
    }

    auto blocksInfo = bpReader.BlocksInfo(varADIOS2, 0);

    //how many blocks needed to be read by adios file
    std::vector<size_t> blocksToReallyRead;

    //one adios variable may includes multiple blocks
    size_t nBlocks = blocksInfo.size();
    blocksToReallyRead.resize(nBlocks);
    std::iota(blocksToReallyRead.begin(),
              blocksToReallyRead.end(),
              0);

    std::cout << "blocks size: " << nBlocks << std::endl;

    //const std::vector<size_t> &blocksToRead =
    //    blocksToReallyRead = blocksToRead;

    std::vector<vtkm::cont::VariantArrayHandle> arrays;
    arrays.reserve(blocksToReallyRead.size());

    for (auto blockId : blocksToReallyRead)
    {
        varADIOS2.SetBlockSelection(blockId);
        arrays.push_back(
            ReadVariableInternal<VariableType>(bpReader, varADIOS2, blockId, cellDim, flatenRead));
    }

    bpReader.Close();

    return arrays;
}

std::vector<vtkm::cont::VariantArrayHandle> loadVaraible(std::string fname, std::string varName, size_t cellDim, bool flatternRead)
{
    //init adios

    std::unique_ptr<adios2::ADIOS> Adios = nullptr;

    Adios.reset(
        new adios2::ADIOS(MPI_COMM_WORLD, adios2::DebugON));
    adios2::IO AdiosIO = Adios->DeclareIO("adios-io-read");
    AdiosIO.SetEngine("bp3");

    adios2::Engine BpReader = AdiosIO.Open(fname, adios2::Mode::Read);

    //AvailVars and the AvailAtts are the info that generated by the bpls
    std::map<std::string, adios2::Params> AvailVars = AdiosIO.AvailableVariables();
    std::map<std::string, adios2::Params> AvailAtts = AdiosIO.AvailableAttributes();

    auto itr = AvailVars.find(varName);
    if (itr == AvailVars.end())
    {
        throw std::runtime_error("Variable " + varName + " was not found.");
    }
    const std::string &type = itr->second["Type"];

    std::cout << "type: " << type << std::endl;

    //  adisTemplateMacro(ReadVariableBlocksInternal<adis_TT>(
    //  this->AdiosIO, this->BpReader, varName, selections));
    //the return value of this array is the vector of the dynamicArrayHandle
    adisTemplateMacro(ReadVariableBlocksInternal<adis_TT>(
        AdiosIO, BpReader, varName, cellDim, flatternRead));
}

int main(int argc, char **argv)
{
    MPI_Init(&argc, &argv);

    if(argc!=2){
        std::cout << "<executable> <data path>"<< std::endl;
        return 0;
    }

    std::string xgcDataDir = argv[1];
    std::string xgcMesh = xgcDataDir + "/xgc.mesh.bp";
    std::string xgcDeta = xgcDataDir + "/xgc.3d.00001.bp";

    //TODO this is supposed to be accuqired from the json files

    //load the coordinates
    std::string varName = "rz";
    //use 2 for inner data structure (there is only x y here)
    size_t cellDim = 2;
    std::cout << "-----start loading " << varName << "------" << std::endl;
    std::vector<vtkm::cont::VariantArrayHandle> coorArrays = loadVaraible(xgcMesh, varName, cellDim, true);

    //use functor to set the two dimentional array into three dimention
    //the z value is 0
    //generate the new array by copy currently
    //TODO the better way is to build a new arrayHandle and extract the x y z from the original class and init the variable
    // how to transfer into type of vtkm::cont::ArrayHandle<vtkm::Vec<T, 3>>??
    //using type2DCoor = vtkm::cont::ArrayHandle<vtkm::Vec<double, 2>>;
    using typeCoor = vtkm::cont::ArrayHandle<double>;
    typeCoor arrayCoor;
    coorArrays[0].CopyTo(arrayCoor);

    //fake loading the single varaible
    vtkm::Id nphi = 8;

    //load the cellSet data (connectivity)
    varName = "nd_connect_list";
    //use one dim inner structure
    cellDim = 1;

    std::cout << "-----start loading " << varName << "------" << std::endl;
    std::vector<vtkm::cont::VariantArrayHandle> connArrays = loadVaraible(xgcMesh, varName, cellDim, true);
    //cast to inner arrayHandler
    vtkm::cont::ArrayHandle<vtkm::Int32> connectivityAH;
    connArrays[0].CopyTo(connectivityAH);

    varName = "nextnode";
    cellDim = 1;
    std::cout << "-----start loading " << varName << "------" << std::endl;
    std::vector<vtkm::cont::VariantArrayHandle> nnodeArrays = loadVaraible(xgcMesh, varName, cellDim, false);
    //cast to inner arrayHandler
    vtkm::cont::ArrayHandle<vtkm::Int32> nnodeAH;
    nnodeArrays[0].CopyTo(nnodeAH);

    bool isCylindrical = false;
    vtkm::Id pointNum = arrayCoor.GetNumberOfValues();

    std::cout<< "debug point num " << pointNum << std::endl;

    //generate the coordinates
    auto coords = vtkm::cont::make_ArrayHandleXGCPointCoordinates(
        reinterpret_cast<double *>(arrayCoor.GetStorage().GetBasePointer()),
        pointNum, nphi, isCylindrical, vtkm::CopyFlag::On);

    std::vector<vtkm::cont::CoordinateSystem> coordSystems;
    coordSystems.push_back(vtkm::cont::CoordinateSystem("test_coords", coords));

    std::cout << "ok to push back the coords"
              << "\n";

    //generate the cellset

    auto cells = vtkm::cont::make_CellSetExtrude(
        connectivityAH,
        coords,
        nnodeAH,
        true);

    //load the xgc Fields, load 8 field, need to be transposed

    varName = "dpot";
    cellDim = 1;
    std::cout << "-----start loading " << varName << "------" << std::endl;
    std::vector<vtkm::cont::VariantArrayHandle> fieldArray = loadVaraible(xgcDeta, varName, cellDim, true);
    int sizeOuter = fieldArray.size();
    int sizeinner = fieldArray[0].GetNumberOfValues();
    std::cout << "size of the fieldArray outer: " << sizeOuter << std::endl;
    std::cout << "size of the fieldArray inner: " << sizeinner << std::endl;

    vtkm::cont::ArrayHandle<double> largeFiled;
    largeFiled.Allocate(sizeOuter * sizeinner);

    for (int i = 0; i < sizeOuter; i++)
    {
        vtkm::cont::ArrayHandle<double> fieldOneBlock;
        fieldArray[i].CopyTo(fieldOneBlock);

        vtkm ::cont ::Algorithm ::CopySubRange(fieldOneBlock, 0 , sizeinner,  largeFiled, i * sizeinner);
    }

    //generate the data set

    vtkm::cont::DataSet dataSet;
    dataSet.SetCellSet(cells);
    dataSet.AddCoordinateSystem(
        vtkm::cont::CoordinateSystem("test_coords", coords));

    dataSet.AddField(vtkm::cont::Field("test_fields", vtkm::cont::Field::Association::POINTS, largeFiled));

    std::cout << "-----print cellset in loaded dataSet-----" << std::endl;
    dataSet.PrintSummary(std::cout);

    std::cout << "-----ok to print dataSet-----" << std::endl;

    vtkm::filter::CleanGrid clean;
    vtkm::cont::DataSet resultds = clean.Execute(dataSet, PolicyExtrude());

    std::cout << "-----ok to get clean dataSet-----" << std::endl;

    for (vtkm::Id i = 0; i < dataSet.GetNumberOfFields(); ++i)
    {
        clean.MapFieldOntoOutput(resultds, dataSet.GetField(i), PolicyExtrude());
    }

    auto outputinvtk = vtkUnstructuredGrid::New();
    vtkNew<vtkUnstructuredGrid> dummy;
    fromvtkm::Convert(resultds, outputinvtk, dummy.Get());


    std::string filename = "test_xgc.vtk";
    auto writer = vtkSmartPointer<vtkDataSetWriter>::New();
    writer->SetInputData(outputinvtk);
    writer->SetFileName(filename.c_str());
    writer->Write();

}
