//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataSetReader.h>
#include <adis/xgc/FieldXGCPlane.h>
#include <vtkm/cont/CellSetExtrude.h>
#include <vtkm/filter/PolicyExtrude.h>

#include <mpi.h>
#include <string>
#include <unordered_map>
#include <vector>

#include <vtkm/cont/Algorithm.h>
#include <vtkm/worklet/DispatcherMapTopology.h>
#include <vtkm/worklet/WorkletMapTopology.h>
#include <vtkm/worklet/ScatterPermutation.h>
#include <vtkm/filter/CleanGrid.h>

#include <vtkUnstructuredGrid.h>
#include <vtkUnstructuredGridWriter.h>
#include <vtkmlib/UnstructuredGridConverter.h>

class CheckTopology : public vtkm::worklet::WorkletMapPointToCell
{
public:
    using ControlSignature = void(CellSetIn cellset,
                                  FieldInCell trueConn,
                                  FieldOutCell outCells);
    using ExecutionSignature = void(IncidentElementIndices, _2, _3);
    using InputDomain = _1;
    using ScatterType = vtkm::worklet::ScatterPermutation<>;

    template <typename ConnType>
    VTKM_EXEC void operator()(const ConnType &inputConn,
                              const vtkm::Id3 &trueConn,
                              bool &isTrue) const
    {
        isTrue = true;
        for (int i = 0; i < 3; i++)
        {
            if (inputConn[i] != trueConn[i])
            {
                isTrue = false;
                break;
            }
        }
    }
};

int main(int argc, char **argv)
{

    if (argc != 3)
    {
        std::cout << "./xgcReaderByAdis <name of the json file> <path of data source folder>"
                  << "\n";
        return 0;
    }

    MPI_Init(&argc, &argv);

    int retVal = 0;

    std::string dataModelFile = std::string(argv[1]);
    std::cout << "data-model: " << dataModelFile << "\n";

    adis::io::DataSetReader reader(dataModelFile);
    std::cout << "xgc reader init ok \n";

    std::unordered_map<std::string, std::string> paths;

    paths["mesh"] = std::string(argv[2]);
    paths["3d"] = std::string(argv[2]);
    paths["bfield"] = std::string(argv[2]);
    paths["oneddiag"] = std::string(argv[2]);

    auto metaData = reader.ReadMetaData(paths);

    adis::metadata::MetaData selections;
    adis::metadata::Vector<size_t> blockSelection;
    blockSelection.Data.push_back(0);
    selections.Set(adis::keys::BLOCK_SELECTION(), blockSelection);
    adis::metadata::Index idx(2);
    selections.Set(adis::keys::STEP_SELECTION(), idx);

    using FieldInfoType =
        adis::metadata::Vector<adis::metadata::FieldInformation>;
    FieldInfoType fieldSelection;
    fieldSelection.Data.push_back(
        adis::metadata::FieldInformation(
            "pot0", vtkm::cont::Field::Association::POINTS));
    fieldSelection.Data.push_back(
        adis::metadata::FieldInformation(
            "potm0", vtkm::cont::Field::Association::POINTS));
    ////fieldSelection.Data.push_back(
    ////    adis::metadata::FieldInformation(
    ////        "dpot", vtkm::cont::Field::Association::POINTS));
    fieldSelection.Data.push_back(
        adis::metadata::FieldInformation(
            "node_data0", vtkm::cont::Field::Association::POINTS));
    //fieldSelection.Data.push_back(
    //    adis::metadata::FieldInformation(
    //        "node_data1", vtkm::cont::Field::Association::POINTS));

    selections.Set(adis::keys::FIELDS(), fieldSelection);

    vtkm::cont::PartitionedDataSet output =
        reader.ReadDataSet(paths, selections);

    std::cout << "-----ok to load dataSet-----" << std::endl;

    vtkm::cont::DataSet ds = output.GetPartition(0);
    ds.PrintSummary(std::cout);

    // check that rz was read correctly into the coordinate system
    std::cout << "--Checking Coordinate System--" << std::endl;
    std::cout << "num coord systems: " << ds.GetNumberOfCoordinateSystems() << std::endl;
    vtkm::cont::CoordinateSystem coordSys = ds.GetCoordinateSystem(0);
    std::cout << "coord sys # of points: " << coordSys.GetNumberOfPoints() << std::endl;
    auto range = coordSys.GetRangeAsArrayHandle();
    auto csRangePortal = range.GetPortalConstControl();
    std::cout << "Min: " << csRangePortal.Get(0).Min << ", Max: " <<
      csRangePortal.Get(0).Max << std::endl;
    std::cout << "Min: " << csRangePortal.Get(1).Min << ", Max: " <<
      csRangePortal.Get(1).Max << std::endl;
    std::cout << "Min: " << csRangePortal.Get(2).Min << ", Max: " <<
      csRangePortal.Get(2).Max << std::endl;

    std::cout << "--Checking CellSet--" << std::endl;
    auto& cellSet = ds.GetCellSet();
    //std::cout << "Number of planes: " << cellSet.GetNumberOfPlanes() << std::endl;
    std::cout << "Total Cells: " << cellSet.GetNumberOfCells() << std::endl;
    std::cout << "Total Points: " << cellSet.GetNumberOfPoints() << std::endl;
    vtkm::Id ptids[6];
    if (cellSet.IsType<vtkm::cont::CellSetExtrude>())
    {
      auto& csXGC = cellSet.Cast<vtkm::cont::CellSetExtrude>();
      csXGC.GetCellPointIds(0, ptids);
      std::cout << "Cell 0 Point Ids: " << ptids[0] << ", " << ptids[1] << ", " <<
        ptids[2] << ", " << ptids[3] << ", " << ptids[4] << ", " << ptids[5] << std::endl;
    }

    vtkm::IdComponent nFields = ds.GetNumberOfFields();
    std::cout << "nFields: " << nFields << std::endl;

    const auto& arrHandle
      = ds.GetField("pot0").GetData().Cast<vtkm::cont::ArrayHandle<
        //double> >();
        double, vtkm::cont::internal::StorageTagExtrudePlane> >();
    vtkm::cont::ArrayHandle<vtkm::Range> rangeArray =
      vtkm::cont::ArrayRangeCompute(arrHandle);
    auto rangePortal = rangeArray.GetPortalConstControl();
    std::cout << "pot0 Min: " << rangePortal.Get(0).Min << ", Max: " <<
      rangePortal.Get(0).Max << std::endl;

    auto pot0Portal = arrHandle.GetPortalConstControl();
    std::cout << pot0Portal.GetNumberOfValues() << std::endl;

    const auto& potmAH
      = ds.GetField("potm0").GetData().Cast<vtkm::cont::ArrayHandle<
        //double> >();
        double, vtkm::cont::internal::StorageTagExtrudePlane> >();
    rangeArray = vtkm::cont::ArrayRangeCompute(potmAH);
    rangePortal = rangeArray.GetPortalConstControl();
    std::cout << "potm0 Min: " << rangePortal.Get(0).Min << ", Max: " <<
      rangePortal.Get(0).Max << std::endl;

    ////const auto& dpotAH
    ////  = ds.GetField("dpot").GetData().Cast<vtkm::cont::ArrayHandle<
    ////    double, vtkm::cont::internal::StorageTagExtrudePlane> >();
    ////rangeArray = vtkm::cont::ArrayRangeCompute(dpotAH);
    ////rangePortal = rangeArray.GetPortalConstControl();
    ////std::cout << "dpot Min: " << rangePortal.Get(0).Min << ", Max: " <<
    ////  rangePortal.Get(0).Max << std::endl;

    //const auto& nd0
    //  = ds.GetField("node_data0").GetData().Cast<vtkm::cont::ArrayHandle<double> >();
    //rangeArray = vtkm::cont::ArrayRangeCompute(nd0);
    //rangePortal = rangeArray.GetPortalConstControl();
    //std::cout << "node_data0 Min: " << rangePortal.Get(0).Min << ", Max: " <<
    //  rangePortal.Get(0).Max << std::endl;

    //const auto& nd1
    //  = ds.GetField("node_data1").GetData().Cast<vtkm::cont::ArrayHandle<double> >();
    //rangeArray = vtkm::cont::ArrayRangeCompute(nd1);
    //rangePortal = rangeArray.GetPortalConstControl();
    //std::cout << "node_data1 Min: " << rangePortal.Get(0).Min << ", Max: " <<
    //  rangePortal.Get(0).Max << std::endl;


    vtkm::filter::CleanGrid clean;
    //auto result = clean.Execute(ds);
    auto result = clean.Execute(ds, vtkm::filter::PolicyExtrude());

    //for (vtkm::Id id = 0; id < ds.GetNumberOfFields(); ++id)
    //{
    //  clean.MapFieldOntoOutput(result, ds.GetField(id), vtkm::filter::PolicyExtrude());
    //}

    auto unstrGrid = vtkUnstructuredGrid::New();
    vtkNew<vtkUnstructuredGrid> dummy;
    fromvtkm::Convert(result, unstrGrid, dummy.Get());
    auto writer = vtkUnstructuredGridWriter::New();
    writer->SetFileName("xgc-output.vtk");
    writer->SetInputData(unstrGrid);
    writer->Write();

    MPI_Finalize();
    return retVal;
}
