
#ifndef adis_xgc_FieldXGCPlane_h
#define adis_xgc_FieldXGCPlane_h

#include <vtkm/internal/IndicesExtrude.h>

#include <vtkm/cont/ArrayHandle.h>


namespace vtkm
{
namespace exec
{

template <typename PortalType>
struct VTKM_ALWAYS_EXPORT ArrayPortalFieldXGCPlane
{
  using ValueType = typename PortalType::ValueType;

  VTKM_SUPPRESS_EXEC_WARNINGS
  VTKM_EXEC_CONT
  ArrayPortalFieldXGCPlane()
    : Portals()
    , NumberOfPlanes(0)
    , BlocksPerPlane(0) {};

  // TODO should numOfPlanes be just on this rank or total planes
  // for now total planes and num planes on this rank = Portals.size()
  ArrayPortalFieldXGCPlane(const std::vector<PortalType>& portals,
      vtkm::Id numOfPlanes, vtkm::Id blocksPerPlane)
    : Portals(portals)
    , NumberOfPlanes(numOfPlanes)
    , BlocksPerPlane(blocksPerPlane)
  {
  }

  VTKM_SUPPRESS_EXEC_WARNINGS
  VTKM_EXEC_CONT
  vtkm::Id GetNumberOfValues() const
  {
    vtkm::Id numValues = 0;
    for (auto& portal : this->Portals)
    {
      numValues += portal.GetNumberOfValues();
    }
    return numValues;
  }

  VTKM_SUPPRESS_EXEC_WARNINGS
  VTKM_EXEC_CONT
  ValueType Get(vtkm::Id index) const
  {
    vtkm::Id numValuesPerPlane = 0;
    for (vtkm::Id block = 0; block < this->BlocksPerPlane; ++block)
    {
      // sum the number of values on all blocks on a single plane
      numValuesPerPlane += this->Portals[block].GetNumberOfValues();
    }
    vtkm::Id planeIdx = index / numValuesPerPlane;
    vtkm::Id blockStart = planeIdx * this->BlocksPerPlane;
    vtkm::Id blockEnd = (planeIdx + 1) * this->BlocksPerPlane;
    vtkm::Id numPrevValues = numValuesPerPlane * planeIdx;
    vtkm::Id portalIdx = -1;
    vtkm::Id valueIdx = -1;
    for (vtkm::Id block = blockStart; block < blockEnd; ++block)
    {
      if (index >= numPrevValues &&
          index < numPrevValues + this->Portals[block].GetNumberOfValues())
      {
        portalIdx = block;
        valueIdx = index - numPrevValues;
        break;
      }
      numPrevValues += this->Portals[block].GetNumberOfValues();
    }
    if (portalIdx == -1 || valueIdx == -1)
    {
      std::cout << "index: " << index << std::endl;
      std::cout << "this->GetNumValues(): " << this->GetNumberOfValues() << std::endl;
      std::cout << "BlocksPerPlane: " << this->BlocksPerPlane << std::endl;
      std::cout << "numValuesPerPlane: " << numValuesPerPlane << std::endl;
      std::cout << "planeIdx: " << planeIdx << std::endl;
      std::cout << "blockStart: " << blockStart << std::endl;
      std::cout << "blockEnd: " << blockEnd << std::endl;
      std::cout << "numPrevValues: " << numPrevValues << std::endl;
      throw vtkm::cont::ErrorBadValue("portalIdx and valueIdx are -1!");
    }
    return this->Portals[portalIdx].Get(valueIdx);
  }

  //VTKM_SUPPRESS_EXEC_WARNINGS
  //VTKM_EXEC_CONT
  //ValueType Get(vtkm::Id2 index) const
  //{
  //  return this->Portal.Get(index[0]);
  //}

  // TODO have to fix GetWedge
  VTKM_SUPPRESS_EXEC_WARNINGS
  VTKM_EXEC_CONT
  vtkm::Vec<ValueType, 6> GetWedge(const IndicesExtrude& index) const;

  VTKM_SUPPRESS_EXEC_WARNINGS
  VTKM_EXEC_CONT
  void Set(vtkm::Id vtkmNotUsed(index), const ValueType& vtkmNotUsed(value)) const {}

  VTKM_EXEC_CONT
  const PortalType GetPortal(vtkm::Id index) const { return this->Portals[index]; }

private:
  std::vector<PortalType> Portals;
  vtkm::Id NumberOfPlanes;
  vtkm::Id BlocksPerPlane;
};
}
} // vtkm::exec

namespace vtkm
{
namespace cont
{
namespace internal
{

struct VTKM_ALWAYS_EXPORT StorageTagFieldXGCPlane
{
};

template <typename T>
class VTKM_ALWAYS_EXPORT Storage<T, StorageTagFieldXGCPlane>
{
  using HandleType = vtkm::cont::ArrayHandle<T>;

public:
  using ValueType = T;

  using PortalConstType =
    vtkm::exec::ArrayPortalFieldXGCPlane<typename HandleType::PortalConstControl>;
  // Note that this array is read only, so you really should only be getting the const
  // version of the portal. If you actually try to write to this portal, you will
  // get an error.
  using PortalType = PortalConstType;

  Storage()
    : NumberOfPlanes(0), BlocksPerPlane(0)
  {
  }

  //Storage(const T* array, vtkm::Id arrayLength, vtkm::Int32 numberOfPlanes)
  //  : Array(vtkm::cont::make_ArrayHandle(array, arrayLength))
  //  , Length(static_cast<vtkm::Int32>(arrayLength)), NumberOfPlanes(numberOfPlanes)
  //{
  //  VTKM_ASSERT(this->Length >= 0);
  //}

  //Storage(const HandleType& array, vtkm::Int32 numberOfPlanes)
  //  : Array(array), Length(static_cast<vtkm::Int32>(array.GetNumberOfValues()))
  //  , NumberOfPlanes(numberOfPlanes)
  //{
  //  VTKM_ASSERT(this->Length >= 0);
  //}

  Storage(const std::vector<HandleType>& arrays, vtkm::Id numberOfPlanes,
      vtkm::Id blocksPerPlane)
    : Arrays(arrays), NumberOfPlanes(numberOfPlanes), BlocksPerPlane(blocksPerPlane)
  {
  }

  PortalType GetPortal()
  {
    throw vtkm::cont::ErrorBadType(
      "XGC ArrayHandles are read only. Cannot get writable portal.");
  }

  PortalConstType GetPortalConst() const
  {
    std::vector<typename HandleType::PortalConstControl> portals;
    portals.reserve(this->NumberOfPlanes * this->BlocksPerPlane);
    for (auto& array : this->Arrays)
    {
      portals.push_back(array.GetPortalConstControl());
    }
    return PortalConstType(portals, this->NumberOfPlanes, this->BlocksPerPlane);
  }

  vtkm::Id GetNumberOfValues() const
  {
    return this->GetNumberOfValuesPerPlane() * this->NumberOfPlanes;
  }

  vtkm::Id GetNumberOfValuesPerPlane() const
  {
    vtkm::Id total = 0;
    for (vtkm::Id block = 0; block < this->BlocksPerPlane; ++block)
    {
      total += this->Arrays[block].GetNumberOfValues();
    }
    return total;
  }

  vtkm::Int32 GetNumberOfPlanes() const
  {
    return this->NumberOfPlanes;
  }

  void Allocate(vtkm::Id vtkmNotUsed(numberOfValues))
  {
    throw vtkm::cont::ErrorBadType(
      "ArrayHandleXGCFieldPlane is read only. It cannot be allocated.");
  }

  void Shrink(vtkm::Id vtkmNotUsed(numberOfValues))
  {
    throw vtkm::cont::ErrorBadType(
      "ArrayHandleXGCFieldPlane is read only. It cannot shrink.");
  }

  void ReleaseResources()
  {
    // This request is ignored since we don't own the memory that was past
    // to us
  }

  const HandleType& GetArray(vtkm::Id index) const
  {
    return this->Arrays[index];
  }

  std::vector<HandleType> Arrays;
  vtkm::Id NumberOfPlanes;
  vtkm::Id BlocksPerPlane;
};

template <typename T, typename Device>
class VTKM_ALWAYS_EXPORT ArrayTransfer<T, internal::StorageTagFieldXGCPlane, Device>
{
  using HandleType = vtkm::cont::ArrayHandle<T>;

public:
  using ValueType = T;
  using StorageType = vtkm::cont::internal::Storage<T, internal::StorageTagFieldXGCPlane>;

  using PortalControl = typename StorageType::PortalType;
  using PortalConstControl = typename StorageType::PortalConstType;

  //meant to be an invalid writeable execution portal
  using PortalExecution = vtkm::exec::ArrayPortalFieldXGCPlane<
    typename HandleType::template ExecutionTypes<Device>::Portal>;
  using PortalConstExecution = vtkm::exec::ArrayPortalFieldXGCPlane<
    typename HandleType::template ExecutionTypes<Device>::PortalConst>;

  VTKM_CONT
  ArrayTransfer(StorageType* storage)
    : ControlData(storage)
  {
  }

  vtkm::Id GetNumberOfValues() const
  {
    return this->ControlData->GetNumberOfValues();
  }

  VTKM_CONT
  PortalConstExecution PrepareForInput(bool vtkmNotUsed(updateData))
  {
    std::vector<typename HandleType::template ExecutionTypes<Device>::PortalConst> portals;
    for (auto& array : this->ControlData->Arrays)
    {
      portals.push_back(array.PrepareForInput(Device()));
    }
    return PortalConstExecution(portals,
                                this->ControlData->NumberOfPlanes,
                                this->ControlData->BlocksPerPlane);
  }

  VTKM_CONT
  PortalExecution PrepareForInPlace(bool& vtkmNotUsed(updateData))
  {
    throw vtkm::cont::ErrorBadValue("ArrayHandleXGCFieldPlane read only. "
                                   "Cannot be used for in-place operations.");
  }

  VTKM_CONT
  PortalExecution PrepareForOutput(vtkm::Id vtkmNotUsed(numberOfValues))
  {
    throw vtkm::cont::ErrorBadValue(
      "ArrayHandleXGCFieldPlane read only. Cannot be used as output.");
  }

  VTKM_CONT
  void RetrieveOutputData(StorageType* vtkmNotUsed(storage)) const
  {
    throw vtkm::cont::ErrorInternal(
      "ArrayHandleXGCFieldPlane read only. "
      "There should be no occurance of the ArrayHandle trying to pull "
      "data from the execution environment.");
  }

  VTKM_CONT
  void Shrink(vtkm::Id vtkmNotUsed(numberOfValues))
  {
    throw vtkm::cont::ErrorBadValue("ArrayHandleXGCFieldPlane read only. Cannot shrink.");
  }

  VTKM_CONT
  void ReleaseResources()
  {
    // This request is ignored since we don't own the memory that was past
    // to us
  }

private:
  const StorageType* const ControlData;
};

}
}
} // vtkm::cont::internal

#endif
