//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#ifndef adis_datamodel_DataSourceManager_H_
#define adis_datamodel_DataSourceManager_H_

#include <adis/ADISTypes.h>
#include <adis/DataSource.h>
#include <adis/MetaData.h>

#include <unordered_map>

namespace adis
{
namespace io
{

using DataSourceType = adis::io::DataSource;
using DataSourcesType =
  std::unordered_map<std::string, std::shared_ptr<DataSourceType> >;

class DataSourceManager
{
public:
  void AddDataSource(const std::string& name, std::shared_ptr<DataSourceType> source);
  void SetDataSourceIO(const std::string& source, void* io);

  void SetDataSourceParameters(const std::string& source,
    const DataSourceParams& params);

  void ClearDataSources();

  size_t GetNumberOfBlocks(
    const std::string& source,
    const std::unordered_map<std::string, std::string>& paths,
    const std::string& varName);

  std::vector<vtkm::cont::VariantArrayHandle> ReadVariable(
    const std::string& source,
    const std::unordered_map<std::string, std::string>& paths,
    const std::string& variableName,
    const adis::metadata::MetaData& selections,
    adis::io::IsVector isItVector);

  std::vector<vtkm::cont::VariantArrayHandle> GetVariableDimensions(
    const std::string& source,
    const std::unordered_map<std::string, std::string>& paths,
    const std::string& varName,
    const adis::metadata::MetaData& selections);

  std::shared_ptr<DataSourceType>& find(const std::string& source);

  bool HasSource(const std::string& source);

  // TODO also a single source version?
  void DoAllReads();

  // TODO also a single source version
  void BeginStep(const std::unordered_map<std::string, std::string>& paths);
  void EndStep();

  // TODO i guess we should return in a non adios type
  adios2::Dims GetShape(
    const std::string& source,
    const std::unordered_map<std::string, std::string>& paths,
    const std::string& varName);

  const std::shared_ptr<DataSourceType>& OpenSource(
    const std::string& source,
    const std::unordered_map<std::string, std::string>& paths,
    const adis::metadata::MetaData& selections = adis::metadata::MetaData(),
    bool advanceStep = false);

  const std::shared_ptr<DataSourceType>& OpenSourceInSeries(
      const std::string& sourceName, size_t step,
      const std::unordered_map<std::string, std::string>& paths);

private:
  DataSourcesType DataSources;
};

}
}

#endif
