//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataSourceManager.h>
#include <iomanip>

namespace adis
{
namespace io
{

void DataSourceManager::AddDataSource(const std::string& name,
  std::shared_ptr<DataSourceType> source)
{
  this->DataSources[name] = source;
}

void DataSourceManager::ClearDataSources()
{
  this->DataSources.clear();
}

//std::shared_ptr<DataSourceType>& DataSourceManager::find(const std::string& source)
//{
//  auto it = this->DataSources.find(source);
//  if (it == this->DataSources.end())
//  {
//    throw std::runtime_error("Source name was not found in DataSources.");
//  }
//  return *(it->second);
//}

void DataSourceManager::SetDataSourceParameters(const std::string& source,
  const DataSourceParams& params)
{
  auto it = this->DataSources.find(source);
  if (it == this->DataSources.end())
  {
    throw std::runtime_error("Source name was not found in DataSources.");
  }
  auto& ds = *(it->second);
  ds.SetDataSourceParameters(params);
}

void DataSourceManager::SetDataSourceIO(const std::string& source, void* io)
{
  auto it = this->DataSources.find(source);
  if (it == this->DataSources.end())
  {
    throw std::runtime_error("Source name was not found in DataSources.");
  }
  auto& ds = *(it->second);
  ds.SetDataSourceIO(io);
}

size_t DataSourceManager::GetNumberOfBlocks(
  const std::string& source,
  const std::unordered_map<std::string, std::string>& paths,
  const std::string& varName)
{
  const auto& ds = this->OpenSource(source, paths);
  return ds->GetNumberOfBlocks(varName);
}

std::vector<vtkm::cont::VariantArrayHandle> DataSourceManager::ReadVariable(
  const std::string& source,
  const std::unordered_map<std::string, std::string>& paths,
  const std::string& variableName,
  const adis::metadata::MetaData& selections,
  adis::io::IsVector isItVector)
{
  const auto& ds = this->OpenSource(source, paths, selections);
  // Here if we have a step selection, remove it for series
  if (ds->StepMode == StepModes::Series && selections.Has(adis::keys::STEP_SELECTION()))
  {
    std::cout << "Removing step selection" << std::endl;
    adis::metadata::MetaData newSelections = selections;
    newSelections.Remove(adis::keys::STEP_SELECTION());
    return ds->ReadVariable(variableName, newSelections, isItVector);
  }
  return ds->ReadVariable(variableName, selections, isItVector);
}

std::vector<vtkm::cont::VariantArrayHandle> DataSourceManager::GetVariableDimensions(
  const std::string& source,
  const std::unordered_map<std::string, std::string>& paths,
  const std::string& varName,
  const adis::metadata::MetaData& selections)
{
  const auto& ds = this->OpenSource(source, paths, selections);
  return ds->GetVariableDimensions(varName, selections);
}

bool DataSourceManager::HasSource(const std::string& source)
{
  auto iter = this->DataSources.find(source);
  return iter != this->DataSources.end();
}

void DataSourceManager::DoAllReads()
{
  for(auto source : this->DataSources)
  {
    source.second->DoAllReads();
  }
}

void DataSourceManager::BeginStep(
  const std::unordered_map<std::string, std::string>& paths)
{
  for(auto source : this->DataSources)
  {
    auto& ds = *(source.second);
    std::string name = source.first;
    auto itr = paths.find(name);
    if (itr == paths.end())
    {
      throw std::runtime_error("BeginStep: Could not find data_source with name "
        + name + " among the input paths.");
    }
    std::string path = itr->second + ds.FileName;
    ds.OpenSource(path);
    ds.BeginStep();
  }
}

void DataSourceManager::EndStep()
{
  for(auto source : this->DataSources)
  {
    source.second->EndStep();
  }
}

adios2::Dims DataSourceManager::GetShape(
  const std::string& source,
  const std::unordered_map<std::string, std::string>& paths,
  const std::string& varName)
{
  // never need to advance file here since the shape of a variable should
  // stay constant across files
  const auto& ds = this->OpenSource(source, paths);
  return ds->GetShape(varName);
}

const std::shared_ptr<DataSourceType>& DataSourceManager::OpenSource(
    const std::string& source,
    const std::unordered_map<std::string, std::string>& paths,
    const adis::metadata::MetaData& selections/*= adis::metadata::MetaData()*/,
    bool advanceStep/*=false*/)
{
  auto itr = paths.find(source);
  if (itr == paths.end())
  {
    throw std::runtime_error("Could not find data_source with name "
        + source + " among the input paths.");
  }
  const auto& ds = this->DataSources[source];

  if (ds->StepMode == StepModes::Series)
  {
    if (selections.Has(adis::keys::STEP_SELECTION()))
    {
      size_t stepSel = selections.Get<adis::metadata::Index>(
          adis::keys::STEP_SELECTION()).Data;
      //ds->CloseAndAdvanceReader(stepSel);
       return this->OpenSourceInSeries(source, stepSel, paths);
    }
    //else
    //{
    //  ds->CloseAndAdvanceReader();
    //}
  }
  std::string path = itr->second + ds->FileName;

  ds->OpenSource(path);
  return this->DataSources[source];
}

const std::shared_ptr<DataSourceType>& DataSourceManager::OpenSourceInSeries(
    const std::string& sourceName, size_t step,
    const std::unordered_map<std::string, std::string>& paths)
{
  auto itr = paths.find(sourceName);
  if (itr == paths.end())
  {
    throw std::runtime_error("Could not find data_source with name "
        + sourceName + " among the input paths.");
  }
  const auto& originalSource = this->DataSources[sourceName];
  if (step == originalSource->FileNameFormat.CurrentStep)
  {
    return this->DataSources[sourceName];
  }

  auto source = std::make_shared<DataSourceType>();
  source->Mode = originalSource->Mode;
  source->StepMode = originalSource->StepMode;
  source->AdiosEngineType = originalSource->AdiosEngineType;
  source->SourceParams = std::move(originalSource->SourceParams);
  source->FileNameFormat = originalSource->FileNameFormat;
  std::stringstream ss;
  ss << source->FileNameFormat.Prefix;
  ss << std::setfill('0') << std::setw(source->FileNameFormat.NumberOfDigits) << step;
  ss << source->FileNameFormat.Suffix;
  std::cout << ss.str() << std::endl;
  source->FileName = ss.str();
  source->FileNameFormat.CurrentStep = step;
  this->DataSources[sourceName] = source;

  std::string path = itr->second + source->FileName;
  source->OpenSource(path);
  return this->DataSources[sourceName];
}

}
}
