//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataSetReader.h>

#include <ios>
#include <stdexcept>
#include <unordered_map>
#include <vector>

#include <rapidjson/document.h>
#include <rapidjson/filereadstream.h>
#include <rapidjson/error/en.h>

#include <vtkm/cont/CoordinateSystem.h>
#include <vtkm/cont/DataSet.h>
#include <vtkm/cont/DynamicCellSet.h>

#include <adis/DataSourceManager.h>
#include <adis/CellSet.h>
#include <adis/CoordinateSystem.h>
#include <adis/Field.h>
#include <adis/Keys.h>

namespace adis
{
namespace io
{

class DataSetReader::DataSetReaderImpl
{
public:
  DataSetReaderImpl(const std::string dataModelFileName,
    const Params& params)
  {
    this->ReadJSONFile(dataModelFileName);
    this->SetDataSourceParameters(params);
  }

  virtual ~DataSetReaderImpl()
  {
    this->Cleanup();
  }

  void Cleanup()
  {
    this->DataSources.ClearDataSources();
    this->CoordinateSystem.reset();
    this->CellSet.reset();
  }

  void SetDataSourceParameters(const Params& params)
  {
    for (const auto& p : params)
    {
      this->SetDataSourceParameters(p.first, p.second);
    }
  }

  void SetDataSourceParameters(const std::string source,
    const DataSourceParams& params)
  {
    this->DataSources.SetDataSourceParameters(source, params);
  }

  void SetDataSourceIO(const std::string source, void* io)
  {
    this->DataSources.SetDataSourceIO(source, io);
  }

  template <typename ValueType>
  void ProcessDataSources(const ValueType& dataSources)
  {
    for(auto& dataSource: dataSources)
    {
      if (!dataSource.IsObject())
      {
        throw std::runtime_error("data_sources must contain data_source objects.");
      }
      if (!dataSource.GetObject().HasMember("name"))
      {
        throw std::runtime_error("data_source objects must have name.");
      }
      std::string name = dataSource.GetObject()["name"].GetString();
      if(name.empty())
      {
        throw std::runtime_error("data_source name must be a non-empty string.");
      }
      if (!dataSource.GetObject().HasMember("filename_mode"))
      {
        throw std::runtime_error("data_source objects must have filename_mode.");
      }
      std::string filename_mode = dataSource.GetObject()["filename_mode"].GetString();
      if(filename_mode.empty())
      {
        throw std::runtime_error("data_source filename_mode must be a non-empty string.");
      }
      auto source = std::make_shared<DataSourceType>();
      if(filename_mode == "input")
      {
        source->Mode = adis::io::FileNameMode::Input;
      }
      else if(filename_mode == "relative")
      {
        source->Mode = adis::io::FileNameMode::Relative;
        if (!dataSource.GetObject().HasMember("filename"))
        {
          throw std::runtime_error("data_source objects must have filename.");
        }
        source->FileName = dataSource.GetObject()["filename"].GetString();
      }
      else
      {
        throw std::runtime_error("data_source filename_mode must be input or relative.");
      }

      std::string step_mode = "multi_step";
      if (dataSource.GetObject().HasMember("step_mode"))
      {
        step_mode = dataSource.GetObject()["step_mode"].GetString();
        if (step_mode.empty())
        {
          throw std::runtime_error("data_source step_mode must be a non-empty string if specified.");
        }
      }
      if (step_mode == "single_step")
      {
        source->StepMode = adis::io::StepModes::SingleStep;
      }
      else if (step_mode == "multi_step")
      {
        source->StepMode = adis::io::StepModes::MultiStep;
      }
      else if (step_mode == "series")
      {
        source->StepMode = adis::io::StepModes::Series;
        if (!dataSource.GetObject().HasMember("series_format"))
        {
          throw std::runtime_error("data_source series must have a series_format");
        }
        std::string format = dataSource.GetObject()["series_format"].GetString();
        if (format.empty())
        {
          throw std::runtime_error("data_source series_format must be a non-empty string.");
        }
        source->SetSeriesFormat(format);
      }
      else
      {
        throw std::runtime_error("data_source step_mode must be single_step, multi_step, or series.");
      }

      this->DataSources.AddDataSource(name, std::move(source));
    }
  }

  void ProcessCoordinateSystem(const rapidjson::Value& coordSys)
  {
    this->CoordinateSystem =
      std::make_shared<adis::datamodel::CoordinateSystem>();
    this->CoordinateSystem->ObjectName = "coordinate_system";

    this->CoordinateSystem->ProcessJSON(coordSys, this->DataSources);
  }

  void ProcessCellSet(const rapidjson::Value& cellSet)
  {
    this->CellSet =
      std::make_shared<adis::datamodel::CellSet>();
    this->CellSet->ObjectName = "cell_set";

    this->CellSet->ProcessJSON(cellSet, this->DataSources);
  }

  std::shared_ptr<adis::datamodel::Field>
    ProcessField(const rapidjson::Value& fieldJson)
  {
    if(!fieldJson.IsObject())
    {
      throw std::runtime_error("field needs to be an object.");
    }
    auto field = std::make_shared<adis::datamodel::Field>();
    field->ProcessJSON(fieldJson, this->DataSources);
    field->ObjectName = "field";
    return field;
  }

  void ProcessFields(const rapidjson::Value& fields)
  {
    this->Fields.clear();
    if(!fields.IsArray())
    {
      throw std::runtime_error("fields is not an array.");
    }
    auto fieldsArray = fields.GetArray();
    for(auto& field : fieldsArray)
    {
      auto fieldPtr = this->ProcessField(field);
      this->Fields[std::make_pair(fieldPtr->Name, fieldPtr->Association)] =
        fieldPtr;
    }
  }

  template <typename ValueType>
  const rapidjson::Value& FindAndReturnObject(ValueType& root,
    const std::string name)
  {
    if (!root.HasMember(name.c_str()))
    {
      throw std::runtime_error("Missing " + name + " member.");
    }
    auto& val = root[name.c_str()];
    if (!val.IsObject())
    {
      throw std::runtime_error(name + " is expected to be an object.");
    }
    return val;
  }

  void ReadJSONFile(const std::string dataModelFileName)
  {
    this->Cleanup();

    // Parse the JSON metadata file
    FILE *fp = std::fopen(dataModelFileName.c_str(), "rb");
    if(!fp)
    {
      throw std::ios_base::failure("Unable to open metadata file; does '" + dataModelFileName + "' exist?");
    }

    std::vector<char> buffer(65536);

    rapidjson::FileReadStream is(fp, buffer.data(), buffer.size());

    rapidjson::Document document;
    document.ParseStream(is);
    std::fclose(fp);

    if (document.HasParseError()) {
      throw std::logic_error("Unable to parse '" + dataModelFileName + "' as a json file. Error: " + rapidjson::GetParseError_En(document.GetParseError()));

    }
    if (!document.IsObject()) {
        throw std::logic_error("Unable to parse '" + dataModelFileName + "' as a json file; is it valid json?");
    }

    auto m = document.GetObject().begin();
    if(m == document.GetObject().end())
    {
      throw std::logic_error("There is no data in '" + dataModelFileName + "'; there is nothing that can be achieved with this file.");
    }
    if(!m->value.IsObject())
    {
      throw std::logic_error("Unable to create a sensible object from '" + dataModelFileName + "'; aborting.");
    }
    const auto obj = m->value.GetObject();
    if (!obj.HasMember("data_sources"))
    {
      throw std::runtime_error("Missing data_sources member.");
    }
    this->ProcessDataSources(obj["data_sources"].GetArray());

    if (!obj.HasMember("coordinate_system"))
    {
      throw std::runtime_error("Missing coordinate_system member.");
    }
    auto& cs = this->FindAndReturnObject(obj, "coordinate_system");
    this->ProcessCoordinateSystem(cs);

    if (!obj.HasMember("cell_set"))
    {
      throw std::runtime_error("Missing cell_set member.");
    }
    auto& cells = this->FindAndReturnObject(obj, "cell_set");
    this->ProcessCellSet(cells);

    if (obj.HasMember("fields"))
    {
      auto& fields = obj["fields"];
      this->ProcessFields(fields);
    }

  }

  std::vector<vtkm::cont::CoordinateSystem> ReadCoordinateSystem(
    const std::unordered_map<std::string, std::string>& paths,
    const adis::metadata::MetaData& selections)
  {
    if(!this->CoordinateSystem)
    {
      throw std::runtime_error("Cannot read missing coordinate system.");
    }
    return this->CoordinateSystem->Read(
      paths, this->DataSources, selections);
  }

  std::vector<vtkm::cont::DynamicCellSet> ReadCellSet(
    const std::unordered_map<std::string, std::string>& paths,
    const adis::metadata::MetaData& selections)
  {
    if(!this->CellSet)
    {
      throw std::runtime_error("Cannot read missing cell set.");
    }
    return this->CellSet->Read(
      paths, this->DataSources, selections);
  }

  adis::metadata::MetaData ReadMetaData(
    const std::unordered_map<std::string, std::string>& paths)
  {
    if(!this->CoordinateSystem)
    {
      throw std::runtime_error("Cannot read missing coordinate system.");
    }
    // FIXME What should be the correct number of blocks to use for XGC data?
    // It's different for various data in the different files.
    size_t nBlocks = this->CoordinateSystem->GetNumberOfBlocks(
      paths, this->DataSources);
    adis::metadata::MetaData metaData;
    adis::metadata::Size nBlocksM(nBlocks);
    metaData.Set(adis::keys::NUMBER_OF_BLOCKS(), nBlocksM);

    if (!this->Fields.empty())
    {
      adis::metadata::Vector<adis::metadata::FieldInformation> fields;
      for(auto& item : this->Fields)
      {
        auto& field = item.second;
        adis::metadata::FieldInformation afield(field->Name, field->Association);
        fields.Data.push_back(afield);
      }
      metaData.Set(adis::keys::FIELDS(), fields);
    }

    return metaData;
  }

  void PostRead(vtkm::cont::PartitionedDataSet& pds,
                const adis::metadata::MetaData& selections)
  {
    this->CellSet->PostRead(pds, selections);
  }

  void DoAllReads()
  {
    this->DataSources.DoAllReads();
  }

  void BeginStep(
    const std::unordered_map<std::string, std::string>& paths)
  {
    this->DataSources.BeginStep(paths);
  }

  void EndStep()
  {
    this->DataSources.EndStep();
  }

  DataSourceManager DataSources;
  std::shared_ptr<adis::datamodel::CoordinateSystem>
    CoordinateSystem = nullptr;
  std::shared_ptr<adis::datamodel::CellSet> CellSet = nullptr;
  using FieldsKeyType =
    std::pair<std::string, vtkm::cont::Field::Association>;
  std::map<FieldsKeyType, std::shared_ptr<adis::datamodel::Field> > Fields;
};

DataSetReader::DataSetReader(const std::string dataModelFilename,
  const Params& params)
: Impl(new DataSetReaderImpl(dataModelFilename, params))
{
}

DataSetReader::~DataSetReader()
{
}

adis::metadata::MetaData DataSetReader::ReadMetaData(
  const std::unordered_map<std::string, std::string>& paths)
{
  return this->Impl->ReadMetaData(paths);
}

vtkm::cont::PartitionedDataSet DataSetReader::ReadDataSet(
  const std::unordered_map<std::string, std::string>& paths,
  const adis::metadata::MetaData& selections)
{
  vtkm::cont::PartitionedDataSet ds = this->ReadDataSetInternal(
    paths, selections);
  this->Impl->DoAllReads();
  this->Impl->PostRead(ds, selections);

  // for(size_t i=0; i<ds.GetNumberOfPartitions(); i++)
  // {
  //   vtkm::io::writer::VTKDataSetWriter writer(
  //     "output" + std::to_string(i) + ".vtk");
  //   writer.WriteDataSet(ds.GetPartition(i));
  // }

  return ds;
}

void DataSetReader::PrepareNextStep(
  const std::unordered_map<std::string, std::string>& paths)
{
  this->Impl->BeginStep(paths);
}

vtkm::cont::PartitionedDataSet DataSetReader::ReadStep(
  const std::unordered_map<std::string, std::string>& paths,
  const adis::metadata::MetaData& selections)
{
  vtkm::cont::PartitionedDataSet ds = this->ReadDataSetInternal(
    paths, selections);
  this->Impl->EndStep();
  this->Impl->PostRead(ds, selections);

  return ds;
}

vtkm::cont::PartitionedDataSet DataSetReader::ReadDataSetInternal(
  const std::unordered_map<std::string, std::string>& paths,
  const adis::metadata::MetaData& selections)
{
  std::vector<vtkm::cont::CoordinateSystem> coordSystems =
    this->Impl->ReadCoordinateSystem(paths, selections);
  size_t nPartitions = coordSystems.size();
  std::vector<vtkm::cont::DataSet> dataSets(nPartitions);
  std::vector<vtkm::cont::DynamicCellSet> cellSets =
    this->Impl->ReadCellSet(paths, selections);
  for(size_t i=0; i<nPartitions; i++)
  {
    dataSets[i].AddCoordinateSystem(coordSystems[i]);
    dataSets[i].SetCellSet(cellSets[i]);
  }

  if (selections.Has(adis::keys::FIELDS()))
  {
    using FieldInfoType =
      adis::metadata::Vector<adis::metadata::FieldInformation>;
    auto& fields = selections.Get<FieldInfoType>(adis::keys::FIELDS());
    for(auto& field : fields.Data)
    {
      auto itr = this->Impl->Fields.find(
        std::make_pair(field.Name, field.Association));
      if (itr != this->Impl->Fields.end())
      {
        std::vector<vtkm::cont::Field> fieldVec =
          itr->second->Read(paths, this->Impl->DataSources, selections);
        for(size_t i=0; i<nPartitions; i++)
        {
          dataSets[i].AddField(fieldVec[i]);
        }
      }
    }
  }
  else
  {
    for(auto& field : this->Impl->Fields)
    {
      std::vector<vtkm::cont::Field> fields =
        field.second->Read(paths, this->Impl->DataSources, selections);
      for(size_t i=0; i<nPartitions; i++)
      {
        dataSets[i].AddField(fields[i]);
      }
    }
  }

  return vtkm::cont::PartitionedDataSet(dataSets);
}

void DataSetReader::SetDataSourceParameters(const std::string source,
  const DataSourceParams& params)
{
  this->Impl->SetDataSourceParameters(source, params);
}

void DataSetReader::SetDataSourceIO(const std::string source, void* io)
{
  this->Impl->SetDataSourceIO(source, io);
}

} // end namespace io
} // end namespace adis
