//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/DataModel.h>

#include <adis/DataSource.h>

namespace adis
{
namespace datamodel
{

std::vector<vtkm::cont::VariantArrayHandle> DataModelBase::ReadSelf(
  const std::unordered_map<std::string, std::string>& paths,
  adis::io::DataSourceManager& sources,
  const adis::metadata::MetaData& selections,
  adis::io::IsVector isItVector)
{
  if(this->IsStatic && !this->Cache.empty())
  {
    return this->Cache;
  }
  std::vector<vtkm::cont::VariantArrayHandle> var =
    sources.ReadVariable(this->DataSourceName, paths, this->VariableName, selections, isItVector);
  if (this->IsStatic)
  {
    this->Cache = var;
  }
  return var;
}

std::string DataModelBase::FindDataSource(
  const rapidjson::Value& dataModel,
  adis::io::DataSourceManager& sources) const
{
  if (!dataModel.HasMember("data_source"))
  {
    throw std::runtime_error(this->ObjectName + " must provide a data_source.");
  }
  std::string dsname = dataModel["data_source"].GetString();
  if (!sources.HasSource(dsname))
  {
    throw std::runtime_error("data_source." + dsname + " was not found.");
  }
  return dsname;
}

void DataModelBase::ProcessJSON(const rapidjson::Value& json,
  adis::io::DataSourceManager& sources)
{
  if (!json.HasMember("variable"))
  {
    throw std::runtime_error(
      this->ObjectName  + " must provide a variable.");
  }
  std::string varName = json["variable"].GetString();
  this->VariableName = varName;
  this->DataSourceName = this->FindDataSource(json, sources);

  if (json.HasMember("static") && json["static"].IsBool())
  {
    if(json["static"].GetBool())
    {
      this->IsStatic = true;
    }
  }
}

}
}
