//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include <adis/Array.h>
//#include <adis/xgc/PointsXGC.h>
//#include <adis/xgc/PointsXGC.hxx>
//#include <adis/xgc/ArrayHandleXGCPointCoordinates.h>
//#include <adis/xgc/ArrayHandleXGCFieldPlane.h>

#include <vtkm/cont/StorageExtrude.h>
#include <vtkm/cont/ArrayHandleExtrudeCoords.h>
#include <vtkm/cont/ArrayHandleExtrudeField.h>
#include <vtkm/cont/ArrayHandleUniformPointCoordinates.h>

namespace adis
{
namespace datamodel
{

std::vector<vtkm::cont::VariantArrayHandle> Array::Read(
  const std::unordered_map<std::string, std::string>& paths,
  adis::io::DataSourceManager& sources,
  const adis::metadata::MetaData& selections)
{
  return this->ArrayImpl->Read(paths, sources, selections);
}

size_t Array::GetNumberOfBlocks(
  const std::unordered_map<std::string, std::string>& paths,
  adis::io::DataSourceManager& sources)
{
  return this->ArrayImpl->GetNumberOfBlocks(paths, sources);
}

void Array::ProcessJSON(const rapidjson::Value& json,
                        adis::io::DataSourceManager& sources)
{
  if (!json.HasMember("array_type") || !json["array_type"].IsString())
  {
    throw std::runtime_error(
      this->ObjectName  + " must provide a valid array_type.");
  }
  const std::string& arrayType = json["array_type"].GetString();
  if (arrayType == "basic")
  {
    this->ArrayImpl.reset(new ArrayBasic());
  }
  else if (arrayType == "uniform_point_coordinates")
  {
    this->ArrayImpl.reset(new ArrayUniformPointCoordinates());
  }
  else if (arrayType == "cartesian_product")
  {
    this->ArrayImpl.reset(new ArrayCartesianProduct());
  }
  else if (arrayType == "xgc2d_point_coordinates")
  {
    this->ArrayImpl.reset(new ArrayXGC2DPointCoordinates());
  }
  else if (arrayType == "xgc_field_plane")
  {
    this->ArrayImpl.reset(new ArrayXGCFieldPlane());
  }
  else
  {
    throw std::runtime_error(arrayType + " is not a valid array type.");
  }
  this->ArrayImpl->ProcessJSON(json, sources);
}

void ArrayBasic::ProcessJSON(const rapidjson::Value& json,
                             adis::io::DataSourceManager& sources)
{
  this->ArrayBase::ProcessJSON(json, sources);

  if (json.HasMember("is_vector"))
  {
    const std::string& isVector = json["is_vector"].GetString();
    if (isVector == "true")
    {
      this->IsVector = adis::io::IsVector::Yes;
    }
    else if (isVector == "false")
    {
      this->IsVector = adis::io::IsVector::No;
    }
    else if (isVector == "auto")
    {
      this->IsVector = adis::io::IsVector::Auto;
    }
    else
    {
      throw std::runtime_error(
        "Unrecognized value for is_vector: " + isVector);
    }
  }
}

std::vector<vtkm::cont::VariantArrayHandle> ArrayBasic::Read(
  const std::unordered_map<std::string, std::string>& paths,
  adis::io::DataSourceManager& sources,
  const adis::metadata::MetaData& selections)
{
  return this->ReadSelf(paths, sources, selections, this->IsVector);
}

size_t ArrayBasic::GetNumberOfBlocks(
  const std::unordered_map<std::string, std::string>& paths,
  adis::io::DataSourceManager& sources)
{
  return sources.GetNumberOfBlocks(this->DataSourceName, paths, this->VariableName);
}

void ArrayUniformPointCoordinates::ProcessJSON(const rapidjson::Value& json,
  adis::io::DataSourceManager& sources)
{
  if (!json.HasMember("dimensions") || !json["dimensions"].IsObject())
  {
    throw std::runtime_error(
      this->ObjectName  + " must provide a dimensions object.");
  }
  this->Dimensions.reset(new Value());
  const auto& dimensions = json["dimensions"];
  this->Dimensions->ProcessJSON(dimensions, sources);

  if (json.HasMember("origin") && json["origin"].IsObject())
  {
    this->Origin.reset(new Value());
    const auto& origin = json["origin"];
    this->Origin->ProcessJSON(origin, sources);
  }

  if (json.HasMember("spacing") && json["spacing"].IsObject())
  {
    this->Spacing.reset(new Value());
    const auto& spacing = json["spacing"];
    this->Spacing->ProcessJSON(spacing, sources);
  }
}

std::vector<vtkm::cont::VariantArrayHandle>
  ArrayUniformPointCoordinates::Read(
    const std::unordered_map<std::string, std::string>& paths,
    adis::io::DataSourceManager& sources,
    const adis::metadata::MetaData& selections)
{
  std::vector<vtkm::cont::VariantArrayHandle> dims =
    this->Dimensions->Read(paths, sources, selections);
  std::vector<vtkm::cont::VariantArrayHandle> origins;
  if (this->Origin)
  {
    origins = this->Origin->Read(paths, sources, selections);
  }
  std::vector<vtkm::cont::VariantArrayHandle> spacings;
  if (this->Spacing)
  {
    spacings = this->Spacing->Read(paths, sources, selections);
  }
  std::vector<vtkm::cont::VariantArrayHandle> ret;
  size_t nDims = dims.size();
  ret.reserve(nDims);
  for(const auto& array : dims)
  {
    // const auto& array = dims[i];
    auto dimsB = array.Cast<vtkm::cont::ArrayHandle<size_t> >();
    auto dimsPortal = dimsB.GetPortalConstControl();
    vtkm::Id3 dimValues(dimsPortal.Get(0),
                        dimsPortal.Get(1),
                        dimsPortal.Get(2));
    vtkm::Vec<vtkm::FloatDefault, 3> originA;
    originA = vtkm::Vec<vtkm::FloatDefault, 3>(0.0, 0.0, 0.0);
    vtkm::Vec<vtkm::FloatDefault, 3> spacingA;
    spacingA = vtkm::Vec<vtkm::FloatDefault, 3>(1.0, 1.0, 1.0);
    if (this->Origin)
    {
      const auto& origin = origins[0];
      auto originB = origin.Cast<vtkm::cont::ArrayHandle<double> >();
      auto originPortal = originB.GetPortalConstControl();
      originA = vtkm::Vec<vtkm::FloatDefault, 3>(originPortal.Get(0),
                                                 originPortal.Get(1),
                                                 originPortal.Get(2));
    }
    if (this->Spacing)
    {
      const auto& spacing = spacings[0];
      auto spacingB = spacing.Cast<vtkm::cont::ArrayHandle<double> >();
      auto spacingPortal = spacingB.GetPortalConstControl();
      spacingA = vtkm::Vec<vtkm::FloatDefault, 3>(spacingPortal.Get(0),
                                                  spacingPortal.Get(1),
                                                  spacingPortal.Get(2));
    }
    // Shift origin to a local value. We have to do this because
    // VTK-m works with dimensions rather than extents and therefore
    // needs local origin.
    for(size_t i=0; i<3; i++)
    {
      originA[i] = originA[i] + spacingA[i] * dimsPortal.Get(i+3);
    }
    vtkm::cont::ArrayHandleUniformPointCoordinates ah(
      dimValues, originA, spacingA);
    ret.push_back(ah);
  }
  return ret;
}

size_t ArrayUniformPointCoordinates::GetNumberOfBlocks(
  const std::unordered_map<std::string, std::string>& paths,
  adis::io::DataSourceManager& sources)
{
  return this->Dimensions->GetNumberOfBlocks(paths, sources);
}

void ArrayCartesianProduct::ProcessJSON(const rapidjson::Value& json,
  adis::io::DataSourceManager& sources)
{
  if (!json.HasMember("x_array") || !json["x_array"].IsObject())
  {
    throw std::runtime_error(
      this->ObjectName  + " must provide a x_array object.");
  }
  this->XArray.reset(new Array());
  const auto& xarray = json["x_array"];
  this->XArray->ProcessJSON(xarray, sources);

  if (!json.HasMember("y_array") || !json["y_array"].IsObject())
  {
    throw std::runtime_error(
      this->ObjectName  + " must provide a y_array object.");
  }
  this->YArray.reset(new Array());
  const auto& yarray = json["y_array"];
  this->YArray->ProcessJSON(yarray, sources);

  if (!json.HasMember("z_array") || !json["z_array"].IsObject())
  {
    throw std::runtime_error(
      this->ObjectName  + " must provide a z_array object.");
  }
  this->ZArray.reset(new Array());
  const auto& zarray = json["z_array"];
  this->ZArray->ProcessJSON(zarray, sources);

}

std::vector<vtkm::cont::VariantArrayHandle>
  ArrayCartesianProduct::Read(
    const std::unordered_map<std::string, std::string>& paths,
    adis::io::DataSourceManager& sources,
    const adis::metadata::MetaData& selections)
{
  std::vector<vtkm::cont::VariantArrayHandle> retVal;
  std::vector<vtkm::cont::VariantArrayHandle> xarrays =
    this->XArray->Read(paths, sources, selections);
  std::vector<vtkm::cont::VariantArrayHandle> yarrays =
    this->YArray->Read(paths, sources, selections);
  std::vector<vtkm::cont::VariantArrayHandle> zarrays =
    this->ZArray->Read(paths, sources, selections);
  size_t nArrays = xarrays.size();
  for(size_t i=0; i<nArrays; i++)
  {
    auto& xarray = xarrays[i];
    auto& yarray = yarrays[i];
    auto& zarray = zarrays[i];
    using floatType = vtkm::cont::ArrayHandle<float>;
    using doubleType = vtkm::cont::ArrayHandle<double>;
    if (xarray.IsType<floatType>() &&
        yarray.IsType<floatType>() &&
        zarray.IsType<floatType>())
    {
      auto xarrayF = xarray.Cast<floatType>();
      auto yarrayF = yarray.Cast<floatType>();
      auto zarrayF = zarray.Cast<floatType>();
      retVal.push_back(
        vtkm::cont::make_ArrayHandleCartesianProduct(
          xarrayF, yarrayF, zarrayF));
    }
    else if (xarray.IsType<doubleType>() &&
             yarray.IsType<doubleType>() &&
             zarray.IsType<doubleType>())
    {
      auto xarrayD = xarray.Cast<doubleType>();
      auto yarrayD = yarray.Cast<doubleType>();
      auto zarrayD = zarray.Cast<doubleType>();
      retVal.push_back(
        vtkm::cont::make_ArrayHandleCartesianProduct(
          xarrayD, yarrayD, zarrayD));
    }
    else
    {
      throw std::runtime_error(
        "Only float and double arrays are supported in cartesian products.");
    }
  }
  return retVal;
}

size_t ArrayCartesianProduct::GetNumberOfBlocks(
  const std::unordered_map<std::string, std::string>& paths,
  adis::io::DataSourceManager& sources)
{
  return this->XArray->GetNumberOfBlocks(paths, sources);
}

size_t ArrayXGC::GetNumberOfBlocks(
    const std::unordered_map<std::string, std::string>& paths,
    adis::io::DataSourceManager &sources)
{
  return sources.GetNumberOfBlocks(this->DataSourceName, paths, this->VariableName);
}

void ArrayXGC::SetNumberOfPlanes(
    const std::unordered_map<std::string, std::string>& paths,
    adis::io::DataSourceManager& sources)
{
  if (this->NumberOfPlanes >= 0)
  {
    return;
  }

  // TODO is this always a good way to get the number of planes?
  // need to figure out a different way of getting the number of planes so it doesn't
  // have to be specified multiple times in the json file
  // can get it from ADIOS, since it will be one of the dimensions (shape) of the data
  // but is this info the total dimension or just what's one this rank?
  // and is the number of planes dimension always the 2nd element?
  std::cout << "varname " << this->VariableName << std::endl;
  auto shape = sources.GetShape(this->DataSourceName, paths, this->VariableName);
  // TODO are the dimensions always in the same order?
  // It looks like for files written with adios 1.x, the number of planes is shape[1],
  // while for 2.x it's shape[0]
  this->NumberOfPlanes = 4; // TODO fix shape[0];
  std::cout << "NumberOfPlanes " << this->NumberOfPlanes << std::endl;
}

void ArrayXGC2DPointCoordinates::ProcessJSON(const rapidjson::Value& json,
  adis::io::DataSourceManager& sources)
{
  this->ArrayBase::ProcessJSON(json, sources);
  //process the number of planes
  if (!json.HasMember("number_of_planes") || !json["number_of_planes"].IsObject())
  {
    throw std::runtime_error(
      this->ObjectName + " must provide a number_of_planes object.");
  }
  this->NumPlanes.reset(new Array());
  this->NumPlanes->ProcessJSON(json["number_of_planes"], sources);

  if (!json.HasMember("is_cylindrical") || !json["is_cylindrical"].IsBool())
  {
    throw std::runtime_error(
      this->ObjectName + " must provide a coordinates_type.");
  }
  this->IsCylindrical = json["is_cylindrical"].GetBool();
}

std::vector<vtkm::cont::VariantArrayHandle> ArrayXGC2DPointCoordinates::Read(
    const std::unordered_map<std::string, std::string>& paths,
    adis::io::DataSourceManager& sources,
    const adis::metadata::MetaData& selections)
{
  auto coordArrays = this->ReadSelf(paths, sources, selections, adis::io::IsVector::No);
  if (coordArrays.size() != 1)
  {
    throw std::runtime_error("ArrayXGC2DPointCoordinates supports only one coordinates array");
  }
  //TODO add in some type checking
  auto coordsAH = coordArrays[0].Cast<vtkm::cont::ArrayHandle<double>>();

  auto nPlanesAH = this->NumPlanes->Read(paths, sources, selections)[0];
  if (nPlanesAH.GetNumberOfValues() != 1)
  {
    throw std::runtime_error("The number of planes should be a single value");
  }
  using intType = vtkm::cont::ArrayHandle<vtkm::Int32>;
  vtkm::Int32 numPlanes;
  if (nPlanesAH.IsType<intType>())
  {
    auto nPlanesI = nPlanesAH.Cast<intType>();
    numPlanes = nPlanesI.GetPortalConstControl().Get(0);
  }

  std::vector<vtkm::cont::VariantArrayHandle> retVal;
  retVal.push_back(
    vtkm::cont::make_ArrayHandleExtrudeCoords(
      coordsAH, numPlanes, this->IsCylindrical));

  return retVal;
}

std::vector<vtkm::cont::VariantArrayHandle> ArrayXGCFieldPlane::Read(
    const std::unordered_map<std::string, std::string>& paths,
    adis::io::DataSourceManager& sources,
    const adis::metadata::MetaData& selections)
{
  std::cout << "xgc field plane read:\n";
  this->SetNumberOfPlanes(paths, sources);

  // TODO so here I think I need to reset the block selections to be the appropriate plane selections
  // how to handle sequential read vs parallel run? for now just handle serial case.
  // this works when there is one block per plane, but not when there are multiple blocks per plane
  // So start by checking what blocks have been selected by the user
  //size_t totalBlocks = this->GetNumberOfBlocks(paths, sources);
  //size_t blocksPerPlane = totalBlocks / this->NumberOfPlanes;
  //size_t numSelectedBlocks = blocksPerPlane;
  metadata::MetaData newSelections = selections;
  //metadata::Vector<size_t> planeSelections;
  if (selections.Has(keys::BLOCK_SELECTION()))
  {
    newSelections.Remove(adis::keys::BLOCK_SELECTION());
    // This assumes that blocks are evenly divided among planes so that the blocks can be
    // given local ids for that plane. So then when a user chooses a block, e.g. block 0,
    // block 0 will be selected on every plane.
    // TODO also move any other selections made to newSelections?
    //auto& blocks = selections.Get<metadata::Vector<size_t> >(keys::BLOCK_SELECTION());
    //std::cout << "number of selected blocks " << blocks.Data.size() << std::endl;
    //numSelectedBlocks = blocks.Data.size();
    //for (int p = 0; p < this->NumberOfPlanes; ++p)
    //{
    //  for (auto block : blocks.Data)
    //  {
    //    if (block >= blocksPerPlane)
    //    {
    //      //error
    //    }
    //    planeSelections.Data.push_back(blocksPerPlane * p + block);
    //  }
    //}
  }
  //// else should we choose all blocks for each plane?
  //newSelections.Set(keys::BLOCK_SELECTION(), planeSelections);

  //if (fieldPlane.size() == 1)
  {
    // This actually has to be done as a post read
    // but even then, i think everything is expecting variant array handles.
    // so still need to do a post read, but will have to copy it into the rest of the array
    // at this point, just get enough storage space for the full array, and then adios will read in
    // the data to the first 1/numPlanes part of the storage.


    //using IdArrayType = vtkm::cont::ArrayHandle<vtkm::Id>;
    //using IdPortalType = IdArrayType::PortalControl;
    //// Then we have a field that is replicated across planes
    //// need an id array and then call make_ArrayHandlePermutation()
    //IdArrayType idArray;
    //vtkm::Id valuesPerPlane = fieldPlane[0].GetNumberOfValues();
    //vtkm::Id totalValues = this->NumberOfPlanes * valuesPerPlane;
    //idArray.Allocate(totalValues);
    //IdPortalType idPortal = idArray.GetPortalControl();
    //// TODO save idArray so we don't have to always compute this
    //for (vtkm::Id ii = 0; ii < totalValues; ++ii)
    //{
    //  idPortal.Set(ii, ii % valuesPerPlane);
    //}
    //auto fullArray = vtkm::cont::make_ArrayHandlePermutation(idArray, fieldAH);
    //vtkm::cont::ArrayHandleVirtual<double> fullArrayVirt(fullArray);
    //std::vector<vtkm::cont::VariantArrayHandle> retVal;
    //retVal.push_back(vtkm::cont::make_ArrayHandleExtrudeField(
    //      fullArrayVirt, this->NumberOfPlanes, false));
    //return retVal;
  }
  // should have one element per plane - not true in some cases, may  have multiple blocks per plane
  //if (this->NumberOfPlanes * numSelectedBlocks != fieldPlane.size())
  //{
  //  throw std::runtime_error("ArrayXGCFieldPlane: Number of blocks do not match number of planes.");
  //}
  //TODO better type handling
  //std::vector<vtkm::cont::ArrayHandle<double> > fieldPlanesD;
  //for (auto& plane : fieldPlane)
  //{
  //  fieldPlanesD.push_back(plane.Cast<vtkm::cont::ArrayHandle<double> >());
  //}

  auto fieldPlane = this->ReadSelf(paths, sources, selections, adis::io::IsVector::No);
  std::vector<vtkm::cont::VariantArrayHandle> retVal;
  for (int p = 0; p < fieldPlane.size(); ++p)
  {
    auto fieldAH = fieldPlane[p].Cast<vtkm::cont::ArrayHandle<double> >();
    retVal.push_back(vtkm::cont::make_ArrayHandleExtrudeField(
      fieldAH, this->NumberOfPlanes));
  }

  return retVal;
}

}
}
