/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkThreadSafeLog.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkThreadSafeLog.h"

#include "vtkObjectFactory.h"
#include "vtkTimerLog.h"

vtkCxxRevisionMacro(vtkThreadSafeLog, "1.6");
vtkStandardNewMacro(vtkThreadSafeLog);

//----------------------------------------------------------------------------
vtkThreadSafeLog::vtkThreadSafeLog()
{
  this->Timer = vtkTimerLog::New();
  this->NumberOfEntries = 0;
}

//----------------------------------------------------------------------------
vtkThreadSafeLog::~vtkThreadSafeLog()
{
  int idx;

  this->Timer->Delete();
  this->Timer = NULL;

  for (idx = 0; idx < this->NumberOfEntries; ++idx)
    {
    if (this->Tags[idx])
      {
      delete [] this->Tags[idx];
      }
    }
  this->NumberOfEntries = 0;
}

//----------------------------------------------------------------------------
void vtkThreadSafeLog::AddEntry(char *tag, float value)
{
  int idx = this->NumberOfEntries;

  if (idx >= VTK_THREAD_SAFE_LOG_MAX)
    {
    vtkErrorMacro("Too many entries");
    }

  if (tag == NULL)
    {
    this->Tags[idx] = NULL;
    }
  else
    {
    this->Tags[idx] = new char[strlen(tag)+1];
    strcpy(this->Tags[idx], tag);
    }
  this->Values[idx] = value;
  this->NumberOfEntries = idx+1;
}

//----------------------------------------------------------------------------
void vtkThreadSafeLog::DumpLog(char *filename)
{
  this->DumpLog(filename, ios::out);
}

//----------------------------------------------------------------------------
void vtkThreadSafeLog::DumpLog(char *filename, int nMode)
{
#if defined(__GNUC__) && (__GNUC__ >= 3)
  ofstream os(filename, static_cast<ios::openmode>(nMode));
#else
  ofstream os(filename, nMode);
#endif
  int idx;
  
  if (nMode == ios::out)
    {
    for (idx = 0; idx < this->NumberOfEntries; ++idx)
      {
      os << setw(10) << this->Tags[idx] << " ";
      }
    os << endl;
    }

  for (idx = 0; idx < this->NumberOfEntries; ++idx)
    {
    os << setw(10) << this->Values[idx] << " ";
    }
  os << endl;
}

//----------------------------------------------------------------------------
void vtkThreadSafeLog::StartTimer()
{
  this->Timer->StartTimer();
}

//----------------------------------------------------------------------------
void vtkThreadSafeLog::StopTimer()
{
  this->Timer->StopTimer();
}

//----------------------------------------------------------------------------
double vtkThreadSafeLog::GetElapsedTime()
{
  return this->Timer->GetElapsedTime();
}

//----------------------------------------------------------------------------
void vtkThreadSafeLog::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
}
  
