// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate git_workarea;
use self::git_workarea::CommitId;

extern crate gitlab;

extern crate itertools;
use self::itertools::Itertools;

extern crate regex;
use self::regex::Regex;

use super::super::actions::check;
use super::traits::*;

use std::cmp::Ord;

lazy_static! {
    static ref MR_UPDATE_RE: Regex =
        Regex::new("^Added [0-9][0-9]* (new )?commits?:\n\
                    \n\
                    (\\* [0-9a-f]+ - [^\n]*\n)*\
                    (\n\\[Compare with previous versions\\]\\(.*\\))?\
                    $").unwrap();
}

fn gitlab_status_info(status: PendingCommitStatus) -> gitlab::CommitStatusInfo {
    gitlab::CommitStatusInfo {
        refname: status.refname,
        name: Some(&status.name),
        target_url: None,
        description: Some(status.description),
    }
}

impl From<gitlab::UserFull> for User {
    fn from(user: gitlab::UserFull) -> Self {
        User {
            id: user.id.value(),
            handle: user.username,
            name: user.name,
            email: user.email,
        }
    }

}

/// Structure used to communicate with a GitLab instance.
pub struct GitlabService {
    gitlab: gitlab::Gitlab,
    user: User,
}

impl GitlabService {
    /// Create a new GitLab communication channel.
    pub fn new(gitlab: gitlab::Gitlab) -> Result<Self> {
        Ok(GitlabService {
            user: try!(gitlab.current_user()
                .map(Into::into)
                .chain_err(|| ErrorKind::Host)),
            gitlab: gitlab,
        })
    }

    fn repo_from_project(&self, project: gitlab::Project) -> Result<Repo> {
        let parent_project = if let Some(ref upstream) = project.forked_from_project {
            let parent_project = try!(self.gitlab.project(upstream.id)
                .chain_err(|| ErrorKind::Host));
            Some(Box::new(try!(self.repo_from_project(parent_project))))
        } else {
            None
        };

        Ok(Repo {
            name: project.path_with_namespace,
            url: project.ssh_url_to_repo,
            id: project.id.value(),
            forked_from: parent_project,
        })
    }

    fn commit_from_project(&self, project: gitlab::Project, commit: &CommitId)
                           -> Result<Commit> {
        let commit = try!(self.gitlab
            .commit(project.id, &commit.0)
            .chain_err(|| ErrorKind::Host));

        Ok(Commit {
            repo: try!(self.repo_from_project(project)),
            refname: None,
            id: CommitId::new(commit.id.value()),
        })
    }

    fn issue_from_project(&self, project: gitlab::Project, id: u64) -> Result<Issue> {
        let issue = try!(self.gitlab
            .issue(project.id, gitlab::IssueId::new(id))
            .chain_err(|| ErrorKind::Host));

        Ok(Issue {
            repo: try!(self.repo_from_project(project)),
            id: issue.id.value(),
            url: issue.web_url,
            description: issue.description,
            labels: issue.labels,
            milestone: issue.milestone.map(|milestone| milestone.title),
            assignee: issue.assignee.map(|assignee| assignee.username),
        })
    }

    fn merge_request_from_project(&self, project: gitlab::Project, id: u64)
                                  -> Result<MergeRequest> {
        let mr = try!(self.gitlab
            .merge_request(project.id, gitlab::MergeRequestId::new(id))
            .chain_err(|| ErrorKind::Host));
        let source_project = try!(self.gitlab
            .project(mr.source_project_id)
            .chain_err(|| ErrorKind::Host));
        let author = try!(self.gitlab
            .user::<gitlab::UserFull>(mr.author.id)
            .map(Into::into)
            .chain_err(|| ErrorKind::Host));
        let status = try!(self.gitlab.commit_latest_statuses(source_project.id, mr.sha.value())
                .chain_err(|| ErrorKind::Host))
            .iter()
            // Only look at statuses posted by the current user.
            .filter(|status| status.author.id.value() == self.user.id)
            // Only look at check result statuses.
            .find(|status| status.name == check::STATUS_NAME)
            .map(|status| {
                match status.status {
                    gitlab::StatusState::Success => CheckStatus::Pass,
                    gitlab::StatusState::Failed => CheckStatus::Fail,
                    gitlab::StatusState::Pending |
                    gitlab::StatusState::Running |
                    gitlab::StatusState::Canceled => CheckStatus::Unchecked,
                }
            })
            .unwrap_or(CheckStatus::Unchecked);

        let source_repo = try!(self.repo_from_project(source_project));
        let target_repo = try!(self.repo_from_project(project));

        Ok(MergeRequest {
            source_repo: source_repo.clone(),
            source_branch: mr.source_branch.clone(),
            target_repo: target_repo,
            target_branch: mr.target_branch,
            id: mr.id.value(),
            url: mr.web_url,
            work_in_progress: mr.work_in_progress,
            description: mr.description.unwrap_or_else(String::new),
            old_commit: None,
            commit: Commit {
                repo: source_repo,
                refname: Some(mr.source_branch),
                id: CommitId::new(mr.sha.value()),
            },
            author: author,
            reference: format!("!{}", mr.iid),
            check_status: status,
            remove_source_branch: mr.force_remove_source_branch.unwrap_or(false),
        })
    }

    fn sort_notes(&self, notes: Vec<gitlab::Note>) -> Result<Vec<Comment>> {
        Ok(try!(notes.into_iter()
            .map(|note| {
                Ok(Comment {
                    id: note.id.value(),
                    is_system: note.system,
                    is_branch_update: note.system && MR_UPDATE_RE.is_match(&note.body),
                    created_at: note.created_at,
                    updated_at: note.updated_at,
                    author: try!(self.gitlab.user::<gitlab::UserFull>(note.author.id)
                        .map(Into::into)
                        .chain_err(|| ErrorKind::Host)),
                    content: note.body,
                })
            })
            .collect::<Result<Vec<_>>>())
            .into_iter()
            .sorted_by(|ref a, ref b| a.id.cmp(&b.id)))
    }

    fn convert_awards(&self, awards: Vec<gitlab::AwardEmoji>) -> Result<Vec<Award>> {
        awards.into_iter()
            .map(|award| {
                Ok(try!(self.gitlab
                    .user::<gitlab::UserFull>(award.user.id)
                    .map(|author| {
                        Award {
                            name: award.name.clone(),
                            author: author.into(),
                        }
                    })
                    .chain_err(|| ErrorKind::Host)))
            })
            .collect::<Result<Vec<_>>>()
    }
}

impl HostingService for GitlabService {
    fn service_user(&self) -> &User {
        &self.user
    }

    fn add_member(&self, project: &str, user: &User, level: u64) -> Result<()> {
        let project = try!(self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host));

        try!(self.gitlab
            .add_user_to_project(project.id, gitlab::UserId::new(user.id), level.into())
            .chain_err(|| ErrorKind::Host));

        Ok(())
    }

    fn members(&self, project: &str) -> Result<Vec<Membership>> {
        let project = try!(self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host));

        let project_members = try!(self.gitlab
            .project_members(project.id)
            .chain_err(|| ErrorKind::Host));
        let namespace_id = project.namespace.owner_id();
        let group_members = if let gitlab::NamespaceId::Group(gid) = namespace_id {
            try!(self.gitlab
                .group_members(gid)
                .chain_err(|| ErrorKind::Host))
        } else {
            vec![]
        };

        project_members.into_iter()
            // Chain with the group membership list.
            .chain(group_members.into_iter())
            // Sort by the user IDs.
            .sorted_by(|a, b| Ord::cmp(&a.id.value(), &b.id.value()))
            .into_iter()
            // Group all members in the list by the ID.
            .group_by(|member| member.id)
            .into_iter()
            // Create GitlabMembership structs.
            .map(|(_, members)| {
                // Find the maximum access level from any membership.
                let max_access = members.max_by_key(|member| member.access_level)
                    .unwrap();

                Ok(Membership {
                    user: try!(self.gitlab.user::<gitlab::UserFull>(max_access.id)
                        .map(Into::into)
                        .chain_err(|| ErrorKind::Host)),
                    access_level: max_access.access_level,
                    expiration: max_access.expires_at,
                })
            })
            .collect()
    }

    fn add_hook(&self, project: &str, url: &str) -> Result<()> {
        let project = try!(self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host));
        let events = gitlab::WebhookEvents::new()
            .with_issues()
            .with_merge_requests()
            .with_note()
            .with_push();

        let _ = try!(self.gitlab
            .add_hook(project.id, url, events)
            .chain_err(|| ErrorKind::Host));

        Ok(())
    }

    fn user(&self, user: &str) -> Result<User> {
        Ok(try!(self.gitlab
            .user_by_name::<gitlab::UserFull>(user)
            .map(Into::into)
            .chain_err(|| ErrorKind::Host)))
    }

    fn commit(&self, project: &str, commit: &CommitId) -> Result<Commit> {
        let project = try!(self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host));
        self.commit_from_project(project, commit)
    }

    fn issue(&self, project: &str, id: u64) -> Result<Issue> {
        let project = try!(self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host));
        self.issue_from_project(project, id)
    }

    fn merge_request(&self, project: &str, id: u64) -> Result<MergeRequest> {
        let project = try!(self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host));
        self.merge_request_from_project(project, id)
    }

    fn repo(&self, project: &str) -> Result<Repo> {
        let project = try!(self.gitlab
            .project_by_name(project)
            .chain_err(|| ErrorKind::Host));
        self.repo_from_project(project)
    }

    fn user_by_id(&self, user: u64) -> Result<User> {
        Ok(try!(self.gitlab
            .user::<gitlab::UserFull>(gitlab::UserId::new(user))
            .map(Into::into)
            .chain_err(|| ErrorKind::Host)))
    }

    fn commit_by_id(&self, project: u64, commit: &CommitId) -> Result<Commit> {
        let project = try!(self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host));
        self.commit_from_project(project, commit)
    }

    fn issue_by_id(&self, project: u64, id: u64) -> Result<Issue> {
        let project = try!(self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host));
        self.issue_from_project(project, id)
    }

    fn merge_request_by_id(&self, project: u64, id: u64) -> Result<MergeRequest> {
        let project = try!(self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host));
        self.merge_request_from_project(project, id)
    }

    fn repo_by_id(&self, project: u64) -> Result<Repo> {
        let project = try!(self.gitlab
            .project(gitlab::ProjectId::new(project))
            .chain_err(|| ErrorKind::Host));
        self.repo_from_project(project)
    }

    fn get_issue_comments(&self, issue: &Issue) -> Result<Vec<Comment>> {
        let notes = try!(self.gitlab
            .issue_notes(gitlab::ProjectId::new(issue.repo.id),
                         gitlab::IssueId::new(issue.id))
            .chain_err(|| ErrorKind::Host));

        self.sort_notes(notes)
    }

    fn post_issue_comment(&self, issue: &Issue, content: &str) -> Result<()> {
        try!(self.gitlab
            .create_issue_note(gitlab::ProjectId::new(issue.repo.id),
                               gitlab::IssueId::new(issue.id),
                               content)
            .chain_err(|| ErrorKind::Host));

        Ok(())
    }

    fn get_mr_comments(&self, mr: &MergeRequest) -> Result<Vec<Comment>> {
        let notes = try!(self.gitlab
            .merge_request_notes(gitlab::ProjectId::new(mr.target_repo.id),
                                 gitlab::MergeRequestId::new(mr.id))
            .chain_err(|| ErrorKind::Host));

        self.sort_notes(notes)
    }

    fn post_mr_comment(&self, mr: &MergeRequest, content: &str) -> Result<()> {
        try!(self.gitlab
            .create_merge_request_note(gitlab::ProjectId::new(mr.target_repo.id),
                                       gitlab::MergeRequestId::new(mr.id),
                                       content)
            .chain_err(|| ErrorKind::Host));

        Ok(())
    }

    fn post_commit_comment(&self, commit: &Commit, content: &str) -> Result<()> {
        try!(self.gitlab
            .create_commit_comment(gitlab::ProjectId::new(commit.repo.id),
                                   commit.id.as_str(),
                                   content)
            .chain_err(|| ErrorKind::Host));

        Ok(())
    }

    fn get_commit_statuses(&self, commit: &Commit) -> Result<Vec<CommitStatus>> {
        try!(self.gitlab
            .commit_latest_statuses(gitlab::ProjectId::new(commit.repo.id),
                                    commit.id.as_str())
            .chain_err(|| ErrorKind::Host))
            .into_iter()
            .map(|status| {
                Ok(try!(self.gitlab
                    .user::<gitlab::UserFull>(status.author.id)
                    .map(|author| {
                        CommitStatus {
                            state: status.status.into(),
                            author: author.into(),
                            refname: status.ref_.clone(),
                            name: status.name.clone(),
                            description: status.description.clone().unwrap_or_else(String::new),
                        }
                    })
                    .chain_err(|| ErrorKind::Host)))
            })
            .collect::<Result<Vec<_>>>()
    }

    fn post_commit_status(&self, status: PendingCommitStatus) -> Result<()> {
        try!(self.gitlab
            .create_commit_status(gitlab::ProjectId::new(status.commit.repo.id),
                                  status.commit.id.as_str(),
                                  status.state.clone().into(),
                                  &gitlab_status_info(status))
            .chain_err(|| ErrorKind::Host));

        Ok(())
    }

    fn get_mr_awards(&self, mr: &MergeRequest) -> Result<Vec<Award>> {
        self.gitlab
            .merge_request_awards(gitlab::ProjectId::new(mr.target_repo.id),
                                  gitlab::MergeRequestId::new(mr.id))
            .chain_err(|| ErrorKind::Host)
            .and_then(|awards| self.convert_awards(awards))
    }

    fn get_mr_comment_awards(&self, mr: &MergeRequest, comment: &Comment)
                             -> Result<Vec<Award>> {
        self.gitlab
            .merge_request_note_awards(gitlab::ProjectId::new(mr.target_repo.id),
                                       gitlab::MergeRequestId::new(mr.id),
                                       gitlab::NoteId::new(comment.id))
            .chain_err(|| ErrorKind::Host)
            .and_then(|awards| self.convert_awards(awards))
    }

    fn award_mr_comment(&self, mr: &MergeRequest, comment: &Comment, award: &str)
                        -> Result<()> {
        try!(self.gitlab
            .award_merge_request_note(gitlab::ProjectId::new(mr.target_repo.id),
                                      gitlab::MergeRequestId::new(mr.id),
                                      gitlab::NoteId::new(comment.id),
                                      award)
            .chain_err(|| ErrorKind::Host));

        Ok(())
    }
}

impl From<CommitStatusState> for gitlab::StatusState {
    fn from(state: CommitStatusState) -> Self {
        match state {
            CommitStatusState::Pending => gitlab::StatusState::Pending,
            CommitStatusState::Running => gitlab::StatusState::Running,
            CommitStatusState::Success => gitlab::StatusState::Success,
            CommitStatusState::Failed => gitlab::StatusState::Failed,
        }
    }
}

impl From<gitlab::StatusState> for CommitStatusState {
    fn from(state: gitlab::StatusState) -> Self {
        match state {
            gitlab::StatusState::Canceled |
            gitlab::StatusState::Pending => CommitStatusState::Pending,
            gitlab::StatusState::Running => CommitStatusState::Running,
            gitlab::StatusState::Success => CommitStatusState::Success,
            gitlab::StatusState::Failed => CommitStatusState::Failed,
        }
    }
}
