// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::UTC;

extern crate git_workarea;
use self::git_workarea::{CommitId, GitContext};

use super::super::super::host::*;

use std::cell::{Cell, RefCell};
use std::cmp::min;
use std::collections::hash_map::HashMap;
use std::rc::Rc;

struct MockIssue {
    issue: Issue,
    comments: Vec<(Comment, Vec<Award>)>,
}

struct MockMergeRequest {
    mr: MergeRequest,
    comments: Vec<(Comment, Vec<Award>)>,
    awards: Vec<Award>,
}

fn commit_status_from_pending(status: PendingCommitStatus, author: User) -> CommitStatus {
    CommitStatus {
        state: status.state,
        author: author,
        refname: status.refname.map(ToString::to_string),
        name: status.name.to_string(),
        description: status.description.to_string(),
    }
}

pub struct MockService {
    projects: HashMap<String, Repo>,
    users: HashMap<String, User>,
    issues: HashMap<u64, Vec<MockIssue>>,
    merge_requests: HashMap<u64, Vec<MockMergeRequest>>,
    user: User,

    step: Cell<usize>,

    issue_comments: RefCell<HashMap<u64, Vec<String>>>,
    mr_comments: RefCell<HashMap<u64, Vec<String>>>,
    commit_comments: RefCell<HashMap<String, Vec<String>>>,
    commit_statuses: RefCell<HashMap<String, Vec<CommitStatus>>>,
    memberships: RefCell<HashMap<String, Vec<Membership>>>,
    hooks: RefCell<HashMap<String, Vec<String>>>,
}

impl MockService {
    fn new() -> Self {
        MockService {
            projects: HashMap::new(),
            users: HashMap::new(),
            issues: HashMap::new(),
            merge_requests: HashMap::new(),
            user: User {
                id: 100,
                handle: "root".to_string(),
                name: "root".to_string(),
                email: "admin@example.com".to_string(),
            },

            step: Cell::new(0),

            issue_comments: RefCell::new(HashMap::new()),
            mr_comments: RefCell::new(HashMap::new()),
            commit_comments: RefCell::new(HashMap::new()),
            commit_statuses: RefCell::new(HashMap::new()),
            memberships: RefCell::new(HashMap::new()),
            hooks: RefCell::new(HashMap::new()),
        }
    }

    fn add_project(mut self, project: Repo) -> Self {
        self.projects.insert(project.name.clone(), project);

        self
    }

    fn add_user(mut self, user: User) -> Self {
        self.users.insert(user.name.clone(), user);

        self
    }

    #[allow(dead_code)]
    fn add_issue(mut self, issue: MockIssue) -> Self {
        self.issues
            .entry(issue.issue.id)
            .or_insert_with(Vec::new)
            .push(issue);

        self
    }

    fn add_merge_request(mut self, mr: MockMergeRequest) -> Self {
        self.merge_requests
            .entry(mr.mr.id)
            .or_insert_with(Vec::new)
            .push(mr);

        self
    }

    fn find_user(&self, user: &str) -> Result<&User> {
        match self.users.get(user) {
            Some(ref p) => Ok(p),
            None => mock_err("invalid user"),
        }
    }

    fn find_user_by_id(&self, user_id: u64) -> Result<&User> {
        let user = self.users
            .iter()
            .find(|&(_, ref user)| user.id == user_id);

        match user {
            Some((_, ref p)) => Ok(p),
            None => mock_err("invalid user"),
        }
    }

    fn find_project(&self, project: &str) -> Result<&Repo> {
        match self.projects.get(project) {
            Some(ref p) => Ok(p),
            None => mock_err("invalid project"),
        }
    }

    fn find_project_by_id(&self, project_id: u64) -> Result<&Repo> {
        let project = self.projects
            .iter()
            .find(|&(_, ref project)| project.id == project_id);

        match project {
            Some((_, ref p)) => Ok(p),
            None => mock_err("invalid project"),
        }
    }

    fn add_membership(&self, project: &str, user: &User, level: u64) {
        self.memberships
            .borrow_mut()
            .entry(project.to_string())
            .or_insert_with(Vec::new)
            .push(Membership {
                user: user.clone(),
                access_level: level,
                expiration: None,
            });
    }

    fn find_memberships(&self, project: &str) -> Result<Vec<Membership>> {
        match self.memberships.borrow().get(project) {
            Some(p) => Ok(p.clone()),
            None => mock_err("invalid project"),
        }
    }

    fn find_issue(&self, id: u64) -> Result<&MockIssue> {
        match self.issues.get(&id) {
            Some(ref p) => {
                let issue_step = min(self.step.get(), p.len() - 1);
                Ok(&p[issue_step])
            },
            None => mock_err("invalid issue"),
        }
    }

    fn find_merge_request(&self, id: u64) -> Result<&MockMergeRequest> {
        match self.merge_requests.get(&id) {
            Some(ref p) => {
                let mr_step = min(self.step.get(), p.len() - 1);
                Ok(&p[mr_step])
            },
            None => mock_err("invalid merge request"),
        }
    }

    pub fn test_service() -> Rc<Self> {
        let base_repo = Repo {
            name: "base".to_string(),
            url: "base".to_string(),
            id: 1,
            forked_from: None,
        };
        let fork_repo = Repo {
            name: "fork".to_string(),
            url: "fork".to_string(),
            id: 2,
            forked_from: None,
        };
        let self_repo = Repo {
            name: "self".to_string(),
            url: concat!(env!("CARGO_MANIFEST_DIR"), "/.git").to_string(),
            id: 3,
            forked_from: None,
        };
        let user_user = User {
            id: 0,
            handle: "user".to_string(),
            name: "user".to_string(),
            email: "user@example.com".to_string(),
        };
        let user_other = User {
            id: 1,
            handle: "other".to_string(),
            name: "other".to_string(),
            email: "other@example.com".to_string(),
        };
        let user_maint = User {
            id: 2,
            handle: "maint".to_string(),
            name: "maint".to_string(),
            email: "maint@example.com".to_string(),
        };
        let user_ignore = User {
            id: 3,
            handle: "ignore".to_string(),
            name: "ignore".to_string(),
            email: "ignore@example.com".to_string(),
        };

        let make_mr = |old_commit: Option<&str>, commit: &str, id: u64, desc: &str| {
            MockMergeRequest {
                mr: MergeRequest {
                    source_repo: fork_repo.clone(),
                    source_branch: format!("topic-{}", id),
                    target_repo: base_repo.clone(),
                    target_branch: "master".to_string(),
                    id: id,
                    url: format!("mr{}", id),
                    work_in_progress: id % 2 == 0,
                    description: desc.to_string(),
                    old_commit: old_commit.map(|c| {
                        Commit {
                            repo: fork_repo.clone(),
                            id: CommitId::new(c),
                            refname: None,
                        }
                    }),
                    commit: Commit {
                        repo: fork_repo.clone(),
                        id: CommitId::new(commit),
                        refname: None,
                    },
                    author: user_user.clone(),
                    check_status: if id % 3 == 0 {
                        CheckStatus::Fail
                    } else {
                        CheckStatus::Pass
                    },

                    reference: format!("!{}", id),
                    remove_source_branch: false,
                },

                comments: Vec::new(),
                awards: Vec::new(),
            }
        };
        let mr1 = |old_commit: Option<&str>, commit: &str| {
            make_mr(old_commit,
                    commit,
                    1,
                    "a simple topic\n```message\nA simple message\n```")
        };
        let mr2 = |old_commit: Option<&str>, commit: &str| {
            make_mr(old_commit, commit, 2, "a different, simple topic")
        };
        let mr3 = |old_commit: Option<&str>, commit: &str| {
            make_mr(old_commit,
                    commit,
                    3,
                    "a topic which conflicts with the update based\n```message\nA \
                     simple\nmultiline message\n```")
        };

        static TOPIC1: &'static str = "7189cf557ba2c7c61881ff8669158710b94d8df1";
        static TOPIC1_CONFLICT: &'static str = "755842266dcc5739c06d61433241f44b9306f24c";
        static TOPIC1_UPDATE: &'static str = "fe70f127605efb6032cacea0bd336428d67ed5a3";
        static TOPIC1_MISSED_UPDATE: &'static str = "1b340d2edcf19077ab3e27ddda7430a6612c2f62";

        static TOPIC2: &'static str = "f6f8de8c7c5f1a081b14f5a47c7798268f383222";

        static TOPIC3: &'static str = "7a28f8ea8759ff4f125ddbca825f976927a61310";

        let mut mr1_base = mr1(None, TOPIC1);
        let mr1_conflict = mr1(Some(TOPIC1), TOPIC1_CONFLICT);
        let mr1_update = mr1(Some(TOPIC1), TOPIC1_UPDATE);
        let mr1_missed_update = mr1(Some(TOPIC1_UPDATE), TOPIC1_MISSED_UPDATE);

        let mr2_base = mr2(None, TOPIC2);

        let mut mr3_base = mr3(None, TOPIC3);

        // Ignore unknown awards.
        mr1_base.awards.push(Award {
            name: "ignored".to_string(),
            author: user_other.clone(),
        });
        // Add trailers based on awards.
        mr1_base.awards.push(Award {
            name: "tada".to_string(),
            author: user_other.clone(),
        });
        // Test tone bits.
        mr1_base.awards.push(Award {
            name: "clap_tone2".to_string(),
            author: user_maint.clone(),
        });
        // Test user references.
        mr1_base.comments
            .push((Comment {
                       id: 0,
                       is_system: false,
                       is_branch_update: false,
                       created_at: UTC::now(),
                       updated_at: UTC::now(),
                       author: user_other.clone(),
                       content: "Tested-by: me".to_string(),
                   },
                   Vec::new()));
        // Duplicate trailers should be collapsed.
        mr1_base.comments
            .push((Comment {
                       id: 1,
                       is_system: false,
                       is_branch_update: false,
                       created_at: UTC::now(),
                       updated_at: UTC::now(),
                       author: user_other.clone(),
                       content: "Tested-by: @other".to_string(),
                   },
                   Vec::new()));
        mr1_base.comments.push((Comment {
                                    id: 2,
                                    is_system: false,
                                    is_branch_update: false,
                                    created_at: UTC::now(),
                                    updated_at: UTC::now(),
                                    author: user_other.clone(),
                                    content: "Tested-by: other <other@example.com>".to_string(),
                                },
                                Vec::new()));
        // +x comments should work.
        mr1_base.comments.push((Comment {
                                    id: 3,
                                    is_system: false,
                                    is_branch_update: false,
                                    created_at: UTC::now(),
                                    updated_at: UTC::now(),
                                    author: user_maint.clone(),
                                    content: "+3".to_string(),
                                },
                                Vec::new()));
        // Unrecognized user references should be ignored.
        mr1_base.comments
            .push((Comment {
                       id: 4,
                       is_system: false,
                       is_branch_update: false,
                       created_at: UTC::now(),
                       updated_at: UTC::now(),
                       author: user_maint.clone(),
                       content: "Acked-by: @unknown".to_string(),
                   },
                   Vec::new()));
        // Allow policies to ignore certain users.
        mr1_base.comments
            .push((Comment {
                       id: 5,
                       is_system: false,
                       is_branch_update: false,
                       created_at: UTC::now(),
                       updated_at: UTC::now(),
                       author: user_ignore.clone(),
                       content: "Acked-by: me".to_string(),
                   },
                   Vec::new()));
        // Policies may ignore trailers.
        mr1_base.comments
            .push((Comment {
                       id: 6,
                       is_system: false,
                       is_branch_update: false,
                       created_at: UTC::now(),
                       updated_at: UTC::now(),
                       author: user_maint.clone(),
                       content: "Meaningless: trailer".to_string(),
                   },
                   Vec::new()));
        // System comments should be ignored.
        mr1_base.comments
            .push((Comment {
                       id: 4,
                       is_system: true,
                       is_branch_update: false,
                       created_at: UTC::now(),
                       updated_at: UTC::now(),
                       author: user_maint.clone(),
                       content: "Rejected-by: @maint".to_string(),
                   },
                   Vec::new()));

        mr3_base.awards.push(Award {
            name: "no_good".to_string(),
            author: user_maint.clone(),
        });

        Rc::new(Self::new()
            .add_project(base_repo.clone())
            .add_project(fork_repo.clone())
            .add_project(self_repo.clone())
            .add_user(user_user.clone())
            .add_user(user_other.clone())
            .add_user(user_maint.clone())
            .add_user(user_ignore.clone())
            .add_merge_request(mr1_base)
            .add_merge_request(mr1_conflict)
            .add_merge_request(mr1_update)
            .add_merge_request(mr1_missed_update)
            .add_merge_request(mr2_base)
            .add_merge_request(mr3_base))
    }

    pub fn step(&self, step: usize) {
        self.step.set(step);
    }

    pub fn reset_data(&self) {
        self.issue_comments.borrow_mut().clear();
        self.mr_comments.borrow_mut().clear();
        self.commit_comments.borrow_mut().clear();
        self.commit_statuses.borrow_mut().clear();
        self.memberships.borrow_mut().clear();
    }

    #[allow(dead_code)]
    pub fn issue_comments(&self, id: u64) -> Vec<String> {
        self.issue_comments
            .borrow_mut()
            .remove(&id)
            .unwrap_or_else(Vec::new)
    }

    pub fn mr_comments(&self, id: u64) -> Vec<String> {
        self.mr_comments
            .borrow_mut()
            .remove(&id)
            .unwrap_or_else(Vec::new)
    }

    #[allow(dead_code)]
    pub fn commit_comments(&self, commit: &str) -> Vec<String> {
        self.commit_comments
            .borrow_mut()
            .remove(commit)
            .unwrap_or_else(Vec::new)
    }

    pub fn commit_statuses(&self, commit: &CommitId) -> Vec<CommitStatus> {
        self.commit_statuses
            .borrow_mut()
            .remove(commit.as_str())
            .unwrap_or_else(Vec::new)
    }

    #[allow(dead_code)]
    pub fn memberships(&self, project: &str) -> Vec<Membership> {
        self.memberships
            .borrow_mut()
            .remove(project)
            .unwrap_or_else(Vec::new)
    }

    #[allow(dead_code)]
    pub fn hooks(&self, project: &str) -> Vec<String> {
        self.hooks
            .borrow_mut()
            .remove(project)
            .unwrap_or_else(Vec::new)
    }

    pub fn remaining_data(&self) -> usize {
        0 +
            self.issue_comments.borrow().len() +
            self.mr_comments.borrow().len() +
            self.commit_comments.borrow().len() +
            self.commit_statuses.borrow().len() +
            self.memberships.borrow().len() +
            self.hooks.borrow().len()
    }
}

error_chain! {
    types {
        MockError, MockErrorKind, MockResultExt, MockResult;
    }
}

fn mock_err_inner<T, M: AsRef<str>>(msg: M) -> MockResult<T> {
    bail!(msg.as_ref())
}

fn mock_err<T, M: AsRef<str>>(msg: M) -> Result<T> {
    ResultExt::chain_err(mock_err_inner(msg), || ErrorKind::Host)
}

impl HostingService for MockService {
    fn service_user(&self) -> &User {
        &self.user
    }

    fn fetch_commit(&self, _: &GitContext, _: &Commit) -> Result<()> {
        Ok(())
    }

    fn fetch_mr(&self, _: &GitContext, _: &MergeRequest) -> Result<()> {
        Ok(())
    }

    fn add_member(&self, project: &str, user: &User, level: u64) -> Result<()> {
        let _ = try!(self.find_project(project));

        Ok(self.add_membership(project, user, level))
    }

    fn members(&self, project: &str) -> Result<Vec<Membership>> {
        let _ = try!(self.find_project(project));

        self.find_memberships(project)
            .map(|memberships| {
                memberships.iter()
                    .cloned()
                    .collect()
            })
    }

    fn add_hook(&self, project: &str, url: &str) -> Result<()> {
        let _ = try!(self.find_project(project));

        let mut hook_map = self.hooks.borrow_mut();
        let mut hooks = hook_map.entry(project.to_string()).or_insert_with(Vec::new);

        info!(target: "test/service",
              "adding a hook to {}",
              project);

        Ok(hooks.push(url.to_string()))
    }

    fn user(&self, user: &str) -> Result<User> {
        Ok(try!(self.find_user(user)).clone())
    }

    fn commit(&self, project: &str, commit: &CommitId) -> Result<Commit> {
        let project = try!(self.find_project(project));

        Ok(Commit {
            repo: project.clone(),
            refname: None,
            // Just act as if the commit is a part of the project.
            id: commit.clone(),
        })
    }

    fn issue(&self, project: &str, id: u64) -> Result<Issue> {
        let _ = try!(self.find_project(project));

        Ok(try!(self.find_issue(id)).issue.clone())
    }

    fn merge_request(&self, project: &str, id: u64) -> Result<MergeRequest> {
        let _ = try!(self.find_project(project));

        Ok(try!(self.find_merge_request(id)).mr.clone())
    }

    fn repo(&self, project: &str) -> Result<Repo> {
        Ok(try!(self.find_project(project)).clone())
    }

    fn user_by_id(&self, user: u64) -> Result<User> {
        Ok(try!(self.find_user_by_id(user)).clone())
    }

    fn commit_by_id(&self, project: u64, commit: &CommitId) -> Result<Commit> {
        let project = try!(self.find_project_by_id(project));

        Ok(Commit {
            repo: project.clone(),
            refname: None,
            // Just act as if the commit is a part of the project.
            id: commit.clone(),
        })
    }

    fn issue_by_id(&self, project: u64, id: u64) -> Result<Issue> {
        let _ = try!(self.find_project_by_id(project));

        Ok(try!(self.find_issue(id)).issue.clone())
    }

    fn merge_request_by_id(&self, project: u64, id: u64) -> Result<MergeRequest> {
        let _ = try!(self.find_project_by_id(project));

        Ok(try!(self.find_merge_request(id)).mr.clone())
    }

    fn repo_by_id(&self, project: u64) -> Result<Repo> {
        Ok(try!(self.find_project_by_id(project)).clone())
    }

    fn get_issue_comments(&self, issue: &Issue) -> Result<Vec<Comment>> {
        let issue = try!(self.find_issue(issue.id));

        Ok(issue.comments
            .iter()
            .map(|issue_comment| issue_comment.0.clone())
            .collect())
    }

    fn post_issue_comment(&self, issue: &Issue, content: &str) -> Result<()> {
        let mut comment_map = self.issue_comments.borrow_mut();
        let mut comments = comment_map.entry(issue.id).or_insert_with(Vec::new);

        info!(target: "test/service",
              "posting a comment to Issue#{}: {}",
              issue.id, content);

        Ok(comments.push(content.to_string()))
    }

    fn get_mr_comments(&self, mr: &MergeRequest) -> Result<Vec<Comment>> {
        let mr = try!(self.find_merge_request(mr.id));

        Ok(mr.comments
            .iter()
            .map(|mr_comment| mr_comment.0.clone())
            .collect())
    }

    fn post_mr_comment(&self, mr: &MergeRequest, content: &str) -> Result<()> {
        let mut comment_map = self.mr_comments.borrow_mut();
        let mut comments = comment_map.entry(mr.id).or_insert_with(Vec::new);

        info!(target: "test/service",
              "posting a comment to MR#{}: {}",
              mr.id, content);

        Ok(comments.push(content.to_string()))
    }

    fn post_commit_comment(&self, commit: &Commit, content: &str) -> Result<()> {
        let mut comment_map = self.commit_comments.borrow_mut();
        let mut comments = comment_map.entry(commit.id.as_str().to_string())
            .or_insert_with(Vec::new);

        info!(target: "test/service",
              "posting a comment to commit {}: {}",
              commit.id, content);

        Ok(comments.push(content.to_string()))
    }

    fn get_commit_statuses(&self, commit: &Commit) -> Result<Vec<CommitStatus>> {
        Ok(self.commit_statuses
            .borrow()
            .get(commit.id.as_str())
            .map(|ref v| {
                v.iter()
                    .cloned()
                    .collect()
            })
            .unwrap_or_else(Vec::new))
    }

    fn post_commit_status(&self, status: PendingCommitStatus) -> Result<()> {
        let mut status_map = self.commit_statuses.borrow_mut();
        let mut statuses = status_map.entry(status.commit.id.as_str().to_string())
            .or_insert_with(Vec::new);

        info!(target: "test/service",
              "posting a {:?} status to commit {} ({:?}) by {}: {}",
              status.state, status.commit.id, status.refname,
              status.name, status.description);

        statuses.push(commit_status_from_pending(status, self.user.clone()));

        Ok(())
    }

    fn get_mr_awards(&self, mr: &MergeRequest) -> Result<Vec<Award>> {
        let mr = try!(self.find_merge_request(mr.id));

        Ok(mr.awards
            .iter()
            .cloned()
            .collect())
    }

    fn get_mr_comment_awards(&self, mr: &MergeRequest, comment: &Comment) -> Result<Vec<Award>> {
        let mr = try!(self.find_merge_request(mr.id));

        Ok(mr.comments[comment.id as usize]
            .1
            .iter()
            .cloned()
            .collect())
    }

    // Leave unimplemented; awards created by actions are dropped because of this.
    fn award_mr_comment(&self, _mr: &MergeRequest, _comment: &Comment, _award: &str) -> Result<()> {
        Ok(())
    }
}
