// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate git_checks;
use self::git_checks::GitCheckConfiguration;
use self::git_checks::checks;

extern crate git_workarea;
use self::git_workarea::{CommitId, GitContext};

use super::host::MockService;
use super::utils::test_workspace_dir;
use super::super::check::*;
use super::super::super::host::{CommitStatusState, HostingService};

use std::path::Path;
use std::process::Command;

static BASE: &'static str = "58b2dee73ab6e6b1f3587b41d0ccdbe2ded785dd";

fn git_context(workspace_path: &Path) -> GitContext {
    let gitdir = workspace_path.join("origin");
    let clone = Command::new("git")
        .arg("clone")
        .arg("--bare")
        .arg(concat!(env!("CARGO_MANIFEST_DIR"), "/.git"))
        .arg(&gitdir)
        .output()
        .unwrap();
    if !clone.status.success() {
        panic!("origin clone failed: {}",
               String::from_utf8_lossy(&clone.stderr));
    }

    GitContext::new(gitdir)
}

fn admins() -> Vec<String> {
    vec![
        "admin".to_string(),
        "admin2".to_string(),
    ]
}

#[test]
// A successful check should not comment, but post a success status on the branch and each commit.
fn test_check_success() {
    let tempdir = test_workspace_dir("test_check_success");
    let ctx = git_context(tempdir.path());
    let service = MockService::test_service();
    let mr = service.merge_request("base", 1).unwrap();
    let config = GitCheckConfiguration::new();

    service.step(2);
    let mr_update = service.merge_request("base", 1).unwrap();
    let admins = admins();
    let check = Check::new(ctx.clone(), service.clone(), config, &admins);

    let result = check.check_mr("test_check_success", &CommitId::new(BASE), &mr_update)
        .unwrap();
    assert_eq!(result, CheckStatus::Pass);

    let mr_commit_statuses_head = service.commit_statuses(&mr_update.commit.id);
    let mr_commit_statuses_previous = service.commit_statuses(&mr.commit.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_statuses_previous.len(), 1);

    let status = &mr_commit_statuses_previous[0];
    assert_eq!(status.state, CommitStatusState::Success);
    assert_eq!(status.refname, None);
    assert_eq!(status.name, "ghostflow-commit-check");
    assert_eq!(status.description, "basic content checks");

    assert_eq!(mr_commit_statuses_head.len(), 2);

    let status = &mr_commit_statuses_head[0];
    assert_eq!(status.state, CommitStatusState::Success);
    assert_eq!(status.refname, None);
    assert_eq!(status.name, "ghostflow-commit-check");
    assert_eq!(status.description, "basic content checks");

    let status = &mr_commit_statuses_head[1];
    assert_eq!(status.state, CommitStatusState::Success);
    assert_eq!(status.refname, Some(mr.source_branch.clone()));
    assert_eq!(status.name, "ghostflow-branch-check");
    assert_eq!(status.description,
               "overall branch status for basic content checks");
}

#[test]
// A failed check should comment, and post a failure status on the branch and each failing commit;
// passing commits should still pass.
fn test_check_failure() {
    let tempdir = test_workspace_dir("test_check_failure");
    let ctx = git_context(tempdir.path());
    let service = MockService::test_service();
    let mr = service.merge_request("base", 1).unwrap();

    // Add a check which rejects the first commit on the branch.
    let bad_commits = checks::BadCommits::new(&[mr.commit.id.as_str()]);

    let mut config = GitCheckConfiguration::new();
    config.add_check(&bad_commits);

    service.step(2);
    let mr_update = service.merge_request("base", 1).unwrap();
    let admins = admins();
    let check = Check::new(ctx.clone(), service.clone(), config, &admins);

    let result = check.check_mr("test_check_failure", &CommitId::new(BASE), &mr_update)
        .unwrap();
    assert_eq!(result, CheckStatus::Fail);

    let mr_comments = service.mr_comments(mr.id);
    let mr_commit_statuses_head = service.commit_statuses(&mr_update.commit.id);
    let mr_commit_statuses_previous = service.commit_statuses(&mr.commit.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_comments.len(), 1);
    assert_eq!(mr_comments[0],
               concat!("Errors:  \n",
                       "  - commit 7189cf5 is a known-bad commit that was removed from the \
                        server.\n",
                       "\n",
                       "Alerts:  \n",
                       "  - commit 7189cf5 was pushed to the server.\n",
                       "\n",
                       "Please rewrite commits to fix the errors listed above (adding fixup \
                        commits will not resolve the errors) and force-push the branch again to \
                        update the merge request.\n",
                       "\n",
                       "Alert: @admin @admin2."));

    let status = &mr_commit_statuses_previous[0];
    assert_eq!(status.state, CommitStatusState::Failed);
    assert_eq!(status.refname, None);
    assert_eq!(status.name, "ghostflow-commit-check");
    assert_eq!(status.description, "basic content checks");

    assert_eq!(mr_commit_statuses_previous.len(), 1);

    let status = &mr_commit_statuses_previous[0];
    assert_eq!(status.state, CommitStatusState::Failed);
    assert_eq!(status.refname, None);
    assert_eq!(status.name, "ghostflow-commit-check");
    assert_eq!(status.description, "basic content checks");

    assert_eq!(mr_commit_statuses_head.len(), 2);

    let status = &mr_commit_statuses_head[0];
    assert_eq!(status.state, CommitStatusState::Success);
    assert_eq!(status.refname, None);
    assert_eq!(status.name, "ghostflow-commit-check");
    assert_eq!(status.description, "basic content checks");

    let status = &mr_commit_statuses_head[1];
    assert_eq!(status.state, CommitStatusState::Failed);
    assert_eq!(status.refname, Some(mr.source_branch.clone()));
    assert_eq!(status.name, "ghostflow-branch-check");
    assert_eq!(status.description,
               "overall branch status for basic content checks");
}

#[test]
// A successful check with warnings should comment, and post a success status on the branch and
// each commit.
fn test_check_warning() {
    let tempdir = test_workspace_dir("test_check_warning");
    let ctx = git_context(tempdir.path());
    let service = MockService::test_service();
    let mr = service.merge_request("base", 2).unwrap();
    let config = GitCheckConfiguration::new();
    let admins = admins();
    let check = Check::new(ctx.clone(), service.clone(), config, &admins);

    let result = check.check_mr("test_check_warning", &CommitId::new(BASE), &mr).unwrap();
    assert_eq!(result, CheckStatus::Pass);

    let mr_comments = service.mr_comments(mr.id);
    let mr_commit_statuses = service.commit_statuses(&mr.commit.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_comments.len(), 1);
    assert_eq!(mr_comments[0],
               concat!("Warnings:  \n",
                       "  - the merge request is marked as a work-in-progress.\n",
                       "\n",
                       "The warnings may be temporary; if they are, a comment with the text \
                        `Do: check` at the end of the comment will rerun the checks."));

    assert_eq!(mr_commit_statuses.len(), 2);

    let status = &mr_commit_statuses[0];
    assert_eq!(status.state, CommitStatusState::Success);
    assert_eq!(status.refname, None);
    assert_eq!(status.name, "ghostflow-commit-check");
    assert_eq!(status.description, "basic content checks");

    let status = &mr_commit_statuses[1];
    assert_eq!(status.state, CommitStatusState::Success);
    assert_eq!(status.refname, Some(mr.source_branch.clone()));
    assert_eq!(status.name, "ghostflow-branch-check");
    assert_eq!(status.description,
               "overall branch status for basic content checks");
}

#[test]
// Merges on passing branches should be silent.
fn test_check_mr_merge_ok() {
    let tempdir = test_workspace_dir("test_check_mr_merge_ok");
    let ctx = git_context(tempdir.path());
    let service = MockService::test_service();
    let mr = service.merge_request("base", 1).unwrap();
    let config = GitCheckConfiguration::new();
    let admins = admins();
    let check = Check::new(ctx.clone(), service.clone(), config, &admins);

    let result = check.check_mr_merge(&mr.commit.id, &mr.author.identity(), &mr)
        .unwrap();
    assert_eq!(result, CheckStatus::Pass);

    assert_eq!(service.remaining_data(), 0);
}

#[test]
// Merges on failing merge commits should comment and fail.
fn test_check_mr_merge_fail() {
    let tempdir = test_workspace_dir("test_check_mr_merge_fail");
    let ctx = git_context(tempdir.path());
    let service = MockService::test_service();
    let mr = service.merge_request("base", 1).unwrap();

    // Add a check which rejects the merge commit.
    let bad_commits = checks::BadCommits::new(&[mr.commit.id.as_str()]);

    let mut config = GitCheckConfiguration::new();
    config.add_check(&bad_commits);

    let admins = admins();
    let check = Check::new(ctx.clone(), service.clone(), config, &admins);

    let result = check.check_mr_merge(&mr.commit.id, &mr.author.identity(), &mr)
        .unwrap();
    assert_eq!(result, CheckStatus::Fail);

    let mr_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_comments.len(), 1);
    assert_eq!(mr_comments[0],
               concat!("The proposed merge commit 7189cf557ba2c7c61881ff8669158710b94d8df1 \
                        failed content checks:\n",
                       "\n",
                       "Errors:  \n",
                       "  - commit 7189cf5 is a known-bad commit that was removed from the \
                        server.\n",
                       "\n",
                       "Alerts:  \n",
                       "  - commit 7189cf5 was pushed to the server.\n",
                       "\n",
                       "Alert: @admin @admin2."));
}

#[test]
// Merges on work-in-progress (but otherwise passing) merge requests should be denied.
fn test_check_mr_merge_wip() {
    let tempdir = test_workspace_dir("test_check_mr_merge_wip");
    let ctx = git_context(tempdir.path());
    let service = MockService::test_service();
    let mr = service.merge_request("base", 2).unwrap();
    let config = GitCheckConfiguration::new();
    let admins = admins();
    let check = Check::new(ctx.clone(), service.clone(), config, &admins);

    let result = check.check_mr_merge(&mr.commit.id, &mr.author.identity(), &mr)
        .unwrap();
    assert_eq!(result, CheckStatus::Fail);

    let mr_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_comments.len(), 1);
    assert_eq!(mr_comments[0],
               concat!("The proposed merge commit f6f8de8c7c5f1a081b14f5a47c7798268f383222 \
                        failed content checks:\n",
                       "\n",
                       "Warnings:  \n",
                       "  - the merge request is marked as a work-in-progress."));
}
