// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `test` action using ref-based testing.
//!
//! This action pushes refs into a ref namespace for use by testing machines.

extern crate git_workarea;
use self::git_workarea::GitContext;

use super::super::super::host::{self, CommitStatusState, HostedProject, MergeRequest};

error_chain! {
    links {
        Host(host::Error, host::ErrorKind)
            #[doc = "Errors from the service host."];
    }

    errors {
        /// An error occurred when executing git commands.
        Git(msg: String) {
            display("git error: {}", msg)
        }
    }
}

/// Implementation of the `test` action.
pub struct TestRefs {
    ctx: GitContext,
    project: HostedProject,
    namespace: String,
    quiet: bool,
}

impl TestRefs {
    /// Create a new test action.
    pub fn new(ctx: GitContext, project: HostedProject) -> Self {
        TestRefs {
            ctx: ctx,
            project: project,
            namespace: "test-topics".to_string(),
            quiet: false,
        }
    }

    /// Reduce the number of comments made by the test action.
    pub fn quiet(&mut self) -> &mut Self {
        self.quiet = true;
        self
    }

    /// The ref namespace to use for test topics.
    pub fn ref_namespace<N>(&mut self, namespace: N) -> &mut Self
        where N: ToString,
    {
        self.namespace = namespace.to_string();
        self
    }

    /// Push a merge request for testing.
    pub fn test_mr(&self, mr: &MergeRequest) -> Result<()> {
        info!(target: "ghostflow/test/refs",
              "pushing a test ref for {}",
              mr.url);

        // Fetch the merge request into the stager's git context.
        try!(self.project.service.fetch_mr(&self.ctx, mr));

        let refname = self.refname(mr);

        let update_ref = try!(self.ctx
            .git()
            .arg("update-ref")
            .arg(&refname)
            .arg(mr.commit.id.as_str())
            .output()
            .chain_err(|| "failed to construct update-ref command"));
        if !update_ref.status.success() {
            bail!(ErrorKind::Git(format!("failed to update the test-topic ref {}: {}",
                                         refname,
                                         String::from_utf8_lossy(&update_ref.stderr))));
        }

        let push = try!(self.ctx
            .git()
            .arg("push")
            .arg("origin")
            .arg("--atomic")
            .arg("--porcelain")
            .arg(format!("{}:{}", refname, refname))
            .output()
            .chain_err(|| "failed to construct push command"));
        if !push.status.success() {
            bail!(ErrorKind::Git(format!("failed to push the test-topic ref {}: {}",
                                         refname,
                                         String::from_utf8_lossy(&push.stderr))));
        }

        self.send_info_mr_comment(mr, "This topic has been pushed for testing.");

        self.send_mr_commit_status(mr, CommitStatusState::Success, "pushed for testing");

        Ok(())
    }

    /// Remove a merge request from the testing set.
    pub fn untest_mr(&self, mr: &MergeRequest) -> Result<()> {
        info!(target: "ghostflow/test/refs",
              "deleting the test ref for {}",
              mr.url);

        let refname = self.refname(mr);

        let show_ref = try!(self.ctx
            .git()
            .arg("show-ref")
            .arg("--quiet")
            .arg("--verify")
            .arg(&refname)
            .status()
            .chain_err(|| "failed to construct show-ref command"));
        if !show_ref.success() {
            // There is no such ref; skip.
            return Ok(());
        }

        try!(self.delete_ref(&refname));

        let push = try!(self.ctx
            .git()
            .arg("push")
            .arg("origin")
            .arg("--atomic")
            .arg("--porcelain")
            .arg(format!(":{}", refname))
            .output()
            .chain_err(|| "failed to construct push command"));
        if !push.status.success() {
            bail!(ErrorKind::Git(format!("failed to push the test-topic ref {}: {}",
                                         refname,
                                         String::from_utf8_lossy(&push.stderr))));
        }

        self.send_mr_commit_status(mr, CommitStatusState::Success, "removed from testing");

        Ok(())
    }

    /// Clear the set of merge requests for testing.
    pub fn clear_all_mrs(&self) -> Result<()> {
        info!(target: "ghostflow/test/refs",
              "clearing all test refs for {}",
              self.project.name);

        let test_refs = try!(self.ctx
            .git()
            .arg("for-each-ref")
            .arg("--format=%(refname:strip=2)")
            .arg(format!("refs/{}/", self.namespace))
            .output()
            .chain_err(|| "failed to construct for-each-ref command"));
        if !test_refs.status.success() {
            bail!(ErrorKind::Git(format!("failed to list all test refs: {}",
                                         String::from_utf8_lossy(&test_refs.stderr))));
        }
        let topic_ids = String::from_utf8_lossy(&test_refs.stdout);
        let cleanup_results = topic_ids.lines()
            .filter_map(|topic_id| {
                match topic_id.parse() {
                    Ok(id) => Some(id),
                    Err(err) => {
                        error!(target: "workflow/test/refs",
                               "failed to parse {} as a topic id; deleting the ref: {:?}",
                               topic_id,
                               err);

                        let refname = format!("refs/{}/{}", self.namespace, topic_id);
                        self.lenient_delete_ref(refname);

                        None
                    },
                }
            })
            .filter_map(|topic_id| {
                match self.project.merge_request(topic_id) {
                    Ok(mr) => Some(mr),
                    Err(err) => {
                        error!(target: "workflow/test/refs",
                               "ref {} is not a valid merge request; deleting the ref: {:?}",
                               topic_id,
                               err);

                        let refname = format!("refs/{}/{}", self.namespace, topic_id);
                        self.lenient_delete_ref(refname);

                        None
                    },
                }
            })
            .map(|mr| self.untest_mr(&mr))
            .collect::<Vec<_>>();

        try!(cleanup_results.into_iter()
            .collect::<Result<Vec<_>>>());

        Ok(())
    }

    fn delete_ref(&self, refname: &str) -> Result<()> {
        info!(target: "workflow/test/refs",
              "deleting test ref {}",
              refname);

        let delete_ref = try!(self.ctx
            .git()
            .arg("update-ref")
            .arg("-d")
            .arg(&refname)
            .output()
            .chain_err(|| "failed to construct update-ref command"));
        if !delete_ref.status.success() {
            bail!(ErrorKind::Git(format!("failed to delete test ref {}: {}",
                                         refname,
                                         String::from_utf8_lossy(&delete_ref.stderr))));
        }

        Ok(())
    }

    fn refname(&self, mr: &MergeRequest) -> String {
        format!("refs/{}/{}", self.namespace, mr.id)
    }

    fn lenient_delete_ref(&self, refname: String) {
        let _ = self.delete_ref(&refname)
            .map_err(|err| {
                error!(target: "workflow/test/refs",
                       "failed to delete the {} ref from {}: {:?}",
                       refname,
                       self.project.name,
                       err);
            });
    }

    fn send_mr_commit_status(&self, mr: &MergeRequest, status: CommitStatusState, desc: &str) {
        let status = mr.create_commit_status(status, "ghostflow-test", desc);
        if let Err(err) = self.project.service.post_commit_status(status) {
            warn!(target: "workflow/test/refs",
                  "failed to post a commit status for mr {} on {} for '{}': {:?}",
                  mr.id,
                  mr.commit.id,
                  desc,
                  err);
        }
    }

    fn send_mr_comment(&self, mr: &MergeRequest, content: &str) {
        if let Err(err) = self.project.service.post_mr_comment(mr, content) {
            error!(target: "workflow/test/refs",
                   "failed to post a comment to merge request: {}, {}: {:?}",
                   self.project.name,
                   mr.id,
                   err);
        }
    }

    fn send_info_mr_comment(&self, mr: &MergeRequest, content: &str) {
        if !self.quiet {
            self.send_mr_comment(mr, content)
        }
    }
}
