// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `merge` action.
//!
//! This action performs the merge of a merge request topic into the target branch. It gathers
//! information from the merge request such as reviewers, testers, acceptance or rejection
//! messages, and more to determine the resulting merge commit message.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate git_workarea;
use self::git_workarea::{CommitId, GitContext, Identity, MergeResult, MergeStatus};

extern crate itertools;
use self::itertools::Itertools;

use super::super::host::{self, Award, Comment, HostedProject, HostingService, MergeRequest, User};
use super::super::utils::{Trailer, TrailerRef};

use std::io::Write;
use std::iter;

error_chain! {
    links {
        GitWorkarea(git_workarea::Error, git_workarea::ErrorKind)
            #[doc = "Errors from the git-workarea crate."];
        Host(host::Error, host::ErrorKind)
            #[doc = "Errors from the service host."];
    }

    errors {
        /// An error occurred when executing git commands.
        Git(msg: String) {
            display("git error: {}", msg)
        }
    }
}

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
/// The result of the merge action.
pub enum MergeActionResult {
    /// Everything worked fine.
    Success,
    /// The push failed.
    ///
    /// This likely means that the remote changed in some way and the merge will need to be
    /// restarted.
    PushFailed,
    /// The merge failed due to conflicts or otherwise unsuitable state of the merge request.
    ///
    /// Failures require user interaction before they may be attempted again.
    Failed,
}

/// A trait which represents a filter for trailers to enforce policies on merging.
pub trait MergePolicyFilter {
    /// A method to process trailers and apply policies.
    ///
    /// The `user` parameter is `None` if no user account is associated (or could be found) with
    /// the trailer value.
    fn process_trailer(&mut self, trailer: Trailer, user: Option<&User>);

    /// The result of the policy.
    ///
    /// The result is either a set of trailers to use for the merge commit message or a list of
    /// reasons the merge is not allowed.
    fn result(self) -> ::std::result::Result<Vec<Trailer>, Vec<String>>;
}

/// A merge policy.
///
/// Merge policies create filters which look at trailers for a merge request and decide what to do
/// with them.
pub trait MergePolicy {
    /// The policy filter type.
    type Filter: MergePolicyFilter;

    /// Create a new policy filter for the given merge request.
    fn for_mr(&self, mr: &MergeRequest) -> Self::Filter;
}

// Merge policies which may be constructed via `Default` can be their own factory.
impl<T> MergePolicy for T
    where T: MergePolicyFilter + Default,
{
    type Filter = T;

    fn for_mr(&self, _: &MergeRequest) -> Self::Filter {
        T::default()
    }
}

/// Implementation of the `merge` action.
pub struct Merge<P>
    where P: MergePolicy,
{
    branch: String,
    ctx: GitContext,
    project: HostedProject,
    policy: P,
    quiet: bool,
    log_limit: Option<usize>,
}

impl<P> Merge<P>
    where P: MergePolicy,
{
    /// Create a new merge action.
    pub fn new<B>(ctx: GitContext, branch: B, project: HostedProject, policy: P) -> Self
        where B: ToString,
    {
        Merge {
            branch: branch.to_string(),
            ctx: ctx,
            project: project,
            policy: policy,
            quiet: false,
            log_limit: None,
        }
    }

    /// Reduce the number of comments made by the merge action.
    ///
    /// The comments created by this action can be a bit much. This reduces the comments to those
    /// which are errors or are important.
    pub fn quiet(&mut self) -> &mut Self {
        self.quiet = true;
        self
    }

    /// Limit the number of log entries in merge commit messages.
    ///
    /// Anything beyond this limit (if present) is elided.
    pub fn log_limit(&mut self, log_limit: Option<usize>) -> &mut Self {
        self.log_limit = log_limit;
        self
    }

    /// Merge a merge request into the branch.
    ///
    /// Information for the merge commit is gathered from the comment stream as well as the merge
    /// request itself. Comments from before the last update are ignored since they do not apply to
    /// the latest incarnation of the topic.
    pub fn merge_mr(&self, mr: &MergeRequest, who: &Identity, when: &DateTime<UTC>)
                    -> Result<MergeActionResult> {
        self._merge_mr(mr, &mr.source_branch, who, when)
    }

    /// Merge a merge request into the branch with a different name.
    pub fn merge_mr_named<B>(&self, mr: &MergeRequest, branch_name: B, who: &Identity,
                             when: &DateTime<UTC>)
                             -> Result<MergeActionResult>
        where B: AsRef<str>,
    {
        self._merge_mr(mr, branch_name.as_ref(), who, when)
    }

    fn _merge_mr(&self, mr: &MergeRequest, branch_name: &str, who: &Identity,
                 when: &DateTime<UTC>)
                 -> Result<MergeActionResult> {
        info!(target: "ghostflow/merge",
              "attempting to merge {}",
              mr.url);

        if mr.work_in_progress {
            self.send_mr_comment(mr,
                                 "This merge request is marked as a Work in Progress and may not \
                                  be merged. Please remove the Work in Progress state first.");
            return Ok(MergeActionResult::Failed);
        }

        // Fetch the commit into the merge's git context.
        try!(self.project.service.fetch_mr(&self.ctx, mr));

        let branch_id = CommitId::new(&self.branch);

        // Determine if the topic is mergeable at all.
        let merge_status = try!(self.ctx.mergeable(&branch_id, &mr.commit.id));
        let bases = if let MergeStatus::Mergeable(bases) = merge_status {
            bases
        } else {
            self.send_mr_comment(mr, &unmerged_status_message(&merge_status));
            return Ok(MergeActionResult::Failed);
        };

        info!(target: "ghostflow/merge",
              "preparing to merge {}",
              mr.url);

        // Prepare a work area to perform the actual merge.
        let workarea = try!(self.ctx.prepare(&branch_id));
        let merge_result = try!(workarea.setup_merge(&bases, &branch_id, &mr.commit.id));
        let mut merge_command = match merge_result {
            MergeResult::Conflict(conflicts) => {
                let mut conflict_paths = conflicts.iter()
                    .map(|conflict| conflict.path().to_string_lossy())
                    .dedup();
                self.send_mr_comment(mr,
                                     &format!("This merge request contains conflicts in the \
                                               following paths:  \n`{}`",
                                              conflict_paths.join("`  \n`")));
                return Ok(MergeActionResult::Failed);
            },
            MergeResult::Ready(command) => command,
            MergeResult::_Phantom(_) => unreachable!(),
        };

        // Add authorship information to the commit message. Committer information is provided by
        // the default git environment.
        merge_command.env("GIT_AUTHOR_NAME", &who.name)
            .env("GIT_AUTHOR_EMAIL", &who.email)
            .env("GIT_AUTHOR_DATE", when.to_rfc2822());

        // Gather trailer information from the merge request.
        let comments = try!(self.project.service.get_mr_comments(mr));
        let mr_awards = match self.project.service.get_mr_awards(mr) {
            Ok(awards) => awards,
            Err(err) => {
                error!(target: "ghostflow/merge",
                       "failed to get awards for mr {}: {:?}",
                       mr.url,
                       err);

                vec![]
            },
        };

        let mut mr_policy = self.policy.for_mr(mr);

        comments.iter()
            // Look at comments from newest to oldest.
            .rev()
            // Stop when we have a branch update comment.
            //
            // TODO: This should instead probably be "newer than the last branch update" since it
            // looks like not all hosting services (e.g., Github) support seeing the "system"
            // comments via the API. However, it seems that the `updated_at` field provided by
            // Gitlab and Github changes with things like editing the description. May require
            // feature requests to Github and a patch to Gitlab.
            .take_while(|comment| !comment.is_branch_update)
            // Ignore system comments.
            .filter(|comment| !comment.is_system)
            // Parse trailers from each comment.
            .map(|comment| {
                parse_comment_for_trailers(self.project.service.as_ref(), &comment)
            })
            // Now that we have all the trailers, gather them up.
            .collect::<Vec<_>>()
            .into_iter()
            // Put them back into chronological order.
            .rev()
            // Put all of the trailers together into a single vector.
            .flatten()
            // Get trailers via awards on the MR itself.
            .chain({
                mr_awards.into_iter()
                    .filter_map(parse_award_as_trailers)
                    .collect::<Vec<_>>()
            })
            // Filter trailers through the policy.
            .foreach(|(trailer, user_opt)| {
                mr_policy.process_trailer(trailer, user_opt.as_ref())
            });

        let trailers = match mr_policy.result() {
            Ok(trailers) => trailers.into_iter().unique(),
            Err(reasons) => {
                let reason = reasons.into_iter()
                    .join("  \n  - ");
                self.send_mr_comment(mr,
                                     &format!("This merge request may not be merged because:  \n  \
                                               - {}",
                                              reason));

                return Ok(MergeActionResult::Failed);
            },
        };

        let commit_message = try!(self.build_commit_message(mr, branch_name, trailers));

        info!(target: "ghostflow/merge",
              "merging {}",
              mr.url);

        let mut merge_process = try!(merge_command.spawn()
            .chain_err(|| "failed to construct commit-tree command"));
        try!(write!(merge_process.stdin.as_mut().unwrap(), "{}", commit_message)
            .chain_err(|| "failed to write the merge commit message"));
        let merge_commit = try!(merge_process.wait_with_output()
            .chain_err(|| "failed to execute commit-tree command"));
        if !merge_commit.status.success() {
            bail!(ErrorKind::Git(format!("failed to commit the merge: {}",
                                         String::from_utf8_lossy(&merge_commit.stderr))));
        }
        let commit_id = String::from_utf8_lossy(&merge_commit.stdout);

        let push = try!(self.ctx
            .git()
            .arg("push")
            .arg("--atomic")
            .arg("--porcelain")
            .arg("origin")
            .arg(format!("{}:{}", commit_id.trim(), self.branch))
            .output()
            .chain_err(|| "failed to construct push command"));
        if !push.status.success() {
            warn!(target: "ghostflow/merge",
                  "failed to push the merge of {} to the remote server: {}",
                  mr.url,
                  String::from_utf8_lossy(&push.stderr));

            self.send_info_mr_comment(mr,
                                      "Automatic merge succeeded, but pushing to the remote \
                                       failed!");

            return Ok(MergeActionResult::PushFailed);
        }

        if mr.remove_source_branch {
            info!(target: "ghostflow/merge",
                  "removing the source branch for {}",
                  mr.url);

            let push_remove = try!(self.ctx
                .git()
                .arg("push")
                .arg("--atomic")
                .arg("--porcelain")
                .arg("--delete")
                .arg(&mr.source_repo.url)
                .arg(&mr.source_branch)
                .output()
                .chain_err(|| "failed to construct push command"));
            if !push_remove.status.success() {
                warn!(target: "ghostflow/merge",
                      "failed to remove the source branch of {} from the remote server: {}",
                      mr.url,
                      String::from_utf8_lossy(&push_remove.stderr));
            }
        }

        self.send_info_mr_comment(mr, "Topic successfully merged and pushed!");

        Ok(MergeActionResult::Success)
    }

    fn build_commit_message<I>(&self, mr: &MergeRequest, branch_name: &str, trailers: I)
                               -> Result<String>
        where I: Iterator<Item = Trailer>,
    {
        let mut branch_summary = mr.description
            // Break it into lines.
            .lines()
            // Find the message block.
            .skip_while(|&line| line != "```message")
            // Skip the entry line.
            .skip(1)
            // Take it until the end of the block.
            .take_while(|&line| line != "```")
            // Add newlines.
            .map(|line| format!("{}\n", line))
            // Join the lines together.
            .join("");
        if !branch_summary.is_empty() {
            // Append a separator if we have a branch description.
            branch_summary.push('\n');
        }

        let mut log_command = self.ctx.git();
        log_command
            .arg("log")
            .arg("--date-order")
            .arg("--format=%h %s")
            // FIXME: Use the new "auto abbrev" functionality in git. It appears that this will
            // give us "7" by default, but it is likely better than this once we outgrow `8`.
            .arg("--abbrev=8");

        if let Some(limit) = self.log_limit {
            // Get up to one more than the maximum. This is done so that we can detect that there
            // are more so that an elision indicator may be added.
            log_command.arg(format!("--max-count={}", limit + 1));
        }

        let log = try!(log_command.arg(format!("{}..{}", self.branch, &mr.commit.id))
            .output()
            .chain_err(|| "failed to construct log command"));
        if !log.status.success() {
            bail!(ErrorKind::Git(format!("failed to get a log of commits on the topic: {}",
                                         String::from_utf8_lossy(&log.stderr))));
        }
        let log_output = String::from_utf8_lossy(&log.stdout);
        let mut log_lines = log_output.lines().collect::<Vec<_>>();
        // Elide the log if there are too many entries.
        if let Some(limit) = self.log_limit {
            if limit == 0 {
                log_lines.clear()
            } else if log_lines.len() > limit {
                log_lines[limit] = "...";
            }
        }
        let mut log_summary = log_lines.into_iter()
            .map(|line| format!("{}\n", line))
            .join("");
        if !log_summary.is_empty() {
            // Append a separator if we have a log summary.
            log_summary.push('\n');
        }

        let trailer_summary =
            trailers.chain(iter::once(Trailer::new("Merge-request", &mr.reference)))
                .map(|trailer| format!("{}\n", trailer))
                .join("");

        Ok(format!("Merge topic \'{}\'\n\
                    \n\
                    {}{}{}",
                   branch_name,
                   branch_summary,
                   log_summary,
                   trailer_summary))
    }

    fn send_mr_comment(&self, mr: &MergeRequest, content: &str) {
        if let Err(err) = self.project.service.post_mr_comment(mr, content) {
            error!(target: "ghostflow/merge",
                   "failed to post a comment to merge request: {}, {}: {:?}",
                   self.project.name,
                   mr.id,
                   err);
        }
    }

    fn send_info_mr_comment(&self, mr: &MergeRequest, content: &str) {
        if !self.quiet {
            self.send_mr_comment(mr, content)
        }
    }
}

fn unmerged_status_message(reason: &MergeStatus) -> String {
    let reason_message = match *reason {
        MergeStatus::NoCommonHistory => "there is no common history",
        MergeStatus::AlreadyMerged => "it has already been merged",
        MergeStatus::Mergeable(_) => "it is…mergeable? Sorry, something went wrong",
    };

    format!("This merge request may not be merged because {}.",
            reason_message)
}

fn make_user_trailer(token: &str, user: &User) -> Trailer {
    Trailer::new(token, format!("{}", user.identity()))
}

fn parse_award_as_trailers(award: Award) -> Option<(Trailer, Option<User>)> {
    let name = &award.name;

    // Handle skin tone color variants as their base version.
    let name = if name[..name.len() - 1].ends_with("_tone") {
        &name[..name.len() - 6]
    } else {
        name
    };

    match name {
        "100" | "clap" | "tada" => {
            Some((make_user_trailer("Acked-by", &award.author), Some(award.author)))
        },
        "no_good" => {
            Some((make_user_trailer("Rejected-by", &award.author), Some(award.author)))
        },
        _ => None,
    }
}

static TRAILER_MARKERS: &'static [(&'static str, &'static [&'static str])] = &[
    ("Acked-by", &[
        "+1",
        ":+1:",
    ]),
    ("Reviewed-by", &[
        "+2",
    ]),
    ("Tested-by", &[
        "+3",
    ]),
    ("Rejected-by", &[
        "-1",
        ":-1:",
    ]),
];

fn parse_comment_for_trailers(service: &HostingService, comment: &Comment)
                              -> Vec<(Trailer, Option<User>)> {
    let explicit_trailers = TrailerRef::extract(&comment.content)
        .into_iter()
        // Transform values based on a some shortcuts like user references and a `me` shortcut.
        .filter_map(|trailer| {
            if !trailer.token.ends_with("-by") {
                // Only `-by` trailers go through the username search.
                Some((trailer.into(), None))
            } else if trailer.value.starts_with('@') {
                // Handle user references.
                service.user_by_id(comment.author.id)
                    // Just drop unknown user references.
                    .ok()
                    .map(|user| {
                        (make_user_trailer(trailer.token, &user),
                         Some(user.clone()))
                    })
            } else if trailer.value == "me" {
                // Handle the special value `me` to mean the comment author.
                Some((make_user_trailer(trailer.token, &comment.author),
                      Some(comment.author.clone())))
            } else {
                // Use the trailer as-is.
                Some((trailer.into(), None))
            }
        });
    // Gather the implicit trailers from things like `+2` lines and the like.
    let implicit_trailers = comment.content
        .lines()
        .filter_map(|l| {
            let line = l.trim();

            TRAILER_MARKERS.iter()
                .filter_map(|&(token, needles)| {
                    needles.iter()
                        .filter_map(|&needle| {
                            if line.starts_with(needle) {
                                Some((make_user_trailer(token, &comment.author),
                                      Some(comment.author.clone())))
                            } else {
                                None
                            }
                        })
                        .next()
                })
                .next()
        });

    explicit_trailers.chain(implicit_trailers)
        .collect()
}
