// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Handlers for jobs for webhooks from hosting services.

extern crate ghostflow;
use self::ghostflow::host::HostingService;

extern crate json_job_dispatch;
use self::json_job_dispatch::Handler;

extern crate serde_json;
use self::serde_json::Value;

pub mod common;
mod gitlab;

use super::config::Host;

use std::error::Error;
use std::rc::Rc;

quick_error! {
    #[derive(Debug)]
    enum HandlerError {
        UnknownApi(api: String) {
            display("unknown api: {}", api)
        }
    }
}

/// Connect to a `HostingService`.
pub fn connect_to_host(api: &str, url: &Option<String>, secrets: Value)
                       -> Result<Rc<HostingService>, Box<Error>> {
    match api {
        "gitlab" => gitlab::connect_to_host(url, secrets),
        _ => Err(Box::new(HandlerError::UnknownApi(api.to_string()))),
    }
}

/// Create handlers for a host configuration.
pub fn create_handler(host: Host, name: &str) -> Result<Box<Handler>, Box<Error>> {
    match host.api.as_str() {
        "gitlab" => gitlab::create_handler(host, name),
        _ => Err(Box::new(HandlerError::UnknownApi(host.api))),
    }
}
