// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate ghostflow;
use self::ghostflow::actions::check;
use self::ghostflow::host::{CheckStatus, CommitStatusState, Error, ErrorKind, HostedComment,
                            HostedCommit, HostedMergeRequest, HostedRepo, HostedUser,
                            HostingService, Result};

extern crate git_workarea;
use self::git_workarea::CommitId;

extern crate gitlab;
use self::gitlab::{MergeRequestState, ProjectId};
use self::gitlab::systemhooks::{ProjectMemberSystemHook, ProjectSystemHook};
use self::gitlab::webhooks::{MergeRequestHookAttrs, NoteHook, NoteHookAttrs, ProjectHookAttrs,
                             PushHook};

use super::super::common::traits::*;

struct GitlabUser {
    id: u64,
    handle: String,
    name: String,
    email: String,
}

impl GitlabUser {
    fn new(user: &HostedUser) -> Self {
        GitlabUser {
            id: user.id(),
            handle: user.handle().to_string(),
            name: user.name().to_string(),
            email: user.email().to_string(),
        }
    }
}

impl HostedUser for GitlabUser {
    fn id(&self) -> u64 {
        self.id
    }

    fn handle(&self) -> &str {
        &self.handle
    }

    fn name(&self) -> &str {
        &self.name
    }

    fn email(&self) -> &str {
        &self.email
    }
}

struct GitlabRepo<'a> {
    project: &'a ProjectHookAttrs,
    id: &'a ProjectId,
}

impl<'a> GitlabRepo<'a> {
    fn new(project: &'a ProjectHookAttrs, id: &'a ProjectId) -> Self {
        GitlabRepo {
            project: project,
            id: id,
        }
    }
}

impl<'a> HostedRepo for GitlabRepo<'a> {
    fn name(&self) -> &str {
        &self.project.path_with_namespace
    }

    fn url(&self) -> &str {
        &self.project.git_ssh_url
    }

    fn id(&self) -> u64 {
        self.id.value()
    }
}

impl<'a> ProjectInfo for GitlabRepo<'a> {
    fn fork_root(&self) -> &str {
        unimplemented!()
    }
}

struct GitlabMergeRequestCommit<'a> {
    repo: GitlabRepo<'a>,
    branch: &'a str,
    id: CommitId,
}

impl<'a> GitlabMergeRequestCommit<'a> {
    fn new(repo: GitlabRepo<'a>, branch: &'a str, id: &str) -> Self {
        GitlabMergeRequestCommit {
            repo: repo,
            branch: branch,
            id: CommitId::new(id),
        }
    }
}

impl<'a> HostedCommit for GitlabMergeRequestCommit<'a> {
    fn project(&self) -> &HostedRepo {
        &self.repo
    }

    fn refname(&self) -> Option<&str> {
        Some(self.branch)
    }

    fn id(&self) -> &CommitId {
        &self.id
    }

    fn _self_ref_hack(&self) -> &HostedCommit {
        self
    }
}

pub struct GitlabMergeRequestInfo<'a> {
    hook: &'a MergeRequestHookAttrs,

    source_repo: GitlabRepo<'a>,
    target_repo: GitlabRepo<'a>,

    url: String,
    author: GitlabUser,
    status: CheckStatus,

    old_commit: Option<GitlabMergeRequestCommit<'a>>,
    new_commit: GitlabMergeRequestCommit<'a>,
}

impl<'a> GitlabMergeRequestInfo<'a> {
    pub fn from_web_hook(service: &HostingService, hook: &'a MergeRequestHookAttrs)
                         -> Result<Self> {
        let last_commit = try!(hook.last_commit
            .as_ref()
            .ok_or_else(|| {
                let err: Error = ErrorKind::Msg("merge request is missing `last_commit`".into())
                    .into();
                err
            }));

        let mr = try!(service.merge_request_by_id(hook.target_project_id.value(), hook.id.value()));
        let new_commit = GitlabMergeRequestCommit::new(GitlabRepo::new(&hook.source,
                                                                       &hook.source_project_id),
                                                       &hook.source_branch,
                                                       last_commit.id.value());
        let status = try!(service.get_commit_statuses(&new_commit))
            .iter()
            // Make sure that the status was posted by the robot.
            .filter(|status| status.author().id() == service.service_user().id())
            // Only look at check result statuses.
            .find(|status| status.name() == check::STATUS_NAME)
            .map(|status| {
                match *status.state() {
                    CommitStatusState::Success => CheckStatus::Pass,
                    CommitStatusState::Failed => CheckStatus::Fail,
                    CommitStatusState::Pending |
                    CommitStatusState::Running => CheckStatus::Unchecked,
                }
            })
            .unwrap_or(CheckStatus::Unchecked);

        Ok(GitlabMergeRequestInfo {
            hook: hook,

            source_repo: GitlabRepo::new(&hook.source, &hook.source_project_id),
            target_repo: GitlabRepo::new(&hook.target, &hook.target_project_id),

            url: mr.url().to_string(),
            author: GitlabUser::new(mr.author()),
            status: status,

            old_commit: hook.oldrev
                .as_ref()
                .map(|rev| {
                    GitlabMergeRequestCommit::new(GitlabRepo::new(&hook.source,
                                                                  &hook.source_project_id),
                                                  &hook.source_branch,
                                                  rev.value())
                }),
            new_commit: new_commit,
        })
    }
}

impl<'a> HostedMergeRequest for GitlabMergeRequestInfo<'a> {
    fn source_repo(&self) -> &HostedRepo {
        &self.source_repo
    }

    fn source_branch(&self) -> &str {
        &self.hook.source_branch
    }

    fn target_repo(&self) -> &HostedRepo {
        &self.target_repo
    }

    fn target_branch(&self) -> &str {
        &self.hook.target_branch
    }

    fn id(&self) -> u64 {
        self.hook.id.value()
    }

    fn url(&self) -> &str {
        &self.url
    }

    fn work_in_progress(&self) -> bool {
        self.hook.work_in_progress
    }

    fn description(&self) -> &str {
        self.hook
            .description
            .as_ref()
            .map(|s| s.as_str())
            .unwrap_or("")
    }

    fn old_commit(&self) -> Option<&HostedCommit> {
        self.old_commit
            .as_ref()
            .map(|commit| commit as &HostedCommit)
    }

    fn commit(&self) -> &HostedCommit {
        &self.new_commit
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn reference(&self) -> String {
        format!("!{}", self.hook.iid)
    }

    fn check_status(&self) -> &CheckStatus {
        &self.status
    }

    fn remove_source_branch(&self) -> bool {
        self.hook.merge_params.force_remove_source_branch()
    }
}

impl<'a> MergeRequestInfo for GitlabMergeRequestInfo<'a> {
    fn as_hosted(&self) -> &HostedMergeRequest {
        self
    }

    fn is_open(&self) -> bool {
        match self.hook.state {
            MergeRequestState::Opened |
            MergeRequestState::Reopened => true,
            MergeRequestState::Closed |
            MergeRequestState::Merged |
            MergeRequestState::Locked => false,
        }
    }

    fn date(&self) -> &DateTime<UTC> {
        self.hook.updated_at.as_ref()
    }
}

struct GitlabNote<'a> {
    hook: &'a NoteHookAttrs,
    author: GitlabUser,
}

impl<'a> GitlabNote<'a> {
    fn new(service: &HostingService, hook: &'a NoteHookAttrs) -> Result<Self> {
        let user = try!(service.user_by_id(hook.author_id.value()));

        Ok(GitlabNote {
            hook: hook,
            author: GitlabUser::new(user.as_ref()),
        })
    }
}

impl<'a> HostedComment for GitlabNote<'a> {
    fn id(&self) -> u64 {
        self.hook.id.value()
    }

    fn is_system(&self) -> bool {
        self.hook.system
    }

    fn is_branch_update(&self) -> bool {
        false
    }

    fn created_at(&self) -> &DateTime<UTC> {
        self.hook.created_at.as_ref()
    }

    fn updated_at(&self) -> &DateTime<UTC> {
        self.hook.updated_at.as_ref()
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn content(&self) -> &str {
        &self.hook.note
    }
}

pub struct GitlabMergeRequestNoteInfo<'a> {
    merge_request: GitlabMergeRequestInfo<'a>,
    note: GitlabNote<'a>,
}

impl<'a> GitlabMergeRequestNoteInfo<'a> {
    pub fn from_web_hook(service: &HostingService, hook: &'a NoteHook) -> Result<Self> {
        let merge_request = try!(hook.merge_request
            .as_ref()
            .ok_or_else(|| {
                let err: Error = ErrorKind::Msg("merge request is missing from note".into()).into();
                err
            }));
        let note = try!(GitlabNote::new(service, &hook.object_attributes));

        GitlabMergeRequestInfo::from_web_hook(service, merge_request).map(|mr_info| {
            GitlabMergeRequestNoteInfo {
                merge_request: mr_info,
                note: note,
            }
        })
    }
}

impl<'a> MergeRequestNoteInfo for GitlabMergeRequestNoteInfo<'a> {
    fn merge_request(&self) -> &MergeRequestInfo {
        &self.merge_request
    }

    fn note(&self) -> &HostedComment {
        &self.note
    }
}

pub struct GitlabPushInfo<'a> {
    hook: &'a PushHook,
    project: GitlabRepo<'a>,
    commit: CommitId,
    author: GitlabUser,
    date: DateTime<UTC>,
}

impl<'a> GitlabPushInfo<'a> {
    pub fn from_web_hook(service: &HostingService, hook: &'a PushHook) -> Result<Self> {
        let user = try!(service.user_by_id(hook.user_id.value()));
        let date = if let Some(commit) = hook.commits.last() {
            commit.timestamp
        } else {
            UTC::now()
        };

        Ok(GitlabPushInfo {
            hook: hook,
            project: GitlabRepo::new(&hook.project, &hook.project_id),
            commit: CommitId::new(hook.after.value()),
            author: GitlabUser::new(user.as_ref()),
            date: date,
        })
    }
}

impl<'a> HostedCommit for GitlabPushInfo<'a> {
    fn project(&self) -> &HostedRepo {
        &self.project
    }

    fn refname(&self) -> Option<&str> {
        Some(&self.hook.ref_)
    }

    fn id(&self) -> &CommitId {
        &self.commit
    }

    fn _self_ref_hack(&self) -> &HostedCommit {
        self
    }
}

impl<'a> PushInfo for GitlabPushInfo<'a> {
    fn project_info(&self) -> &ProjectInfo {
        &self.project
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn date(&self) -> &DateTime<UTC> {
        &self.date
    }
}

pub struct GitlabProjectInfo<'a> {
    path_with_namespace: &'a str,
    project_id: &'a ProjectId,
    url: String,
}

impl<'a> GitlabProjectInfo<'a> {
    pub fn from_create_hook(service: &HostingService, hook: &'a ProjectSystemHook) -> Result<Self> {
        Self::from_info(service, &hook.path_with_namespace, &hook.project_id)
    }

    pub fn from_membership_hook(service: &HostingService, hook: &'a ProjectMemberSystemHook)
                                -> Result<Self> {
        Self::from_info(service, &hook.project_path_with_namespace, &hook.project_id)
    }

    fn from_info(service: &HostingService, path: &'a str, id: &'a ProjectId) -> Result<Self> {
        let project = try!(service.repo_by_id(id.value()));

        Ok(GitlabProjectInfo {
            path_with_namespace: path,
            project_id: id,
            url: project.url().to_string(),
        })
    }
}

impl<'a> HostedRepo for GitlabProjectInfo<'a> {
    fn name(&self) -> &str {
        self.path_with_namespace
    }

    fn url(&self) -> &str {
        &self.url
    }

    fn id(&self) -> u64 {
        self.project_id.value()
    }
}

impl<'a> ProjectInfo for GitlabProjectInfo<'a> {
    fn fork_root(&self) -> &str {
        unimplemented!()
    }
}
