// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate ghostflow;
use self::ghostflow::host::{GitlabService, HostingService};

extern crate gitlab;
use self::gitlab::Gitlab;
use self::gitlab::systemhooks::{GroupMemberSystemHook, ProjectMemberSystemHook, ProjectSystemHook};
use self::gitlab::types::NoteType;
use self::gitlab::webhooks::{MergeRequestHook, NoteHook, PushHook};

extern crate json_job_dispatch;
use self::json_job_dispatch::{Director, Handler, HandlerResult};
use self::json_job_dispatch::Error as HandlerError;

extern crate serde;
use self::serde::Deserialize;

extern crate serde_json;
use self::serde_json::Value;

use super::super::super::config::Host;
use super::super::common::handlers::*;
use super::super::common::jobs::{BatchBranchJob, TagStage};
use super::traits::*;

use std::rc::Rc;
use std::error::Error;

pub fn connect_to_host(url: &Option<String>, secrets: Value)
                       -> Result<Rc<HostingService>, Box<Error>> {
    let host = url.as_ref().map_or("gitlab.com", |u| u.as_str());
    let token = match secrets.pointer("/token").and_then(|t| t.as_str()) {
        Some(t) => t,
        None => return Err("gitlab requires a token to be provided".into()),
    };
    let use_ssl = secrets.pointer("/use_ssl").and_then(|t| t.as_bool()).unwrap_or(true);

    let ctor = if use_ssl {
        Gitlab::new
    } else {
        Gitlab::new_insecure
    };

    let gitlab = try!(ctor(host, token));
    Ok(try!(GitlabService::new(gitlab).map(|service| Rc::new(service) as Rc<HostingService>)))
}

struct GitlabHandler {
    host: Host,
    name: String,
}

impl GitlabHandler {
    fn new(host: Host, name: &str) -> Self {
        GitlabHandler {
            host: host,
            name: name.to_string(),
        }
    }

    fn parse_object<F, T>(object: &Value, callback: F) -> Result<HandlerResult, HandlerError>
        where T: Deserialize,
              F: Fn(T) -> Result<HandlerResult, HandlerError>,
    {
        match serde_json::from_value::<T>(object.clone()) {
            Ok(hook) => callback(hook),
            Err(err) => Ok(HandlerResult::Fail(Box::new(err))),
        }
    }

    fn handle_kind(&self, kind: &str, object: &Value) -> Result<HandlerResult, HandlerError> {
        match kind {
            "merge_request" => {
                Self::parse_object(object, |hook: MergeRequestHook| {
                    GitlabMergeRequestInfo::from_web_hook(self.host.service.as_ref(),
                                                          &hook.object_attributes)
                        .map(|mr| handle_merge_request_update(object, &self.host, mr))
                        .unwrap_or_else(|err| Ok(HandlerResult::Fail(Box::new(err))))
                })
            },
            "note" => {
                Self::parse_object(object, |hook: NoteHook| {
                    let note_type = hook.object_attributes.noteable_type;
                    match note_type {
                        NoteType::MergeRequest => {
                            GitlabMergeRequestNoteInfo::from_web_hook(self.host.service.as_ref(),
                                                                      &hook)
                                .map(|note| handle_merge_request_note(object, &self.host, note))
                                .unwrap_or_else(|err| Ok(HandlerResult::Fail(Box::new(err))))
                        },
                        _ => {
                            Ok(HandlerResult::Reject(format!("unhandled noteable type: {}",
                                                             note_type.as_str())))
                        },
                    }
                })
            },
            "push" => {
                Self::parse_object(object, |hook: PushHook| {
                    GitlabPushInfo::from_web_hook(self.host.service.as_ref(), &hook)
                        .map(|push| handle_push(object, &self.host, push))
                        .unwrap_or_else(|err| Ok(HandlerResult::Fail(Box::new(err))))
                })
            },
            "project_create" => {
                Self::parse_object(object, |hook: ProjectSystemHook| {
                    GitlabProjectInfo::from_create_hook(self.host.service.as_ref(), &hook)
                        .map(|project| handle_project_creation(object, &self.host, project))
                        .unwrap_or_else(|err| Ok(HandlerResult::Fail(Box::new(err))))
                })
            },
            "user_add_to_team" |
            "user_remove_from_team" => {
                Self::parse_object(object, |hook: ProjectMemberSystemHook| {
                    GitlabProjectInfo::from_membership_hook(self.host.service.as_ref(), &hook)
                        .map(|project| {
                            handle_project_membership_refresh(object, &self.host, project)
                        })
                        .unwrap_or_else(|err| Ok(HandlerResult::Fail(Box::new(err))))
                })
            },
            "user_add_to_group" |
            "user_remove_from_group" => {
                Self::parse_object(object, |hook: GroupMemberSystemHook| {
                    handle_group_membership_refresh(object, &self.host, &hook.group_name)
                })
            },
            "tag_stage" => {
                Self::parse_object(object, |data: BatchBranchJob<TagStage>| {
                    Ok(handle_stage_tag(object, &self.host, data))
                })
            },
            _ => Ok(HandlerResult::Reject(format!("unhandled kind: {}", kind))),
        }
    }
}

impl Handler for GitlabHandler {
    fn add_to_director<'a>(&'a self, director: &mut Director<'a>) -> Result<(), HandlerError> {
        let mut add_handler = |kind| director.add_handler(&format!("{}:{}", self.name, kind), self);

        try!(add_handler("merge_request"));
        try!(add_handler("note"));
        try!(add_handler("push"));
        try!(add_handler("project_create"));
        try!(add_handler("user_add_to_team"));
        try!(add_handler("user_remove_from_team"));
        try!(add_handler("user_add_to_group"));
        try!(add_handler("user_remove_from_group"));

        try!(add_handler("tag_stage"));

        Ok(())
    }

    fn handle(&self, kind: &str, object: &Value) -> Result<HandlerResult, HandlerError> {
        let mut split = kind.split(':');

        if let Some(level) = split.next() {
            if level != self.name {
                return Ok(HandlerResult::Reject(format!("handler mismatch: {}", level)));
            }
        } else {
            return Ok(HandlerResult::Reject("handler mismatch".to_string()));
        }

        if let Some(kind) = split.next() {
            self.handle_kind(kind, object)
        } else {
            Ok(HandlerResult::Reject("missing kind".to_string()))
        }
    }
}

pub fn create_handler(host: Host, name: &str) -> Result<Box<Handler>, Box<Error>> {
    Ok(Box::new(GitlabHandler::new(host, name)))
}
