// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Core handler traits for kitware workflow actions.
//!
//! This module contains traits necessary to carry out the actual actions wanted for the requested
//! branches.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate ghostflow;
use self::ghostflow::host::{HostedComment, HostedCommit, HostedMembership, HostedMergeRequest,
                            HostedRepo, HostedUser};

/// Trait representing information required to handle a push event.
pub trait ProjectInfo: HostedRepo {
    /// The root of the project's fork tree.
    fn fork_root(&self) -> &str;
}

/// Trait representing information required to handle a push event.
pub trait PushInfo: HostedCommit {
    /// The project pushed into.
    fn project_info(&self) -> &ProjectInfo;
    /// The user who pushed the commit.
    fn author(&self) -> &HostedUser;
    /// When the push occurred.
    fn date(&self) -> &DateTime<UTC>;
}

/// Trait representing information required to handle a merge request update.
pub trait MergeRequestInfo: HostedMergeRequest {
    /// A `HostedMergeRequest` for the merge request.
    fn as_hosted(&self) -> &HostedMergeRequest;

    /// Whether the merge request is open or not.
    fn is_open(&self) -> bool;

    /// When the merge request was last updated.
    fn date(&self) -> &DateTime<UTC>;
}

/// Trait representing information required to handle a note on a merge request.
pub trait MergeRequestNoteInfo {
    /// The merge request.
    fn merge_request(&self) -> &MergeRequestInfo;
    /// The note.
    fn note(&self) -> &HostedComment;
}

/// Trait representing information required to handle the addition of a user to a project.
pub trait MembershipAdditionInfo {
    /// The project.
    fn project(&self) -> &ProjectInfo;

    /// The membership.
    fn into_membership(&self) -> Box<HostedMembership>;
}

/// Trait representing information required to handle the removal of a user from a project.
pub trait MembershipRemovalInfo {
    /// The project.
    fn project(&self) -> &ProjectInfo;

    /// The user.
    fn user(&self) -> &HostedUser;
}
