// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Core handler logic for kitware workflow actions.
//!
//! This module contains functions which carry out the actual actions wanted for the requested
//! branches.

extern crate ghostflow;
use self::ghostflow::actions::check::{self, CheckStatus};
use self::ghostflow::actions::merge::MergeActionResult;
use self::ghostflow::host::HostedMergeRequest;
use self::ghostflow::host::Result as HostResult;
use self::ghostflow::utils::TrailerRef;

extern crate itertools;
use self::itertools::Itertools;

extern crate json_job_dispatch;
use self::json_job_dispatch::{Error, HandlerResult};

extern crate serde_json;
use self::serde_json::Value;

use super::commands::Commands;
use super::jobs::{BatchBranchJob, TagStage};
use super::traits::{MembershipAdditionInfo, MembershipRemovalInfo, MergeRequestInfo,
                    MergeRequestNoteInfo, ProjectInfo, PushInfo};
use super::super::super::config::{Branch, Host, Project, StageUpdatePolicy};

macro_rules! try_action {
    ( $action:expr ) => {
        match $action {
            Ok(res) => res,
            Err(handler_result) => return Ok(handler_result),
        }
    }
}

fn get_project<'a>(host: &'a Host, project_name: &str) -> Result<&'a Project, HandlerResult> {
    host.projects
        .get(project_name)
        .ok_or_else(|| HandlerResult::Reject(format!("unwatched project {}", project_name)))
}

fn get_branch<'a>(host: &'a Host, project_name: &str, branch_name: &str)
                  -> Result<(&'a Project, &'a Branch), HandlerResult> {
    get_project(host, project_name).and_then(|project| {
        project.branches
            .get(branch_name)
            .map(|branch| (project, branch))
            .ok_or_else(|| {
                HandlerResult::Reject(format!("unwatched branch {} for project {}",
                                              branch_name,
                                              project_name))
            })
    })
}

fn handle_result<F>(notes: Vec<String>, defer: bool, post_comment: F) -> HandlerResult
    where F: Fn(&str) -> HostResult<()>,
{
    let txt_comment = notes.join("\n");

    if defer {
        info!(target: "ghostflow-director/handler",
              "Deferring a job: {}",
              txt_comment);

        HandlerResult::Defer(txt_comment)
    } else if notes.is_empty() {
        HandlerResult::Accept
    } else {
        let md_comment = notes.join("  \n");

        if let Err(err) = post_comment(&md_comment) {
            error!(target: "ghostflow-director/handler",
                   "Failed to post comment:\n'{}'\n'{}'.",
                   md_comment, err);
        }

        HandlerResult::Reject(txt_comment)
    }
}

/// Handle a push to a repository.
pub fn handle_push<P: PushInfo>(_: &Value, host: &Host, push: P) -> Result<HandlerResult, Error> {
    let refs_heads_prefix = "refs/heads/";
    let refname = if let Some(name) = push.refname() {
        if !name.starts_with(refs_heads_prefix) {
            return Ok(HandlerResult::Reject(format!("non-branch ref push {}", name)));
        } else {
            name
        }
    } else {
        return Ok(HandlerResult::Reject("unknown refname".to_string()));
    };

    let branch = &refname[refs_heads_prefix.len()..];
    let (project, branch) = try_action!(get_branch(host, push.project().name(), branch));

    let mut comment_notes = vec![];

    let fetch_res = project.context.fetch_into("origin", format!("+{}", refname), refname);
    if let Err(err) = fetch_res {
        comment_notes.push(format!("Failed to fetch from the repository: {:?}.", err));
    }

    if let Some(stage_action) = branch.stage() {
        let mut stage = stage_action.stage();
        let res = stage.base_branch_update(&push, &push.author().identity(), push.date());

        if let Err(err) = res {
            comment_notes.push(format!("Error occurred when updating the base branch ({}): \
                                        `{:?}`",
                                       host.maintainers.join(" "),
                                       err));
        }
    }

    Ok(handle_result(comment_notes,
                     false,
                     |comment| host.service.post_commit_comment(&push, comment)))
}

/// Handle an update to a merge request.
pub fn handle_merge_request_update<M: MergeRequestInfo>(_: &Value, host: &Host, mr: M)
                                                        -> Result<HandlerResult, Error> {
    let (project, branch) =
        try_action!(get_branch(host, mr.target_repo().name(), mr.target_branch()));

    let mut comment_notes = vec![];

    if !mr.is_open() {
        if let Some(stage_action) = branch.stage() {
            let res = stage_action.stage()
                .unstage_update_merge_request(&mr, "because it has been closed");

            if let Err(err) = res {
                comment_notes.push(format!("Error occurred when unstaging a closed merge request \
                                            ({}): `{:?}`",
                                           host.maintainers.join(" "),
                                           err));
                handle_result(comment_notes,
                              false,
                              |comment| host.service.post_mr_comment(&mr, comment));
            }
        }

        return Ok(HandlerResult::Reject(format!("the merge request for the branch {} has been \
                                                 closed",
                                                mr.source_branch())));
    }

    // TODO: If the MR has been edited to change the target branch, we may need to remove it from
    // other branches, but we don't have the old branch name :/ .

    let source_project = mr.source_repo();
    let fetch_res = project.context.fetch(source_project.url(), &[mr.source_branch()]);
    if let Err(err) = fetch_res {
        comment_notes.push(format!("Failed to fetch from the repository: {:?}.", err));
    }

    let reserve_res = project.context
        .reserve_ref(&format!("mr/{}", mr.id()), mr.commit().id());
    if let Err(err) = reserve_res {
        comment_notes.push(format!("Failed to reserve ref {} for the merge request: {:?}",
                                   mr.commit().id(),
                                   err));
    }

    let is_ok = if mr.check_status().is_ok() {
        true
    } else {
        match check_mr(project, branch, mr.as_hosted()) {
            Ok(b) => b,
            Err(err) => {
                comment_notes.push(format!("Failed to run the checks: {:?}", err));

                false
            },
        }
    };
    let is_wip = mr.work_in_progress();

    if let Some(stage_action) = branch.stage() {
        let mut stage = stage_action.stage();

        let (policy, reason) = if !is_ok {
            (StageUpdatePolicy::Unstage, "since it is failing its checks")
        } else if is_wip {
            (StageUpdatePolicy::Unstage, "since it is marked as work-in-progress")
        } else if stage.stager().find_topic_by_id(mr.id()).is_none() {
            (StageUpdatePolicy::Ignore, "it is not on the stage currently")
        } else {
            (stage_action.policy, "as per policy")
        };

        let (what, res) = match policy {
            StageUpdatePolicy::Ignore => ("ignore", Ok(())),
            StageUpdatePolicy::Restage => {
                ("restage", stage.stage_merge_request(&mr, &mr.author().identity(), mr.date()))
            },
            StageUpdatePolicy::Unstage => {
                ("unstage", stage.unstage_update_merge_request(&mr, reason))
            },
        };

        if let Err(err) = res {
            comment_notes.push(format!("Failed to {}: `{:?}`", what, err));
        }
    }

    Ok(handle_result(comment_notes,
                     false,
                     |comment| host.service.post_mr_comment(&mr, comment)))
}

fn check_mr(project: &Project, branch: &Branch, mr: &HostedMergeRequest)
            -> Result<bool, check::Error> {
    project.check_mr(branch, mr)
        .map(|status| {
            if let CheckStatus::Pass = status {
                true
            } else {
                false
            }
        })
}

fn handle_commands(host: &Host, commands: &Commands, project: &Project, branch: &Branch,
                   mr_note: &MergeRequestNoteInfo, notes: &mut Vec<String>)
                   -> bool {
    let mr = mr_note.merge_request();
    let note = mr_note.note();

    if commands.check.requested() {
        match check_mr(project, branch, mr.as_hosted()) {
            Ok(true) => (),
            Ok(false) => return false,
            Err(err) => {
                notes.push(format!("failed to run checks: {:?}", err));
            },
        }
    }

    if commands.stage.requested() {
        if let Some(stage_action) = branch.stage() {
            let mut stage = stage_action.stage();

            if !mr.check_status().is_ok() {
                notes.push("Refusing to stage; topic is failing the checks.".to_string())
            } else if commands.stage.arguments().is_empty() {
                let res = stage.stage_merge_request(mr.as_hosted(),
                                                    &note.author().identity(),
                                                    note.created_at());

                if let Err(err) = res {
                    notes.push(format!("Error occurred during stage action ({}): `{:?}`",
                                       host.maintainers.join(" "),
                                       err));
                }
            } else {
                notes.push(format!("Unrecognized `stage` arguments: `{}`.",
                                   commands.stage.arguments().iter().join("`, `")));
            }
        }
    }

    if commands.unstage.requested() {
        if let Some(stage_action) = branch.stage() {
            let mut stage = stage_action.stage();

            if commands.stage.arguments().is_empty() {
                let res = stage.unstage_merge_request(mr.as_hosted());

                if let Err(err) = res {
                    notes.push(format!("Error occurred during unstage action ({}): `{:?}`",
                                       host.maintainers.join(" "),
                                       err));
                }
            } else {
                notes.push(format!("Unrecognized `unstage` arguments: `{}`.",
                                   commands.stage.arguments().iter().join("`, `")));
            }
        }
    }

    if commands.merge.requested() {
        if let Some(merge_action) = branch.merge() {
            if !mr.check_status().is_ok() {
                notes.push("Refusing to merge; topic is failing the checks.".to_string())
            } else if commands.merge.arguments().is_empty() {
                let merge = merge_action.merge();

                let res =
                    merge.merge_mr(mr.as_hosted(), &note.author().identity(), note.created_at());

                match res {
                    Ok(MergeActionResult::Success) |
                    Ok(MergeActionResult::Failed) => (),
                    // Defer the action if merging failed.
                    Ok(MergeActionResult::PushFailed) => {
                        notes.push("Failed to push the resulting merge".to_string());
                        return true;
                    },
                    Err(err) => {
                        notes.push(format!("Error occurred during merge action ({}): `{:?}`",
                                           host.maintainers.join(" "),
                                           err));
                    },
                }
            } else {
                notes.push(format!("Unrecognized `merge` arguments: `{}`.",
                                   commands.merge.arguments().iter().join("`, `")));
            }
        }
    }

    false
}

/// Handle a note on a merge request.
pub fn handle_merge_request_note<M: MergeRequestNoteInfo>(_: &Value, host: &Host, mr_note: M)
                                                          -> Result<HandlerResult, Error> {
    let mr = mr_note.merge_request();
    let note = mr_note.note();

    if !mr.is_open() {
        return Ok(HandlerResult::Reject(format!("the merge request for the branch {} has been \
                                                 closed",
                                                mr.source_branch())));
    }

    let already_handled = match host.service.get_mr_comment_awards(mr.as_hosted(), note) {
        Ok(awards) => {
            awards.into_iter()
                .any(|award| {
                    award.name() == "robot" &&
                    award.author().id() == host.service.service_user().id()
                })
        },
        Err(err) => {
            error!(target: "ghostflow-director/handler",
                   "failed to get the awards for a note at {}: {:?}",
                   mr.url(),
                   err);

            false
        },
    };

    if already_handled {
        return Ok(HandlerResult::Reject("this note has already been handled".to_string()));
    }

    let (project, branch) =
        try_action!(get_branch(host, mr.target_repo().name(), mr.target_branch()));

    let mut commands = Commands::new(branch);

    let trailers = TrailerRef::extract(mr_note.note().content());

    if trailers.is_empty() {
        return Ok(HandlerResult::Reject("no trailers present".to_string()));
    }

    let access = project.access_level(note.author());
    let is_submitter = mr.author().id() == note.author().id();
    commands.add_from_trailers(access, is_submitter, &trailers);

    if commands.is_empty() {
        return Ok(HandlerResult::Reject("no commands present".to_string()));
    }

    let mut comment_notes = vec![];

    let errors = commands.error_messages(note.author());
    let defer = if errors.is_empty() {
        handle_commands(host,
                        &commands,
                        project,
                        branch,
                        &mr_note,
                        &mut comment_notes)
    } else {
        comment_notes.extend(errors);

        false
    };

    // Award the command comment to indicate that we handled it.
    if let Err(err) = host.service.award_mr_comment(mr.as_hosted(), note, "robot") {
        comment_notes.push(format!("Failed to mark the comment as seen: {:?}.", err));
    }

    Ok(handle_result(comment_notes,
                     defer,
                     |comment| host.service.post_mr_comment(mr.as_hosted(), comment)))
}

/// Handle a user being added to a project.
pub fn handle_user_addition<M: MembershipAdditionInfo>(_: &Value, host: &Host, membership: M)
                                                       -> Result<HandlerResult, Error> {
    let project = try_action!(get_project(host, membership.project().name()));

    project.add_member(membership.into_membership());

    Ok(HandlerResult::Accept)
}

/// Handle a user being added to a project.
pub fn handle_user_removal<M: MembershipRemovalInfo>(_: &Value, host: &Host, membership: M)
                                                     -> Result<HandlerResult, Error> {
    let project = try_action!(get_project(host, membership.project().name()));

    project.remove_member(membership.user());

    Ok(HandlerResult::Accept)
}

/// Handle a project's membership needing completely refreshed.
pub fn handle_project_membership_refresh<P: ProjectInfo>(_: &Value, host: &Host, project_info: P)
                                                         -> Result<HandlerResult, Error> {
    let project = try_action!(get_project(host, project_info.name()));

    project.refresh_membership(project_info.name())
        .map(|_| HandlerResult::Accept)
        .or_else(|err| {
            Ok(HandlerResult::Reject(format!("failed to update memberships for {}: {:?}",
                                             project_info.name(),
                                             err)))
        })
}

/// Handle a group's membership needing completely refreshed.
pub fn handle_group_membership_refresh(_: &Value, host: &Host, group: &str)
                                       -> Result<HandlerResult, Error> {
    let prefix = format!("{}/", group);

    let (succeeded, failed) = host.projects
        .iter()
        .map(|(name, project)| {
            if name.starts_with(&prefix) {
                project.refresh_membership(name)
            } else {
                Ok(())
            }
        })
        .fold((0, 0), |(succeeded, failed), res| {
            match res {
                Ok(_) => (succeeded + 1, failed),
                Err(_) => (succeeded, failed + 1),
            }
        });

    Ok(if succeeded == 0 {
        HandlerResult::Accept
    } else {
        HandlerResult::Reject(format!("failed to update {} repository memberships ({} succeeded)",
                                      failed,
                                      succeeded))
    })
}

/// Handle the creation of a project on the service.
pub fn handle_project_creation<P: ProjectInfo>(_: &Value, host: &Host, project: P)
                                               -> Result<HandlerResult, Error> {
    let hook_result = host.webhook_url
        .as_ref()
        .map(|url| {
            host.service
                .add_hook(project.name(), url)
                .map(|_| HandlerResult::Accept)
                .or_else(|err| {
                    Ok(HandlerResult::Reject(format!("failed to add webhook for {}: {:?}",
                                                     project.name(),
                                                     err)))
                })
        });

    hook_result.unwrap_or_else(|| Ok(HandlerResult::Reject("nothing to do".to_string())))
}

/// Handle requests to tag staging branches.
pub fn handle_stage_tag(_: &Value, host: &Host, data: BatchBranchJob<TagStage>) -> HandlerResult {
    data.into_iter_branch()
        .map(|(project_name, branch_name, tag_stage)| {
            get_branch(host, &project_name, &branch_name).map(|(_, branch)| {
                (project_name,
                 branch_name,
                 branch.stage()
                     .map(|stage| {
                         stage.stage().tag_stage(&tag_stage.reason,
                                                 &tag_stage.ref_date_format,
                                                 tag_stage.policy.into())
                     }))
            })
        })
        .fold(HandlerResult::Accept, |result, branch_result| {
            let branch_handler_result = match branch_result {
                Ok((project, branch, stage_result)) => {
                    match stage_result {
                        Some(Ok(())) => HandlerResult::Accept,
                        Some(Err(stage_err)) => {
                            HandlerResult::Reject(format!("stage error on {}/{}: {:?}",
                                                          project,
                                                          branch,
                                                          stage_err))
                        },
                        None => {
                            HandlerResult::Reject(format!("no stage for {}/{}", project, branch))
                        },
                    }
                },
                Err(err) => err,
            };

            result.combine(branch_handler_result)
        })
}
