// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::UTC;

extern crate ghostflow;
use self::ghostflow::actions::{check, clone, merge, stage};
use self::ghostflow::host::{HostedMembership, HostedMergeRequest, HostedProject, HostedUser,
                            HostingService};
use self::ghostflow::host::Result as HostResult;
use self::ghostflow::utils::Trailer;

extern crate git_checks;
use self::git_checks::{BranchCheck, Check, GitCheckConfiguration};

extern crate git_topic_stage;
use self::git_topic_stage::Stager;

extern crate git_workarea;
use self::git_workarea::{CommitId, GitContext, Identity};

extern crate itertools;
use self::itertools::Itertools;

extern crate serde_json;
use self::serde_json::Value;

use super::checks;
use super::io;

use std::cell::{RefCell, RefMut};
use std::collections::hash_map::HashMap;
use std::collections::hash_set::HashSet;
use std::error::Error;
use std::path::{Path, PathBuf};
use std::rc::Rc;

pub use super::io::StageUpdatePolicy;

/// Main configuration object for the robot.
pub struct Config {
    /// The archive directory.
    pub archive_dir: PathBuf,
    /// The directory to watch for new jobs.
    pub queue_dir: PathBuf,
    /// Host configuration.
    pub hosts: HashMap<String, Host>,
}

impl Config {
    /// Read the configuration from a path.
    ///
    /// The `connect_to_host` function constructs `HostingService` instances given an API and a URL
    /// to communicate with.
    pub fn from_path<P, F>(path: P, connect_to_host: F) -> Result<Self, Box<Error>>
        where P: AsRef<Path>,
              F: Fn(&str, &Option<String>, Value) -> Result<Rc<HostingService>, Box<Error>>,
    {
        io::Config::from_path(path).and_then(|config| {
            let archive_dir = config.archive_dir().to_path_buf();
            let queue_dir = config.queue_dir().to_path_buf();
            let workdir = PathBuf::from(config.workdir);
            let hosts = try!(config.hosts
                .into_iter()
                .map(|(name, host)| {
                    let host_workdir = workdir.join(&name);
                    let secrets = try!(host.secrets());
                    let service = try!(connect_to_host(&host.host_api, &host.host_url, secrets));
                    let host = try!(Host::new(host, service, host_workdir));

                    Ok((name, host))
                })
                .collect::<Result<HashMap<_, _>, Box<Error>>>());

            Ok(Config {
                archive_dir: archive_dir,
                queue_dir: queue_dir,
                hosts: hosts,
            })
        })
    }
}

/// Configuration for a host.
pub struct Host {
    /// The API used to communicate with the host.
    pub api: String,
    /// The `HostingService` instance to communicate with the host.
    pub service: Rc<HostingService>,
    /// The users to notify when errors occur with the workflow.
    pub maintainers: Vec<String>,
    /// Projects which have workflow actions configured.
    pub projects: HashMap<String, Project>,
    /// A directory which may be used for on-disk scratchspace.
    pub workdir: PathBuf,
    /// The webhook URL to register for new projects.
    pub webhook_url: Option<String>,
}

quick_error! {
    #[derive(Debug)]
    enum HostError {
        NoMaintainers {
            display("no maintainers given")
        }
    }
}

impl Host {
    fn new(host: io::Host, service: Rc<HostingService>, workdir: PathBuf)
           -> Result<Self, Box<Error>> {
        if host.maintainers.is_empty() {
            return Err(Box::new(HostError::NoMaintainers));
        }

        let project_workdir = workdir.join("projects");
        let maintainers = host.maintainers.clone();
        let projects = try!(host.projects
            .into_iter()
            .map(|(name, project)| {
                let project = try!(Project::new(&name,
                                                project,
                                                service.clone(),
                                                &project_workdir,
                                                &maintainers));

                Ok((name, project))
            })
            .collect::<Result<HashMap<_, _>, Box<Error>>>());

        Ok(Host {
            api: host.host_api,
            service: service,
            maintainers: host.maintainers,
            projects: projects,
            workdir: workdir,
            webhook_url: host.webhook_url,
        })
    }
}

/// Configuration for a project.
pub struct Project {
    /// The `HostingService` instance to communicate with the host.
    pub service: Rc<HostingService>,
    /// A context for working with the git repository of the project.
    pub context: GitContext,
    /// Branches which have workflow actions configured.
    pub branches: HashMap<String, Branch>,
    /// Users who have non-anonymous access to the project.
    members: RefCell<HashMap<u64, Box<HostedMembership>>>,
    /// Users who should be notified on important events.
    maintainers: Vec<String>,
}

impl Project {
    fn new(name: &str, project: io::Project, service: Rc<HostingService>, workdir: &Path,
           admins: &[String])
           -> Result<Self, Box<Error>> {
        let hosted_project = HostedProject {
            name: name.to_string(),
            service: service.clone(),
        };
        let members = Self::member_map(try!(hosted_project.members()));
        let mut clone = clone::Clone_::new(workdir, hosted_project);
        project.submodules
            .iter()
            .foreach(|(name, path)| {
                let link = clone::CloneSubmoduleLink::new(path);
                clone.with_submodule(name, link);
            });
        let context = try!(clone.clone_watched_repo());
        let identity = Identity::new(project.name.clone(), project.email.clone());
        let project_checks = project.checks.clone();
        let maintainers = project.maintainers
            .iter()
            .cloned()
            .chain(admins.iter().cloned())
            .collect();

        let branches = try!(project.branches
            .into_iter()
            .map(|(branch_name, branch)| {
                let checks = try!(Checks::new(branch.checks.iter().chain(project_checks.iter())));
                let merge_checks =
                    try!(Checks::new(branch.merge_checks.iter().chain(project_checks.iter())));

                let mut cbranch = try!(Branch::new(&branch_name,
                                                   checks,
                                                   merge_checks,
                                                   context.clone(),
                                                   identity.clone()));

                if let Some(ref merge) = branch.merge {
                    let hosted_project = HostedProject {
                        name: name.to_string(),
                        service: service.clone(),
                    };

                    try!(cbranch.add_merge(&merge, hosted_project));
                }

                if let Some(ref stage) = branch.stage {
                    let hosted_project = HostedProject {
                        name: name.to_string(),
                        service: service.clone(),
                    };

                    try!(cbranch.add_stage(&stage, hosted_project));
                }

                Ok((branch_name.clone(), cbranch))
            })
            .collect::<Result<HashMap<_, _>, Box<Error>>>());

        Ok(Project {
            service: service,
            context: context,
            branches: branches,
            members: RefCell::new(members),
            maintainers: maintainers,
        })
    }

    /// Add a member to the project.
    ///
    /// Overwrites the previous membership structure.
    pub fn add_member(&self, member: Box<HostedMembership>) {
        self.members
            .borrow_mut()
            .insert(member.user().id(), member);
    }

    /// Remove a member from the project.
    pub fn remove_member(&self, user: &HostedUser) {
        self.members
            .borrow_mut()
            .remove(&user.id());
    }

    /// Get the access level of a user to the project.
    pub fn access_level(&self, user: &HostedUser) -> u64 {
        self.members
            .borrow()
            .get(&user.id())
            .and_then(|member| {
                // If the user's membership has expired, ignore their access level.
                if let Some(expiration) = member.expiration() {
                    if expiration <= &UTC::now() {
                        return None;
                    }
                }

                Some(member.access_level())
            })
            .unwrap_or(0)
    }

    /// Fetch membership information from the host.
    pub fn refresh_membership(&self, project: &str) -> HostResult<()> {
        *self.members.borrow_mut() = Self::member_map(try!(self.service.members(project)));

        Ok(())
    }

    fn member_map(members: Vec<Box<HostedMembership>>) -> HashMap<u64, Box<HostedMembership>> {
        members.into_iter()
            .map(|member| (member.user().id(), member))
            .collect()
    }

    fn check_for_branch<'a>(&'a self, branch: &'a Branch, merge: bool) -> check::Check<'a> {
        let checks = if merge {
            &branch.merge_checks
        } else {
            &branch.checks
        };
        let conf = checks.config();

        check::Check::new(self.context.clone(),
                          self.service.clone(),
                          conf,
                          &self.maintainers)
    }

    /// Check that a merge request is OK.
    pub fn check_mr(&self, branch: &Branch, mr: &HostedMergeRequest)
                    -> check::Result<check::CheckStatus> {
        self.check_for_branch(branch, false)
            .check_mr(format!("mr/{}", mr.id()), &CommitId::new(&branch.name), mr)
    }
}

struct Checks {
    checks: Vec<Box<Check>>,
    branch_checks: Vec<Box<BranchCheck>>,
}

impl Checks {
    fn new<'a, I>(check_configs: I) -> Result<Self, Box<Error>>
        where I: Iterator<Item = (&'a String, &'a io::CheckConfig)>,
    {
        let mut checks = vec![];
        let mut branch_checks = vec![];
        let mut seen = HashSet::new();

        for (name, check) in check_configs {
            if !seen.insert(name) {
                continue;
            }

            if let Some(ref name) = check.name {
                try!(checks::create_check(name,
                                          check.config.clone(),
                                          &mut checks,
                                          &mut branch_checks));
            }
        }

        if checks.is_empty() && branch_checks.is_empty() {
            warn!(target: "ghostflow-director/checks",
                  "no checks configured for a branch");
        }

        Ok(Checks {
            checks: checks,
            branch_checks: branch_checks,
        })
    }

    fn config(&self) -> GitCheckConfiguration {
        let mut conf = GitCheckConfiguration::new();

        for check in &self.checks {
            conf.add_check(check.as_ref());
        }

        for check in &self.branch_checks {
            conf.add_branch_check(check.as_ref());
        }

        conf
    }
}

/// The filter for the workflow policy.
pub struct WorkflowMergePolicyFilter {
    has_check: bool,
    trailers: Vec<Trailer>,
    errors: Vec<String>,
}

impl merge::MergePolicyFilter for WorkflowMergePolicyFilter {
    fn process_trailer(&mut self, trailer: Trailer, _: Option<&HostedUser>) {
        // Ignore trailers without a `-by` suffix.
        if !trailer.token.ends_with("-by") {
            return;
        }

        // TODO: Allow only a whitelist of trailers.
        // TODO: Require at least an Acked-by (with sufficient permissions).
        // TODO: Require Reviewed-by (with sufficient permissions/assignee(s)).
        // TODO: Look at Rejected-by.

        // Accept the trailer.
        self.trailers.push(trailer);
    }

    fn result(mut self) -> Result<Vec<Trailer>, Vec<String>> {
        if !self.has_check {
            self.errors.push("The merge request has not been checked.".to_string());
        }

        if self.errors.is_empty() {
            Ok(self.trailers)
        } else {
            Err(self.errors)
        }
    }
}

#[derive(Debug)]
/// The policy structure for merging.
pub struct WorkflowMergePolicy {
    conf: io::MergePolicy,
    identity: Identity,
}

impl WorkflowMergePolicy {
    fn new(conf: io::MergePolicy, identity: Identity) -> Self {
        WorkflowMergePolicy {
            conf: conf,
            identity: identity,
        }
    }
}

impl merge::MergePolicy for WorkflowMergePolicy {
    type Filter = WorkflowMergePolicyFilter;

    fn for_mr(&self, mr: &HostedMergeRequest) -> Self::Filter {
        let has_check = mr.check_status().is_ok();
        let mut trailers = vec![];

        if has_check {
            trailers.push(Trailer::new("Acked-by", format!("{}", self.identity)));
        }

        WorkflowMergePolicyFilter {
            has_check: has_check,
            trailers: trailers,
            errors: vec![],
        }
    }
}

/// Representation of a merge action for a branch.
pub struct MergeAction {
    merge: merge::Merge<WorkflowMergePolicy>,
    /// Access level requirement to be able to use the stage action for a branch.
    pub access_level: u64,
}

impl MergeAction {
    /// A mutable reference to the stage workflow action.
    pub fn merge(&self) -> &merge::Merge<WorkflowMergePolicy> {
        &self.merge
    }
}

/// Representation of a stage action for a branch.
pub struct StageAction {
    stage: RefCell<stage::Stage>,
    /// The policy for updating branches on this stage.
    pub policy: StageUpdatePolicy,
    /// Access level requirement to be able to use the stage action for a branch.
    pub access_level: u64,
}

impl StageAction {
    /// A mutable reference to the stage workflow action.
    pub fn stage(&self) -> RefMut<stage::Stage> {
        self.stage.borrow_mut()
    }
}

/// Configuration for a branch within a project.
pub struct Branch {
    /// The name of the branch.
    pub name: String,

    checks: Checks,
    merge_checks: Checks,

    context: GitContext,
    identity: Identity,

    merge: Option<MergeAction>,
    stage: Option<StageAction>,
}

impl Branch {
    fn new(name: &str, checks: Checks, merge_checks: Checks, context: GitContext,
           identity: Identity)
           -> Result<Self, Box<Error>> {
        Ok(Branch {
            name: name.to_string(),

            checks: checks,
            merge_checks: merge_checks,

            context: context,
            identity: identity,

            merge: None,
            stage: None,
        })
    }

    fn add_merge(&mut self, merge_conf: &io::Merge, project: HostedProject)
                 -> Result<&mut Self, merge::Error> {
        let mut merge = merge::Merge::new(self.context.clone(),
                                          self.name.clone(),
                                          project,
                                          WorkflowMergePolicy::new(merge_conf.policy.clone(),
                                                                   self.identity.clone()));

        if merge_conf.quiet {
            merge.quiet();
        }

        merge.log_limit(merge_conf.log_limit);

        self.merge = Some(MergeAction {
            merge: merge,
            access_level: merge_conf.required_access_level,
        });
        Ok(self)
    }

    fn add_stage(&mut self, stage_conf: &io::Stage, project: HostedProject)
                 -> Result<&mut Self, stage::Error> {
        let branch = format!("refs/stage/{}/head", self.name);
        let stager = try!(Stager::from_branch(&self.context,
                                              CommitId::new(&self.name),
                                              CommitId::new(branch),
                                              self.identity.clone()));

        let mut stage = try!(stage::Stage::new(stager, &self.name, project));

        if stage_conf.quiet {
            stage.quiet();
        }

        self.stage = Some(StageAction {
            stage: RefCell::new(stage),
            policy: stage_conf.update_policy,
            access_level: stage_conf.required_access_level,
        });
        Ok(self)
    }

    /// Get the merge action for the branch.
    pub fn merge(&self) -> Option<&MergeAction> {
        self.merge.as_ref()
    }

    /// Get the stage action for the branch.
    pub fn stage(&self) -> Option<&StageAction> {
        self.stage.as_ref()
    }
}
