//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//============================================================================

#include "CoordinateSystem.h"

#include <vtkm/cont/CoordinateSystem.hxx>

namespace adis
{
namespace datamodel
{

void CoordinateSystem::ProcessJSON(const rapidjson::Value& json,
                                   DataSourcesType& sources)
{
  this->Array.reset();
  if (!json.HasMember("array") || !json["array"].IsObject())
  {
    throw std::runtime_error(
      this->ObjectName  + " must provide an array object.");
  }
  this->Array = std::make_shared<adis::datamodel::Array>();
  this->Array->ObjectName = "array";
  this->Array->ProcessJSON(json["array"], sources);
}

size_t CoordinateSystem::GetNumberOfBlocks(
  const std::unordered_map<std::string, std::string>& paths,
  DataSourcesType& sources)
{
  return this->Array->GetNumberOfBlocks(paths, sources);
}

template <typename T>
using VecType2 = vtkm::Vec<T, 2>;
template <typename T>
using VecType3 = vtkm::Vec<T, 3>;

struct TransformFunctor
{
  template<typename T>
  VTKM_EXEC VecType3<T> operator()(const VecType2<T>& input) const
  {
    return VecType3<T>{input[0], input[1], 0};
  }
};

struct Converter
{
  // 2D points not supported, convert them to 3D points by copying.
  template <typename T>
  void operator()(const vtkm::cont::ArrayHandle<VecType2<T> >& aHandle,
    std::vector<vtkm::cont::CoordinateSystem>& coordSystems)
  {
    auto tranformHandle = make_ArrayHandleTransform(aHandle, TransformFunctor{});
    coordSystems.push_back(
      vtkm::cont::CoordinateSystem("coordinates", tranformHandle));
  }

  // Use 3D points directly.
  template<typename T, typename StorageType>
  void operator()(const vtkm::cont::ArrayHandle<T, StorageType>& aHandle,
    std::vector<vtkm::cont::CoordinateSystem>& coordSystems)
  {
    coordSystems.push_back(
      vtkm::cont::CoordinateSystem("coordinates", aHandle));
  }
};

std::vector<vtkm::cont::CoordinateSystem> CoordinateSystem::Read(
  const std::unordered_map<std::string, std::string>& paths,
  DataSourcesType& sources,
  const adis::metadata::MetaData& selections)
{
  std::vector<vtkm::cont::DynamicArrayHandle> arrays =
    this->Array->Read(paths, sources, selections);
  std::vector<vtkm::cont::CoordinateSystem> coordSystems;
  coordSystems.reserve(arrays.size());
  using VectorTypes =
    vtkm::ListTagBase<VecType2<float>, VecType2<double>, VecType3<float>, VecType3<double> >;
  using CoordinateStorageTypes =
    vtkm::ListTagBase<vtkm::cont::ArrayHandleUniformPointCoordinates::StorageTag,
                      vtkm::cont::StorageTagBasic>;
  for(auto& array : arrays)
  {
    auto vecTypeArray = array.ResetTypeList(VectorTypes{});
    auto expandedTypeArray = vecTypeArray.ResetStorageList(CoordinateStorageTypes{});
    vtkm::cont::CastAndCall(expandedTypeArray, Converter{}, coordSystems);
  }
  return coordSystems;
}

}
}