// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::UTC;

extern crate rand;
use self::rand::Rng;

extern crate serde_json;
use self::serde_json::{Map, Value};

use std::collections::hash_map::HashMap;
use std::error::Error;
use std::fs::File;
use std::path::{Path, PathBuf};

#[derive(Deserialize, Debug)]
struct Filter {
    kind: String,

    #[serde(default)]
    have_keys: Vec<String>,
    #[serde(default)]
    items: Map<String, Value>,
}

impl Filter {
    fn is_match(&self, object: &Value) -> bool {
        for key in &self.have_keys {
            if object.pointer(key).is_none() {
                return false;
            }
        }

        for (pointer, expected) in &self.items {
            let matches = object.pointer(pointer)
                .map(|value| value == expected);

            if !matches.unwrap_or(false) {
                return false;
            }
        }

        true
    }
}

#[derive(Deserialize, Debug)]
pub struct Handler {
    path: String,
    filters: Vec<Filter>,
}

impl Handler {
    pub fn kind(&self, object: &Value) -> Option<&str> {
        for filter in &self.filters {
            if filter.is_match(object) {
                info!("matched an event of kind {}", filter.kind);

                return Some(&filter.kind);
            }
        }

        None
    }

    pub fn write_object(&self, kind: &str, object: Value) -> Result<(), Box<Error>> {
        let rndpart = rand::thread_rng()
            .gen_ascii_chars()
            .take(12)
            .collect::<String>();
        let filename = format!("{}-{}.json", UTC::now().to_rfc3339(), rndpart);

        let mut filepath = PathBuf::from(&self.path);
        filepath.push(filename);

        debug!("writing an event of kind {} to {}",
               kind,
               filepath.display());

        let mut fout = File::create(&filepath)?;

        let output = json!({
            "kind": kind,
            "data": object
        });

        Ok(serde_json::to_writer(&mut fout, &output)?)
    }
}

pub type HandlerMap = HashMap<String, Handler>;

#[derive(Deserialize, Debug)]
pub struct Config {
    pub handlers: HandlerMap,
}

impl Config {
    /// Read the configuration from a path.
    pub fn from_path<P: AsRef<Path>>(path: P) -> Result<Self, Box<Error>> {
        let fin = File::open(path.as_ref())?;
        Ok(serde_json::from_reader(fin)?)
    }
}
