#=============================================================================
#
#  Copyright (c) Kitware, Inc.
#  All rights reserved.
#  See LICENSE.txt for details.
#
#  This software is distributed WITHOUT ANY WARRANTY; without even
#  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the above copyright notice for more information.
#
#=============================================================================
from __future__ import print_function
import datetime
import os
import warnings
print('loading', os.path.basename(__file__))

import smtk
import smtk.attribute
import smtk.model
from cardformat import CardFormat
from namelist import Namelist

# Import material writers, which are used as mixin classes
import materialwriter
reload(materialwriter)
from materialwriter import MaterialWriter
import legacymaterialwriter
reload(legacymaterialwriter)
from legacymaterialwriter import LegacyMaterialWriter


# ---------------------------------------------------------------------
# Strings used in "if_condition" arguments (for convenience only)
NAMELIST = 'namelist'
THERMAL_ANALYSIS = 'thermal-analysis'
ONLY_THERMAL_ANALYSIS = 'only-thermal-analysis'
FLOW_ANALYSIS = 'flow-analysis'
INVISCID_FLOW = 'inviscid_flow'
VISCOUS_FLOW = 'viscous-flow'
FLUID_PHASE = 'fluid-phase'
SOLID_PHASE = 'solid-phase'
MASS_LIMITER = 'mass-limiter'
BC_INFLOW = 'bc-inflow'
VOID_MATERIAL = 'void'  # (must match material/material-type item)
ENCLOSURE_RADIATION = 'enclosure-radiation'
MOVING_RADIATION = 'moving-radiation'
INDUCTION_HEATING = 'induction heating'

# ---------------------------------------------------------------------
class Writer(object, MaterialWriter, LegacyMaterialWriter):
  '''Top level writer class for Truchas input files.
  '''

# ---------------------------------------------------------------------
  def __init__(self, operator_spec, mesh_filename='NOT-FOUND', altmesh_filename='NOT-FOUND'):
    '''
    '''
    self.sim_atts = smtk.attribute.Resource.CastTo(operator_spec.find('attributes').value())
    print('sim_atts', self.sim_atts)
    if self.sim_atts is None:
        msg = 'ERROR - No simulation attributes'
        print(msg)
        raise Exception(msg)

    model_entity = smtk.model.Entity.CastTo(operator_spec.find('model').objectValue(0))
    self.model_resource = smtk.model.Resource.CastTo(model_entity.resource())

    if self.model_resource is None:
        msg = 'ERROR - No model'
        print(msg)
        raise Exception(msg)

    self.analyses = list()
    self.categories = set()
    self.checked_attributes = set()  # attributes that have been validated
    self.background_material_id = None
    self.format_table = dict()
    self.interface_set_ids = list()
    self.enclosure_surface_set_ids = list()
    self.moving_enclosure_surface_set_ids = list()
    self.material_number_dict = {0: None}  # <phase att id, material number>
    self.mesh_file = mesh_filename
    self.altmesh_file = altmesh_filename
    self.namelist_sequence = list()
    self.skip_void_material = False
    self.out = None
    self.warning_messages = list()

# ---------------------------------------------------------------------
  def write(self, path, namelist_sequence, format_table):
    '''
    '''
    self.namelist_sequence = namelist_sequence
    self.format_table = format_table

    self._setup()

    completed = False
    with open(path, 'w') as out:
      # Write any user-provided description
      desc_att = self.sim_atts.findAttribute('description')
      if desc_att:
        desc = desc_att.findString('description').value(0)
        if desc:
          out.write('{}\n'.format(desc))
        else:
          out.write('* Truchas input file\n')

      # Add cmb output strings
      dt_string = datetime.datetime.now().strftime('%d-%b-%Y  %H:%M')
      out.write('* Generated by CMB {}\n'.format(dt_string))

      analysis_string = ', '.join(self.analyses)
      out.write('* Analysis Modules: {}\n'.format(analysis_string))


      self.out = out

      for namelist in self.namelist_sequence:
        # Check conditions
        if not CardFormat.test_conditions(namelist.if_condition):
          continue

        if namelist.separator:
          out.write('\n')
          label = '### {} '.format(namelist.separator)
          line = label.ljust(80, '#')
          out.write(line)
          out.write('\n')

        format_list = self.format_table.get(namelist.title)

        # Namelists can assign custom method
        if namelist.custom_method is not None:
          if not hasattr(self, namelist.custom_method):
            print('ERROR: For namelist', namelist.title, \
              ', custom_method', namelist.custom_method, \
              'not found')
          else:
            method = getattr(self, namelist.custom_method)
            method(namelist, format_list)
          continue

        # Else use the default namelist writer
        else:
          self._write_namelist_default(namelist, format_list)

      completed = True
      print('Wrote project file %s' % path)
    return completed

# ---------------------------------------------------------------------
  def _start_namelist(self, namelist_or_title):
    if isinstance(namelist_or_title, Namelist):
      title = namelist_or_title.title
    else:
      title = namelist_or_title
    self.out.write('\n&%s\n' % title)

# ---------------------------------------------------------------------
  def _finish_namelist(self):
    self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_namelist_default(self, namelist, format_list):
    '''
    '''
    print('Writing', namelist.title, 'namelists')

    # If namelist specifies attribute, process each one
    if namelist.att_type is not None:
      att_list = self.sim_atts.findAttributes(namelist.att_type)
      att_list.sort(key=lambda att: att.name())
      for att in att_list:
        if att.isMemberOf(list(self.categories)):
          self._check_attribute(att)
          self._write_default_att(att, namelist, format_list)
      return

    # Otherwise process each card in format_list
    else:
      self.out.write('\n&%s\n' % namelist.title)
      for card in format_list:
        if card.att_type is None:
          raise RuntimeError('card {} missing att_type'.format(card.keyword))
        att_list = self.sim_atts.findAttributes(card.att_type)
        for att in att_list:
          if att.isMemberOf(list(self.categories)):
            self._check_attribute(att)
            card.write(self.out, att, base_item_path=namelist.base_item_path)
      self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_default_att(self, att, namelist, format_list):
    '''Writes namelist for 1 attribute
    '''
    self.out.write('\n&%s\n' % namelist.title)
    self._check_attribute(att)

    for card in format_list:
      if card.att_type is None:
        card.write(
          self.out, att, base_item_path=namelist.base_item_path)
      else:
        card_att_list = self.sim_atts.findAttributes(card.att_type)
        for card_att in card_att_list:
          self._check_attribute(card_att)
          card.write(self.out, card_att)
    self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_mesh(self, namelist, format_list):
    '''
    '''
    print('Writing', namelist.title)
    self._start_namelist(namelist)

    att_list = self.sim_atts.findAttributes(namelist.att_type)
    mesh_att = att_list[0]
    self._check_attribute(mesh_att)
    mesh_comment = None
    for card in format_list:
      if not card.is_custom:
        card.write(self.out, mesh_att)
      elif 'mesh_file' == card.keyword:
        CardFormat.write_value(self.out, card.keyword, self.mesh_file)
      elif 'altmesh_file' == card.keyword:
        CardFormat.write_value(self.out, card.keyword, self.altmesh_file)
      elif 'interface_side_sets' == card.keyword:
        if self.interface_set_ids:
          side_set_string = ','.join([str(x) for x in self.interface_set_ids])
          CardFormat.write_value(
            self.out, 'interface_side_sets', side_set_string, quote_string=False)

    self._finish_namelist()

    if mesh_comment:
      self.out.write('* Mesh file on local file system at:\n')
      self.out.write('*  {}\n'.format(mesh_comment))

# ---------------------------------------------------------------------
  def _write_physics(self, namelist, format_list):
    """Need custom logic because analysis attribute has no categories

    """
    print('Writing', namelist.title)
    self.out.write('\n&%s\n' % namelist.title)
    for card in format_list:
      card.write(self.out, None)
    self.out.write('/\n')


# ---------------------------------------------------------------------
  def _write_enclosure(self, namelist, format_list):
    '''
    '''
    if not 'Enclosure Radiation' in self.categories:
      return
    print('Writing', namelist.title)

    # Also need enclosure *surface* format list
    surface_namelist = 'ENCLOSURE_SURFACE'
    surface_format_list = self.format_table.get(surface_namelist)

    att_list = self.sim_atts.findAttributes(namelist.att_type)
    for enclosure_att in att_list:
      self._check_attribute(enclosure_att)
      self._start_namelist(namelist)
      for card in format_list:
        card.write(self.out, enclosure_att)
      self._finish_namelist()
      enclosure_name = enclosure_att.name()

      # Write associated surface namelists
      surface_ref = enclosure_att.associations()
      num_refs = surface_ref.numberOfValues()
      if num_refs == 0:
        msg = 'WARNING: Invalid Eclosure \"{}\" -- not associated with any surfaces).\n'.format(enclosure_name)
        warnings.warn(msg)
        self.warning_messages.append(msg)
        continue

      surface_count = 0
      for i in range(num_refs):
        if not surface_ref.isSet(i):
          continue
        surface_att = surface_ref.objectValue(i)
        surface_count += 1
        self._check_attribute(surface_att)

        self._start_namelist(surface_namelist)
        CardFormat.write_value(self.out, 'name', surface_att.name())
        CardFormat.write_value(self.out, 'enclosure_name', enclosure_name)
        for card in surface_format_list:
          card.write(self.out, surface_att)
        self._finish_namelist()

      if surface_count == 0:
        msg = 'WARNING: Invalid Eclosure \"{}\" -- not associated with any surfaces).\n'.format(enclosure_name)
        warnings.warn(msg)
        self.warning_messages.append(msg)


# ---------------------------------------------------------------------
  def _write_thermal_condition(self, namelist, format_list):
    '''Common writer for ds_boundary_condition, ds_interface_condition, enclosure-surface
    '''
    if not 'Heat Transfer' in self.categories:
      return
    print('Writing %s namelists' % (namelist.title))

    def write_items(att, name):
        item = att.findDouble(name)
        name = name.replace('-', '_')
        if item.isExpression(0):
          expression_att = item.expression(0)
          CardFormat.write_value(self.out, '{}_func'.format(name), expression_att.name())
        else:
          CardFormat.write_value(self.out, name, item.value(0))

    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att_list.sort(key=lambda att: att.name())
    for att in att_list:
      #print('Writing att ', att.name())
      reference_item = att.associations()
      if reference_item is None or (0 == reference_item.numberOfValues()):
        print('Skipping attribute type \"%s\", name \"%s\" -- no associations' % \
          (att.type(), att.name()))
        continue

      self._check_attribute(att)

      self._start_namelist(namelist)
      CardFormat.write_value(self.out, 'name', att.name())

      # To get condition, split the type
      bc_type = att.type().split('.')[-1]
      CardFormat.write_value(self.out, 'type', bc_type)

      face_set_string = CardFormat.get_model_entity_ids(reference_item, as_string=True)
      CardFormat.write_value(
        self.out, 'face_set_ids', face_set_string, quote_string=False)

      if bc_type == 'temperature':
        write_items(att, 'temp')
      elif bc_type == 'flux':
        write_items(att, 'flux')
      elif bc_type == 'htc':
        write_items(att, 'htc')
        write_items(att, 'ambient-temp')
      elif bc_type == 'radiation':
        write_items(att, 'emissivity')
        write_items(att, 'ambient-temp')
      elif bc_type == 'interface-htc':
        write_items(att, 'htc')
      elif bc_type == 'gap-radiation':
        write_items(att, 'emissivity')

      self._finish_namelist()
    return

# ---------------------------------------------------------------------
  def _write_bc(self, namelist, format_list):
    '''Common writer for fluid boundary conditions

    '''
    print('Writing', namelist.title, 'namelists')
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att_list.sort(key=lambda att: att.name())
    for att in att_list:
      if not att.isMemberOf(list(self.categories)):
        continue

      reference_item = att.associations()
      if reference_item is None or (0 == reference_item.numberOfValues()):
        print('Skipping attribute type \"%s\", name \"%s\" -- no associations' % \
          (att.type(), att.name()))
        continue

      self._check_attribute(att)

      bc_value = None
      bc_expression = None
      # Get bc_type
      bc_type = 'dirichlet'  # default
      var_item = att.findString('variable')
      if ('velocity' == var_item.value(0)):
        #item = var_item.findChild('velocity-bc-type', smtk.attribute.SearchStyle.ACTIVE_CHILDREN)
        bc_type_item = att.itemAtPath('variable/velocity-bc-type', '/')
        bc_type = bc_type_item.value(0)

        if bc_type:
          bc_value_item = att.itemAtPath(
            'variable/velocity-bc-type/velocity-group/velocity-value', '/')
          value_list = [
            bc_value_item.value(1),
            bc_value_item.value(2),
            bc_value_item.value(3)]
          string_list = [str(x) for x in value_list]
          bc_value = ', '.join(string_list)
      else:
        bc_value_item = att.itemAtPath('variable/pressure-value', '/')
        if bc_value_item.isExpression(0):
          expression_att = bc_value_item.expression(0)
          bc_expression = expression_att.name()
        else:
          bc_value = bc_value_item.value(0)

      # Get enabled state for inflow group
      inflow_item = att.itemAtPath('variable/inflow', '/')
      if inflow_item.isEnabled():
        CardFormat.Conditions.add(BC_INFLOW)

      self.out.write('\n&%s\n' % namelist.title)
      for card in format_list:
        if 'bc_type' == card.keyword:
          CardFormat.write_value(self.out, card.keyword, bc_type)
        elif 'bc_value' == card.keyword:
          if bc_expression is not None:
            CardFormat.write_value(
              self.out, card.expression_keyword, bc_expression)
          elif bc_value is not None:
            CardFormat.write_value(
              self.out, card.keyword, bc_value, quote_string=False)
        else:
          card.write(
            self.out, att, base_item_path=namelist.base_item_path)
      self.out.write('/\n')
      CardFormat.Conditions.discard(BC_INFLOW)

# ---------------------------------------------------------------------
  def _write_body(self, namelist, format_list):
    '''Common writer for BODY namelists

    '''
    print('Writing', namelist.title, 'namelists')

    # FYI the attribute type for writing body namelists is "phase"
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    if not att_list:
      return

    # Local method to find attribute of a specified type on
    # a given model entity
    def _get_attribute(att_resource, model_entity, att_type):
      # Get set of attributes associated to model_entity
      ent_set = att_resource.attributes(model_entity)
      # Get list of attributes of given type
      type_list = att_resource.findAttributes(att_type)
      # Intersection gives set that meet both cases
      common_set = ent_set.intersection(set(type_list))
      if common_set:
        return common_set.pop()  # only return 1 att
      # (else)
      return None

    # Traverse phase attributes
    att_list.sort(key=lambda att: att.name())
    for att in att_list:
      if not att.isMemberOf(list(self.categories)):
        continue

      reference_item = att.associations()
      if reference_item is None or (0 == reference_item.numberOfValues()):
        print('Skipping attribute type \"%s\", name \"%s\" -- no associations' % \
          (att.type(), att.name()))
        continue

      # Write separate BODY namelist for each model item
      for i in range(reference_item.numberOfValues()):
        model_ent = reference_item.objectValue(i)

        self._start_namelist(namelist)
        for card in format_list:
          if 'material_number' == card.keyword:
            key = self._material_key(att, 0)
            material_number = self.material_number_dict.get(key)
            CardFormat.write_value(self.out, card.keyword, material_number)
          elif card.keyword == 'mesh_material_number':
            model_ent_id = CardFormat.get_model_entity_id(model_ent)
            CardFormat.write_value(self.out, card.keyword, model_ent_id)
          elif card.keyword in ['temperature', 'velocity']:
            # Initial conditions are separate attributes on the same model entity
            ic_att = _get_attribute(self.sim_atts, model_ent, card.att_type)
            if ic_att:
              card.write(self.out, ic_att)
          else:
            card.write(self.out, att)
        self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_outputs(self, namelist, format_list):
    '''Writes OUTPUTS namelist

    Needs custom logic because of item organization
      attribute type "outputs"
        double item "start-time"
        double item "end-time"
        double item  "output-dt"
        optional extensible group item "output-times"
          double item "time"
            component 0 = output_t
            component 1 = output_dt

    '''
    print('Writing', namelist.title)
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    time_list = list()
    dt_list = list()
    try:
      att = att_list[0]
      self._check_attribute(att)
      start_time = att.findDouble('start-time').value(0)
      end_time = att.findDouble('end-time').value(0)
      dt = att.findDouble('output-dt').value(0)

      time_list.append(start_time)
      dt_list.append(dt)

      # Check for optional output-times
      group_item = att.itemAtPath('output-times', '/')
      num_groups = group_item.numberOfGroups()
      for i in range(num_groups):
        double_item = group_item.item(i, 0)
        time_list.append(double_item.value(0))
        dt_list.append(double_item.value(1))
      time_list.append(end_time)
    except Exception as ex:
      print('ERROR ', ex)
      return

    # Write each component as separate array
    self.out.write('\n&%s\n' % namelist.title)

    formatted_list = ['%s' % value for value in time_list]
    formatted_string = ', '.join(formatted_list)
    CardFormat.write_value(
      self.out, 'output_t', formatted_string, quote_string=False, tab=10)

    formatted_list = ['%s' % value for value in dt_list]
    formatted_string = ', '.join(formatted_list)
    CardFormat.write_value(
      self.out, 'output_dt', formatted_string, quote_string=False, tab=10)

    self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_simcontrol(self, namelist, format_list):
    '''Common writer for SIMULATION_CONTROL namelist

    '''
    print('Writing', namelist.title, 'namelist')
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att = att_list[0]

    # Get group item, which is optional
    item = att.itemAtPath('simulation-control', '/')
    if not item.isEnabled():
      return

    # (else)
    self.out.write('\n&%s\n' % namelist.title)
    for card in format_list:
      card.write(self.out, att)
    self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_function(self, namelist, format_list):
    '''Common writer for FUNCTION namelists

    '''
    print('Writing {} namelists ({})'.format(namelist.title, namelist.att_type))
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att_list.sort(key=lambda att: att.name())
    for att in att_list:
      if not att.isMemberOf(list(self.categories)):
        continue

      self._check_attribute(att)
      self.out.write('\n&%s\n' % namelist.title)
      for card in format_list:
        card.write(self.out, att)

      # Write type-specific function  data
      type_item = att.findString('type')
      function_type = type_item.value(0)

      if function_type == 'polynomial':
        group_item = att.itemAtPath('type/polynomial-terms')
        num_groups = group_item.numberOfGroups()

        if num_groups < 1:
          print('Warning: function {} contents are empty -- skipping'.format(att.name()))
          continue

        # Get number of independent vars
        exp_item0 = group_item.item(0, 1)
        num_vars = exp_item0.numberOfValues()

        coef_list = [0.0] * num_groups
        # Create 2D list for exponentials [group][variable]
        exp_lists = [ [0 for x in range(num_vars)] for y in range(num_groups) ]

        for i in range(num_groups):
          coef_item = group_item.item(i, 0)
          coef_list[i] = coef_item.value()

          exp_item = group_item.item(i, 1)
          for j in range(num_vars):
            exp_lists[i][j] = exp_item.value(j)

        CardFormat.write_value(self.out, 'poly_coefficients', coef_list)
        if num_vars == 1:
          # For one var, can flatten to a single row of exponents
          keyword = 'poly_exponents(1,:)'
          flat_list = [exp[0] for exp in exp_lists]
          CardFormat.write_value(self.out, keyword, flat_list)
        else:
          # For the general case, write out the array
          for k in range(num_groups):
            keyword = 'poly_exponents(:, {})'.format(k+1)
            CardFormat.write_value(self.out, keyword, exp_lists[k])

        # Finish with the center value
        CardFormat('poly_refvars', item_path='type/center').write(self.out, att)
      elif function_type == 'tabular':
        # For multi-dimensional functions, the "tabular dimension" is stored as
        # the "independent-variable" item.
        ivar_item = att.itemAtPath('type/independent-variable')
        if ivar_item is not None:
          tabular_dim = ivar_item.value()
          CardFormat.write_value(self.out, 'tabular_dim', tabular_dim)

        group_item = att.itemAtPath('type/tabular-data')
        num_terms = group_item.numberOfGroups()
        xvals = [0.0] * num_terms
        yvals = [0.0] * num_terms
        for i in range(num_terms):
          xvals[i] = group_item.item(i, 0).value()
          yvals[i] = group_item.item(i, 1).value()
        CardFormat.write_value(self.out, 'tabular_data(1,:)', xvals)
        CardFormat.write_value(self.out, 'tabular_data(2,:)', yvals)

        # Finish with interpolation & extrapolation
        CardFormat('tabular_interp', item_path='type/interpolation').write(self.out, att)
        CardFormat('tabular_extrap', item_path='type/extrapolation').write(self.out, att)

      self.out.write('/\n')

#
  def _write_materials(self, namelist, format_list):
    """"""
    # These methods are in mixin classes
    self._write_physical_materials(namelist, format_list)
    self._write_legacy_materials(namelist, format_list)

# ---------------------------------------------------------------------
  def _write_electromagnetics(self, namelist, format_list):
    """Custom method for electromagnetics namelist

    Need custom method to handle extensible group item ("source")
    """
    print('Writing', namelist.title, 'namelists')
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    if not att_list:
      print('Warning: no {} attribute found'.format(namelist.att_type))
      return
    em_att = att_list[0]
    self._check_attribute(em_att)

    # First assemble lists of time & frequency values, from the induction-heating att
    att_list = self.sim_atts.findAttributes('induction-heating')
    if not att_list:
      print('Warning: no induction-heating attribute found')
      return
    ih_att = att_list[0]
    self._check_attribute(ih_att)

    source_item = ih_att.findGroup('source')
    if source_item is None:
      print('Warning: induction-heating attribute has no source groupitem')
      return

    num_groups = source_item.numberOfGroups()
    time_value = [None] * num_groups
    freq_value = [None] * num_groups
    for i in range(num_groups):
      time_value[i] = source_item.find(i, 'time').value()
      freq_value[i] = source_item.find(i, 'frequency').value()

    # Write the namelist
    self._start_namelist(namelist)

    for card in format_list:
      if card.keyword == 'source_times':
        if num_groups > 1:
          CardFormat.write_value(self.out, card.keyword, time_value[1:])
      elif card.keyword == 'source_frequency':
        CardFormat.write_value(self.out, card.keyword, freq_value)
      else:
        card.write(self.out, em_att)

    self._finish_namelist()

# ---------------------------------------------------------------------
  def _write_induction_coils(self, namelist, format_list):
    """Custom method for induction coils

    Need custom method to handle extensible group item ("coils")
    """
    print('Writing', namelist.title, 'namelists')
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    if not att_list:
      print('Warning: no {} attribute found'.format(namelist.att_type))
      return

    att = att_list[0]
    self._check_attribute(att)
    coils_item = att.findGroup('coils')
    if coils_item is None:
      print('Warning: {} attribute has no coils groupitem'.format(namelist.att_type))
      return

    for i in range(coils_item.numberOfGroups()):
      self._start_namelist(namelist)
      for card in format_list:
        card_item = coils_item.find(i, card.item_path)
        card.write_item(card_item, self.out)
      self._finish_namelist()

# ---------------------------------------------------------------------
  def _material_key(self, material_att, phase_number):
    """Return a string for a given phase of a material attribute

    Used for assigning material number
    """
    return '{}.{}'.format(material_att.id(), phase_number)

# ---------------------------------------------------------------------
  def _check_attribute(self, att):
    """Checks attribute validity"""
    if att in self.checked_attributes: # avoid redundancy
      return

    self.checked_attributes.add(att)
    if not att.isValid(self.categories):
      msg = 'WARNING: Invalid Attribute \"{}\" (type \"{}\").'.format(att.name(), att.type())
      warnings.warn(msg)
      self.warning_messages.append(msg)

# ---------------------------------------------------------------------
  def _setup(self):
    '''
    '''
    CardFormat.ModelManager = self.model_resource

    # Get analysis type(s) and set CardFormat conditions
    analysis_att = self.sim_atts.findAttribute('analysis')
    if not analysis_att:
      raise RuntimeError('Internal Error -- missing analysis att')

    # Get categories based on enabled analyses
    ER = 'Enclosure Radiation'
    FF = 'Fluid Flow'
    HT = 'Heat Transfer'
    IH = 'Induction Heating'
    SM = 'Solid Mechanics'

    self.analyses = list()
    self.categories = set()

    # Only looking at Heat Transfer and Fluid Flow initially
    er_item = None  # enclosure radition
    ff_item = analysis_att.findVoid(FF)
    ht_item = analysis_att.findGroup(HT)
    ih_item = None  # induction heating
    sm_item = analysis_att.findVoid(SM)
    analyses = self.sim_atts.analyses()
    category_set = set()
    if ht_item.isEnabled():
      ht_name = analyses.find(HT).name()

      CardFormat.Conditions.add(THERMAL_ANALYSIS)
      if not ff_item or not ff_item.isEnabled():
        CardFormat.Conditions.add(ONLY_THERMAL_ANALYSIS)
      self.analyses.append(HT)
      category_set |= analyses.find(HT).categories()

      er_item = ht_item.find(ER)
      if er_item and er_item.isEnabled():
        CardFormat.Conditions.add(ENCLOSURE_RADIATION)
        self.analyses.append(ER)
        category_set |= analyses.find(ER).categories()

      ih_item = ht_item.find(IH)
      if ih_item and ih_item.isEnabled():
        CardFormat.Conditions.add(INDUCTION_HEATING)
        self.analyses.append(IH)
        category_set |= analyses.find(IH).categories()

    if ff_item and ff_item.isEnabled():
      CardFormat.Conditions.add(FLOW_ANALYSIS)
      self.analyses.append(FF)
      category_set |= analyses.find(FF).categories(FF)
      # Todo VISCOUS_FLOW vs INVISCID_FLOW

    if sm_item and sm_item.isEnabled():
      self.analyses.append(SM)
      category_set |= analyses.find(SM).categories()
      raise RuntimeError('Solid Mechanics selected but NOT yet supported')

    self.categories = category_set
    print('Enabled analyses: {}'.format(self.analyses))
    print('Enabled categories: {}'.format(self.categories))
    # Iterim logic during development
    # att_list = self.sim_atts.findAttributes('ht.solver')
    # for att in att_list:
    #   analysis_item = att.findString('analysis')
    #   if analysis_item is not None:
    #     analysis = analysis_item.value(0)
    #     if analysis == 'thermal':
    #       print('Adding analysis conditions "{}" "{}"'.format(
    #         THERMAL_ANALYSIS, ONLY_THERMAL_ANALYSIS))
    #       CardFormat.Conditions.add(THERMAL_ANALYSIS)
    #       CardFormat.Conditions.add(ONLY_THERMAL_ANALYSIS)
    #     elif analysis == 'thermal-plus-fluid':
    #       print('Adding analysis conditions "{}", "{}"'.format(
    #         THERMAL_ANALYSIS, FLOW_ANALYSIS))
    #       CardFormat.Conditions.add(THERMAL_ANALYSIS)
    #       CardFormat.Conditions.add(FLOW_ANALYSIS)

    # Get the background material attribute's id
    att_list = self.sim_atts.findAttributes('background-material')
    if att_list:
      att = att_list[0]
      ref_item = att.findComponent('background-material')
      if ref_item.isSet(0):
        background_material_att = ref_item.objectValue(0)
        self.background_material_id = background_material_att.id()

    # Void material should be written (only) if either:
    #  - void material is associated with model entity(ies)
    #  - void material is set as the background material
    write_void_material = False
    # Check material.void and phase.void (legacy)
    for void_type in ['material.void', 'phase.void']:
      void_list = self.sim_atts.findAttributes(void_type)
      if len(void_list) == 1:
        void_att = void_list[0]
        if void_att.associations().numberOfValues() > 0:
          write_void_material = True
          break
        elif self.background_material_id == void_att.id():
          write_void_material = True
          break
      # else:
      #   print('Warning - missing {} attribute'.format(void_type))
    self.skip_void_material = not write_void_material

    # Find all interface_set_ids and enclosure_surface_set_ids
    self.interface_set_ids = list()
    self.enclosure_surface_set_ids = list()
    self.moving_enclosure_surface_set_ids = list()

    att_list = self.sim_atts.findAttributes('ht.interface')
    for att in att_list:
      reference_item = att.associations()
      if reference_item:
        self.interface_set_ids += CardFormat.get_model_entity_ids(reference_item)

      # type_item = att.findString('type')
      # if type_item.value(0) == 'ds-interface-condition':
      #   self.interface_set_ids += CardFormat.get_model_entity_ids(reference_item)
      # elif type_item.value(0) == 'enclosure-surface':
      #   surface_ids = CardFormat.get_model_entity_ids(reference_item)
      #   self.enclosure_surface_set_ids += surface_ids
      #   moving_item = att.itemAtPath('type/moving', '/')
      #   if moving_item and moving_item.isEnabled():
      #     self.moving_enclosure_surface_set_ids += surface_ids

    self.interface_set_ids.sort()
    self.enclosure_surface_set_ids.sort()
    self.moving_enclosure_surface_set_ids.sort()
    print('interface_set_ids', self.interface_set_ids)
    print('enclosure_surface_set_ids', self.enclosure_surface_set_ids)
    print('moving_enclosure_surface_set_ids', self.moving_enclosure_surface_set_ids)
