//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
// .NAME validators.h - functions for validating material attributes

#include "smtk/PublicPointerDefs.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/DoubleItem.h"
#include "smtk/attribute/GroupItem.h"
#include "smtk/attribute/Item.h"
#include "smtk/attribute/ItemDefinition.h"
#include "smtk/attribute/Resource.h"
#include "smtk/extension/qt/qtUIManager.h"

#include <QDebug>

#include <set>
#include <string>
#include <vector>

bool isPhaseElementValid(smtk::attribute::GroupItemPtr phaseItem, std::size_t element,
  smtk::extension::qtUIManager* uiManager);
bool isTransitionElementValid(smtk::attribute::GroupItemPtr groupItem, std::size_t element,
  smtk::extension::qtUIManager* uiManager);
bool isItemValid(smtk::attribute::ItemPtr item, smtk::extension::qtUIManager* uiManager);

bool isAttributeValid(smtk::attribute::AttributePtr att, smtk::extension::qtUIManager* uiManager)
{
  // Check shared items
  auto sharedItem = att->findAs<smtk::attribute::GroupItem>("shared-properties");
  for (std::size_t i = 0; i < sharedItem->numberOfItemsPerGroup(); ++i)
  {
    auto item = sharedItem->item(i);
    if (!isItemValid(item, uiManager))
    {
      return false;
    }
  } // for (i)

  // Check phase items
  auto phasesItem = att->findAs<smtk::attribute::GroupItem>("phases");
  for (std::size_t i = 0; i < phasesItem->numberOfGroups(); ++i)
  {
    if (!isPhaseElementValid(phasesItem, i, uiManager))
    {
      return false;
    }
  }

  // Check transition items
  auto transitionsItem = att->findAs<smtk::attribute::GroupItem>("transitions");
  for (std::size_t i = 0; i < transitionsItem->numberOfGroups(); ++i)
  {
    if (!isTransitionElementValid(transitionsItem, i, uiManager))
    {
      return false;
    }
  }

  return true;
}

bool isPhaseElementValid(smtk::attribute::GroupItemPtr phaseItem, std::size_t element,
  smtk::extension::qtUIManager* uiManager)
{
  // Validating phase items requires some custom considerations:
  //  - Some phase items are excluded by the current categories
  //  - Multiphase items can be superseded by "shared properties".
  //  - Multiphase materials must specify a "name" for each phase.

  // Our logic presumes that the shared-properties group item has the same set of
  // property names as the phase item, and that categories match for each shared-
  // and corresponding phase-item.

  // Get the property item names that are both (i) member of the
  // current category set and (ii) *not* shared
  auto materialAtt = phaseItem->attribute();
  auto attResource = materialAtt->attributeResource();

  // Use the shared-properties item to get the list of names to check
  bool isMultiphase = phaseItem->numberOfGroups() > 1;
  auto sharedItem = materialAtt->findGroup("shared-properties");
  assert(sharedItem);
  std::set<std::string> namesToCheck;
  for (std::size_t i = 0; i < sharedItem->numberOfItemsPerGroup(); ++i)
  {
    auto item = sharedItem->item(i);
    if (!uiManager->passItemCategoryCheck(item->definition()))
    {
      continue; // omit if not in current categories
    }
    if (!isMultiphase || !item->isEnabled())
    {
      // If single phase or if not shared, include in the check list
      namesToCheck.insert(item->name());
    }
  } // for (i)

  // For multiphase materials, also add phase "name"
  if (isMultiphase)
  {
    namesToCheck.insert("name");
  }

  // Check items in the set
  for (auto iter = namesToCheck.begin(); iter != namesToCheck.end(); ++iter)
  {
    auto name = *iter;
    auto item = phaseItem->find(element, name, smtk::attribute::RECURSIVE_ACTIVE);
    if (item && !item->isValid())
    {
      return false;
    }
  } // for (iter)

  return true;
}

bool isTransitionElementValid(smtk::attribute::GroupItemPtr groupItem, std::size_t element,
  smtk::extension::qtUIManager* uiManager)
{
  // Check individual items
  for (std::size_t i = 0; i < groupItem->numberOfItemsPerGroup(); ++i)
  {
    smtk::attribute::ItemPtr item = groupItem->item(element, i);
    if (!isItemValid(item, uiManager))
    {
      return false;
    }
  }

  // Check that low transition temp less than high transition temp
  bool isValid = true;
  if (groupItem->name() == "transitions")
  {
    double highTemp, lowTemp;
    int version = groupItem->definition()->version();
    if (version == 0)
    {
      highTemp =
        groupItem->findAs<smtk::attribute::DoubleItem>(element, "upper-transition-temperature")
          ->value();
      lowTemp =
        groupItem->findAs<smtk::attribute::DoubleItem>(element, "lower-transition-temperature")
          ->value();
      isValid = lowTemp < highTemp;
    }
    else if (version == 1)
    {
      qDebug() << "Todo validate transitions version 1 format";
    }
    else
    {
      qDebug() << "Unexpected version" << version << "for" << groupItem->name().c_str() << "item";
      isValid = false;
    }
  } // if (transitions item)

  return isValid;
}

bool isItemValid(smtk::attribute::ItemPtr item, smtk::extension::qtUIManager* uiManager)
{
  if (!item->isEnabled() || !uiManager->passItemCategoryCheck(item->definition()))
  {
    return true; // skip if not enabled or not in current categories
  }

  // (else)
  return item->isValid();
}
