#=============================================================================
#
#  Copyright (c) Kitware, Inc.
#  All rights reserved.
#  See LICENSE.txt for details.
#
#  This software is distributed WITHOUT ANY WARRANTY; without even
#  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the above copyright notice for more information.
#
#=============================================================================

import datetime
import imp
import os
import warnings
print('loading', os.path.basename(__file__))

import smtk
import smtk.attribute
import smtk.model
from .cardformat import CardFormat
from .namelist import Namelist

# Base class
from . import basewriter
imp.reload(basewriter)
from .basewriter import BaseWriter

# Formatter objects
from . import truchasformats
imp.reload(truchasformats)

# Import material writers, which are used as mixin classes
from . import materialwriter
imp.reload(materialwriter)
from .materialwriter import MaterialWriter

# Predefined strings for "if_condition" arguments
from .truchasformats import THERMAL_ANALYSIS, ONLY_THERMAL_ANALYSIS, FLOW_ANALYSIS, \
  VISCOUS_FLOW, INVISCID_FLOW, FLUID_PHASE, MASS_LIMITER, BC_INFLOW, \
  VOID_MATERIAL, ENCLOSURE_RADIATION, MOVING_RADIATION, INDUCTION_HEATING

# ---------------------------------------------------------------------
class TruchasWriter(BaseWriter, MaterialWriter):
  """Top level writer class for Truchas input files."""

# ---------------------------------------------------------------------
  def __init__(self, operator_spec, mesh_filename='NOT-FOUND', altmesh_filename='NOT-FOUND'):
    """"""
    BaseWriter.__init__(self, operator_spec, mesh_filename, altmesh_filename)
    self.background_material_id = None
    self.format_table = dict()
    self.interface_set_ids = list()
    self.namelist_sequence = list()
    self.test_mode = False

    test_item = operator_spec.findVoid('test-mode')
    if test_item is not None:
        self.test_mode = test_item.isEnabled()

# ---------------------------------------------------------------------
  def write(self, path):
    """"""
    self.namelist_sequence = truchasformats.namelist_sequence
    self.format_table = truchasformats.format_table

    self._setup()

    completed = False
    with open(path, 'w') as out:
      # Write any user-provided description
      desc_att = self.sim_atts.findAttribute('description')
      if desc_att:
        desc = desc_att.findString('description').value(0)
        if desc:
          out.write('{}\n'.format(desc))
        else:
          out.write('* Truchas input file\n')
      # Add cmb output strings
      line = '* Generated by CMB'
      if not self.test_mode:
          dt_string = datetime.datetime.now().strftime('%d-%b-%Y  %H:%M')
          line = '{} {}'.format(line, dt_string)
      out.write(line)
      out.write('\n')

      self.out = out
      for namelist in self.namelist_sequence:
        if namelist.if_condition is not None:
          print('NameList {} if_condition {}'.format(namelist.title, namelist.if_condition))
        # Check conditions
        if not CardFormat.test_conditions(namelist.if_condition):
          continue

        format_list = self.format_table.get(namelist.title)

        # Namelists can assign custom method
        if namelist.custom_method is not None:
          if not hasattr(self, namelist.custom_method):
            print('ERROR: For namelist', namelist.title, \
              ', custom_method', namelist.custom_method, \
              'not found')
          else:
            method = getattr(self, namelist.custom_method)
            method(namelist, format_list)
          continue

        # Else use the default namelist writer
        else:
          self._write_namelist_default(namelist, format_list)

      completed = True
      print('Wrote project file %s' % path)

    return completed

# ---------------------------------------------------------------------
  def _write_namelist_default(self, namelist, format_list):
    """"""
    print('Writing', namelist.title, 'namelists')

    # If namelist specifies attribute, process each one
    separator = namelist.separator
    if namelist.att_type is not None:
      att_list = self.sim_atts.findAttributes(namelist.att_type)
      att_list.sort(key=lambda att: att.name())
      for att in att_list:
        if att.categories().passes(self.categories):
          self._check_attribute(att)

          if separator is not None:
            self._write_separator(separator)
            separator = None

          self._write_default_att(att, namelist, format_list)
      return

    # Otherwise process each card in format_list
    else:
      self.out.write('\n&%s\n' % namelist.title)
      for card in format_list:
        if card.att_type is None:
          raise RuntimeError('card {} missing att_type'.format(card.keyword))
        att_list = self.sim_atts.findAttributes(card.att_type)
        for att in att_list:
          if att.categories().passes(self.categories):
            self._check_attribute(att)
            card.write(self.out, att, base_item_path=namelist.base_item_path)
      self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_default_att(self, att, namelist, format_list):
    """Writes namelist for 1 attribute."""
    self.out.write('\n&%s\n' % namelist.title)
    self._check_attribute(att)

    for card in format_list:
      if card.att_type is None:
        card.write(
          self.out, att, base_item_path=namelist.base_item_path)
      else:
        card_att_list = self.sim_atts.findAttributes(card.att_type)
        for card_att in card_att_list:
          self._check_attribute(card_att)
          card.write(self.out, card_att)
    self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_mesh(self, namelist, format_list):
    """"""
    print('Writing', namelist.title)
    self._start_namelist(namelist)

    att_list = self.sim_atts.findAttributes(namelist.att_type)
    mesh_att = att_list[0]
    self._check_attribute(mesh_att)
    mesh_comment = None
    for card in format_list:
      if not card.is_custom:
        card.write(self.out, mesh_att)
      elif 'mesh_file' == card.keyword:
        # For contract testing, write filename instead of full path.
        filename = os.path.basename(self.mesh_file) if self.test_mode else self.mesh_file
        CardFormat.write_value(self.out, card.keyword, filename)
      elif 'altmesh_file' == card.keyword:
        # For contract testing, write filename instead of full path.
        filename = os.path.basename(self.altmesh_file) if self.test_mode else self.altmesh_file
        CardFormat.write_value(self.out, card.keyword, filename)
      elif 'interface_side_sets' == card.keyword:
        if self.interface_set_ids:
          side_set_string = ','.join([str(x) for x in self.interface_set_ids])
          CardFormat.write_value(
            self.out, 'interface_side_sets', side_set_string, quote_string=False)

    self._finish_namelist()

    if mesh_comment:
      self.out.write('* Mesh file on local file system at:\n')
      self.out.write('*  {}\n'.format(mesh_comment))

# ---------------------------------------------------------------------
  def _write_physics(self, namelist, format_list):
    """Need custom logic for materials list."""
    print('Writing', namelist.title)
    self._start_namelist(namelist.title)

    for card in format_list:
      card.write(self.out, None)

    # Get list of materials names
    att_list = self.sim_atts.findAttributes('material.real')
    att_names = [att.name() for att in att_list]
    att_names.sort()

    # Append void if void.material assigned to any body attribute
    void_att = self._find_instanced_attribute('material.void')
    body_atts = self.sim_atts.findAttributes('body')
    for body_att in body_atts:
      ref_item = body_att.find('material')
      if ref_item.value() == void_att:
        att_names.append('VOID')
        break

    materials_string =  ', '.join('"{0}"'.format(name) for name in att_names)
    CardFormat.write_value(self.out, 'materials', materials_string, quote_string=False)

    self._finish_namelist()

# ---------------------------------------------------------------------
  def _write_enclosure(self, namelist, format_list):
    """"""
    if not 'Enclosure Radiation' in self.categories:
      return
    print('Writing', namelist.title)

    # Also need enclosure *surface* format list and the emissivity CardFormat
    surface_namelist = 'ENCLOSURE_SURFACE'
    surface_format_list = self.format_table.get(surface_namelist)
    emiss_format = None
    for fmt in surface_format_list:
      if fmt.keyword == 'emissivity_constant':
        emiss_format = fmt
        break
    if emiss_format is None:
      raise RuntimeError('INTERNAL ERROR: missing CardFormat for emissivity')

    # First generate a map of <sideset_id, emissivity_att> for grouping surfaces
    ssid_emiss_map = dict()
    emiss_attlist = self.sim_atts.findAttributes('emissivity')
    for emiss_att in emiss_attlist:
      ssid_list = CardFormat.get_associated_entity_ids(emiss_att)
      for ssid in ssid_list:
        ssid_emiss_map[ssid] = emiss_att

    # Internal class for grouping surfaces
    class EnclosureSurface:
      def __init__(self):
        self.emiss_att = None
        self.sideset_ids = list()

    # Traverse enclosure attributes
    separator = namelist.separator
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    for enclosure_att in att_list:
      self._check_attribute(enclosure_att)

      if separator is not None:
        self._write_separator(separator)
        separator = None

      # Generate dictionary of <emiss_att, Surface>
      emiss_surface_map = dict()
      ssid_list = CardFormat.get_associated_entity_ids(enclosure_att)
      for ssid in ssid_list:
        emiss_att = ssid_emiss_map.get(ssid)
        if emiss_att is None:
          msg = 'Error: side set {} used in enclosure but missing emissivity assignment'.format(ssid)
          self.warning_messages.append(msg)
          continue
        surface = emiss_surface_map.get(emiss_att)
        if surface is None:
          surface = EnclosureSurface()
          surface.emiss_att = emiss_att
        surface.sideset_ids.append(ssid)
        emiss_surface_map[emiss_att] = surface

      # Write ENCLOSURE_RADIATION namelist
      self._start_namelist(namelist)
      for card in format_list:
        if card.keyword == 'enclosure_file':
          filename = '{}.re'.format(enclosure_att.name())
          CardFormat.write_value(self.out, 'enclosure_file', filename)
        elif card.att_type is not None:
          self._write_instanced_attribute(card)
        else:
          card.write(self.out, enclosure_att)
      self._finish_namelist()
      enclosure_name = enclosure_att.name()

      for emiss_att,surface in emiss_surface_map.items():
        self._start_namelist('ENCLOSURE_SURFACE')
        CardFormat.write_value(self.out, 'enclosure_name', enclosure_name)
        surface_name = '{}-{}'.format(enclosure_name, emiss_att.name())
        CardFormat.write_value(self.out, 'name', surface_name)
        CardFormat.write_value(self.out, 'face_block_ids', surface.sideset_ids)
        emiss_item = emiss_att.findDouble('emissivity')
        emiss_format.write_item(emiss_item, self.out)
        self._finish_namelist()

# ---------------------------------------------------------------------
  def _write_thermal_condition(self, namelist, format_list):
    """Common writer for ds_boundary_condition, ds_interface_condition, enclosure-surface."""
    if not 'Heat Transfer' in self.categories:
      return
    print('Writing %s namelists' % (namelist.title))

    def write_items(att, name):
        item = att.findDouble(name)
        name = name.replace('-', '_')
        if item.isExpression(0):
          expression_att = item.expression(0)
          CardFormat.write_value(self.out, '{}_func'.format(name), expression_att.name())
        else:
          CardFormat.write_value(self.out, name, item.value(0))

    separator = namelist.separator
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att_list.sort(key=lambda att: att.name())
    for att in att_list:
      #print('Writing att ', att.name())
      reference_item = att.associations()
      if reference_item is None or (0 == reference_item.numberOfValues()):
        print('Skipping attribute type \"%s\", name \"%s\" -- no associations' % \
          (att.type(), att.name()))
        continue

      self._check_attribute(att)

      if separator is not None:
        self._write_separator(separator)
        separator = None

      self._start_namelist(namelist)
      CardFormat.write_value(self.out, 'name', att.name())

      # To get condition, split the type
      bc_type = att.type().split('.')[-1]
      CardFormat.write_value(self.out, 'type', bc_type)

      face_set_string = CardFormat.get_model_entity_ids(reference_item, as_string=True)
      CardFormat.write_value(
        self.out, 'face_set_ids', face_set_string, quote_string=False)

      if bc_type == 'temperature':
        write_items(att, 'temp')
      elif bc_type == 'flux':
        write_items(att, 'flux')
      elif bc_type == 'htc':
        write_items(att, 'htc')
        write_items(att, 'ambient-temp')
      elif bc_type == 'radiation':
        write_items(att, 'emissivity')
        write_items(att, 'ambient-temp')
      elif bc_type == 'interface-htc':
        write_items(att, 'htc')
      elif bc_type == 'gap-radiation':
        write_items(att, 'emissivity')

      self._finish_namelist()
    return

# ---------------------------------------------------------------------
  def _write_bc(self, namelist, format_list):
    """Common writer for fluid boundary conditions."""
    print('Writing', namelist.title, 'namelists')
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att_list.sort(key=lambda att: att.name())
    for att in att_list:
      if att.categories().passes(self.categories):
          continue

      reference_item = att.associations()
      if reference_item is None or (0 == reference_item.numberOfValues()):
        print('Skipping attribute type \"%s\", name \"%s\" -- no associations' % \
          (att.type(), att.name()))
        continue

      self._check_attribute(att)

      bc_value = None
      bc_expression = None
      # Get bc_type
      bc_type = 'dirichlet'  # default
      var_item = att.findString('variable')
      if ('velocity' == var_item.value(0)):
        #item = var_item.findChild('velocity-bc-type', smtk.attribute.SearchStyle.ACTIVE_CHILDREN)
        bc_type_item = att.itemAtPath('variable/velocity-bc-type', '/')
        bc_type = bc_type_item.value(0)

        if bc_type:
          bc_value_item = att.itemAtPath(
            'variable/velocity-bc-type/velocity-group/velocity-value', '/')
          value_list = [
            bc_value_item.value(1),
            bc_value_item.value(2),
            bc_value_item.value(3)]
          string_list = [str(x) for x in value_list]
          bc_value = ', '.join(string_list)
      else:
        bc_value_item = att.itemAtPath('variable/pressure-value', '/')
        if bc_value_item.isExpression(0):
          expression_att = bc_value_item.expression(0)
          bc_expression = expression_att.name()
        else:
          bc_value = bc_value_item.value(0)

      # Get enabled state for inflow group
      inflow_item = att.itemAtPath('variable/inflow', '/')
      if inflow_item.isEnabled():
        CardFormat.Conditions.add(BC_INFLOW)

      self.out.write('\n&%s\n' % namelist.title)
      for card in format_list:
        if 'bc_type' == card.keyword:
          CardFormat.write_value(self.out, card.keyword, bc_type)
        elif 'bc_value' == card.keyword:
          if bc_expression is not None:
            CardFormat.write_value(
              self.out, card.expression_keyword, bc_expression)
          elif bc_value is not None:
            CardFormat.write_value(
              self.out, card.keyword, bc_value, quote_string=False)
        else:
          card.write(
            self.out, att, base_item_path=namelist.base_item_path)
      self.out.write('/\n')
      CardFormat.Conditions.discard(BC_INFLOW)

# ---------------------------------------------------------------------
  def _write_body(self, namelist, format_list):
    """Common writer for BODY namelists."""
    print('Writing', namelist.title, 'namelists')

    separator = namelist.separator
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    if not att_list:
      return

    for att in att_list:
      self._check_attribute(att)

      id_list = CardFormat.get_associated_entity_ids(att)
      if not id_list:
        msg = 'body attribute {} not associated to any model entities'.format(att.name())
        print(msg)
        self.warning_messages.append(msg)
        continue

      if separator is not None:
        self._write_separator(separator)
        separator = None

      self._start_namelist(namelist)
      for card in format_list:
        if card.keyword == 'material_name':
          ref_item = att.find('material')
          mat_att = ref_item.value()
          mat_name = 'VOID' if mat_att.type() == 'material.void' else mat_att.name()
          CardFormat.write_value(self.out, card.keyword, mat_name)
        else:
          card.write(self.out, att)
      self._finish_namelist()

# ---------------------------------------------------------------------
  def _write_outputs(self, namelist, format_list):
    """Writes OUTPUTS namelist.

    Needs custom logic because of item organization
      attribute type "outputs"
        double item "start-time"
        double item "end-time"
        double item  "output-dt"
        optional extensible group item "output-times"
          double item "time"
            component 0 = output_t
            component 1 = output_dt

    """
    print('Writing', namelist.title)
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    time_list = list()
    dt_list = list()
    try:
      att = att_list[0]
      self._check_attribute(att)
      start_time = att.findDouble('start-time').value(0)
      end_time = att.findDouble('end-time').value(0)
      dt = att.findDouble('output-dt').value(0)

      time_list.append(start_time)
      dt_list.append(dt)

      # Check for optional output-times
      group_item = att.itemAtPath('output-times', '/')
      num_groups = group_item.numberOfGroups()
      for i in range(num_groups):
        double_item = group_item.item(i, 0)
        time_list.append(double_item.value(0))
        dt_list.append(double_item.value(1))
      time_list.append(end_time)
    except Exception as ex:
      print('ERROR ', ex)
      return

    # Write each component as separate array
    self.out.write('\n&%s\n' % namelist.title)

    formatted_list = ['%s' % value for value in time_list]
    formatted_string = ', '.join(formatted_list)
    CardFormat.write_value(
      self.out, 'output_t', formatted_string, quote_string=False, tab=10)

    formatted_list = ['%s' % value for value in dt_list]
    formatted_string = ', '.join(formatted_list)
    CardFormat.write_value(
      self.out, 'output_dt', formatted_string, quote_string=False, tab=10)

    self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_simcontrol(self, namelist, format_list):
    """Common writer for SIMULATION_CONTROL namelist."""
    print('Writing', namelist.title, 'namelist')
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att = att_list[0]

    # Get group item, which is optional
    item = att.itemAtPath('simulation-control', '/')
    if not item.isEnabled():
      return

    # (else)
    self.out.write('\n&%s\n' % namelist.title)
    for card in format_list:
      card.write(self.out, att)
    self.out.write('/\n')

# ---------------------------------------------------------------------
  def _write_function(self, namelist, format_list):
    """Common writer for FUNCTION namelists."""
    print('Writing {} namelists ({})'.format(namelist.title, namelist.att_type))
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    att_list.sort(key=lambda att: att.name())
    for att in att_list:
      if not att.categories().passes(self.categories):
        continue

      self._check_attribute(att)
      self.out.write('\n&%s\n' % namelist.title)
      for card in format_list:
        card.write(self.out, att)

      # Write type-specific function  data
      type_item = att.findString('type')
      function_type = type_item.value(0)

      if function_type == 'polynomial':
        group_item = att.itemAtPath('type/polynomial-terms')
        num_groups = group_item.numberOfGroups()

        if num_groups < 1:
          print('Warning: function {} contents are empty -- skipping'.format(att.name()))
          continue

        # Get number of independent vars
        exp_item0 = group_item.item(0, 1)
        num_vars = exp_item0.numberOfValues()

        coef_list = [0.0] * num_groups
        # Create 2D list for exponentials [group][variable]
        exp_lists = [ [0 for x in range(num_vars)] for y in range(num_groups) ]

        for i in range(num_groups):
          coef_item = group_item.item(i, 0)
          coef_list[i] = coef_item.value()

          exp_item = group_item.item(i, 1)
          for j in range(num_vars):
            exp_lists[i][j] = exp_item.value(j)

        CardFormat.write_value(self.out, 'poly_coefficients', coef_list)
        if num_vars == 1:
          # For one var, can flatten to a single row of exponents
          keyword = 'poly_exponents(1,:)'
          flat_list = [exp[0] for exp in exp_lists]
          CardFormat.write_value(self.out, keyword, flat_list)
        else:
          # For the general case, write out the array
          for k in range(num_groups):
            keyword = 'poly_exponents(:, {})'.format(k+1)
            CardFormat.write_value(self.out, keyword, exp_lists[k])

        # Finish with the center value
        CardFormat('poly_refvars', item_path='type/center').write(self.out, att)
      elif function_type == 'tabular':
        # For multi-dimensional functions, the "tabular dimension" is stored as
        # the "independent-variable" item.
        ivar_item = att.itemAtPath('type/independent-variable')
        if ivar_item is not None:
          tabular_dim = int(ivar_item.value())
          CardFormat.write_value(self.out, 'tabular_dim', tabular_dim)

        group_item = att.itemAtPath('type/tabular-data')
        num_terms = group_item.numberOfGroups()
        xvals = [0.0] * num_terms
        yvals = [0.0] * num_terms
        for i in range(num_terms):
          xvals[i] = group_item.item(i, 0).value()
          yvals[i] = group_item.item(i, 1).value()
        CardFormat.write_value(self.out, 'tabular_data(1,:)', xvals)
        CardFormat.write_value(self.out, 'tabular_data(2,:)', yvals)

        # Finish with interpolation & extrapolation
        CardFormat('tabular_interp', item_path='type/interpolation').write(self.out, att)
        CardFormat('tabular_extrap', item_path='type/extrapolation').write(self.out, att)

      self.out.write('/\n')

#
  def _write_materials(self, namelist, format_list):
    """"""
    # This method is in the mixin class
    self._write_physical_materials(namelist, format_list)

# ---------------------------------------------------------------------
  def _write_electromagnetics(self, namelist, format_list):
    """Custom method for electromagnetics namelist.

    Need custom method to handle extensible group item ("source").
    """
    print('Writing', namelist.title, 'namelists')
    self._check_induction_heating_att()
    att_list = self.sim_atts.findAttributes(namelist.att_type)
    if not att_list:
      print('Warning: no {} attribute found'.format(namelist.att_type))
      return
    em_att = att_list[0]

    self._check_attribute(em_att)
    self._write_separator(namelist.separator)

    # First assemble lists of time & frequency values, from the induction-heating att
    att_list = self.sim_atts.findAttributes('induction-heating')
    if not att_list:
      print('Warning: no induction-heating attribute found')
      return
    ih_att = att_list[0]
    self._check_attribute(ih_att)

    source_item = ih_att.findGroup('source')
    if source_item is None:
      print('Warning: induction-heating attribute has no source groupitem')
      return

    num_groups = source_item.numberOfGroups()
    time_value = [None] * num_groups
    freq_value = [None] * num_groups
    for i in range(num_groups):
      time_value[i] = source_item.find(i, 'time').value()
      freq_value[i] = source_item.find(i, 'frequency').value()

    # Write the namelist
    self._start_namelist(namelist)

    for card in format_list:
      if card.keyword == 'source_times':
        if num_groups > 1:
          CardFormat.write_value(self.out, card.keyword, time_value[1:])
      elif card.keyword == 'source_frequency':
        CardFormat.write_value(self.out, card.keyword, freq_value)
      else:
        card.write(self.out, em_att)

    self._finish_namelist()

# ---------------------------------------------------------------------
  def _write_induction_coils(self, namelist, format_list):
    """Custom method for induction coils.

    Need custom method to handle extensible group item ("coils").
    """
    print('Writing', namelist.title, 'namelists')
    self._check_induction_heating_att()

    att_list = self.sim_atts.findAttributes(namelist.att_type)
    if not att_list:
      print('Internal Error: attribute \"{}\" not found'.format(namelist.att_type))
      return
    att = att_list[0]

    coils_item = att.findGroup('coils')
    if coils_item is None:
      print('Warning: {} attribute has no coils groupitem'.format(namelist.att_type))
      return

    for i in range(coils_item.numberOfGroups()):
      self._start_namelist(namelist)
      for card in format_list:
        card_item = coils_item.find(i, card.item_path)
        card.write_item(card_item, self.out)
      self._finish_namelist()

# ---------------------------------------------------------------------
  def _check_induction_heating_att(self):
    """Custom logic to validate the induction-heating attribute.

    This attribute represents the induction coils and sources.
    Special logic includes:
      - Add warning if no coils specified
      - Add warning if no source specified
      - For coils with 1 turn, the length item is not set
      - The first source item's time value is not set
    """
    att = self.sim_atts.findAttribute('induction-heating')
    if att in self.checked_attributes:
      return
    self.checked_attributes.add(att)

    # Check source item
    source_item = att.findGroup('source')
    if source_item.numberOfGroups() == 0:
      msg = 'Warning: Induction Heating specified but no current sources are defined'
      print(msg)
      self.warning_messages.append(msg)

      for i in range(source_item.numberOfGroups()):
        if i > 0:
          time_item = source_item.find(i, 'time')
          if not time_item.isValid():
            msg = 'Warning: Invalid time value row {}'.format(i+1)
            self.warning_messages.append(msg)
        freq_item = source_item.find(i, 'frequency')
        if not freq_item.isValid():
          msg = 'Warning: Invalid frequency value in row {}'.format(i+1)
          self.warning_messages.append(msg)

    # Check coils item
    coils_item = att.findGroup('coils')
    if coils_item.numberOfGroups() == 0:
      msg = 'Warning: Induction Heating specified but no coils are defined'
      print(msg)
      self.warning_messages.append(msg)

    std_names = ['nturns', 'radius', 'center', 'current']
    for i in range(coils_item.numberOfGroups()):
      for name in std_names:
        item = coils_item.find(i, name)
        if not item.isValid():
          msg = 'Warning: Induction Coil {} has invalid {} value'.format(i+1, name)
          self.warning_messages.append(msg)

      # Only check length item if more than 1 turn
      turns_item = coils_item.find(i, 'nturns')
      if turns_item.value() > 1:
        length_item = coils_item.find(i, 'length')
        if not length_item.isValid():
          msg = 'Warning: Induction Coil {} has invalid {} value'.format(i+1, name)
          self.warning_messages.append(msg)

# ---------------------------------------------------------------------
  def _setup(self):
    """"""
    # Set CardFormat conditions (categories are set in BaseWriter.__init__())
    for category in self.categories:
      CardFormat.Conditions.add(category)
    if 'Heat Transfer' in self.categories and 'Flow Flow' not in self.categories:
      CardFormat.Conditions.add(ONLY_THERMAL_ANALYSIS)

    # Get the background material attribute's id
    att_list = self.sim_atts.findAttributes('background-material')
    if att_list:
      att = att_list[0]
      ref_item = att.findComponent('background-material')
      if ref_item.isSet(0):
        background_material_att = ref_item.objectValue(0)
        self.background_material_id = background_material_att.id()

    # Find all interface_set_ids
    self.interface_set_ids = list()
    att_list = self.sim_atts.findAttributes('ht.interface')
    for att in att_list:
      reference_item = att.associations()
      if reference_item:
        self.interface_set_ids += CardFormat.get_model_entity_ids(reference_item)

    self.interface_set_ids.sort()
    print('interface_set_ids', self.interface_set_ids)
