// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate futures;
use self::futures::{BoxFuture, Future};

extern crate serde_json;
use self::serde_json::Value;

use super::director::Director;
use super::error::{Error, Result};

use std::error;

#[derive(Debug)]
/// Results from an event.
pub enum JobStatus {
    /// The event was accepted and acted upon.
    Accept,
    /// The event was deferred until a later time for the given reason.
    Defer(String),
    /// The event was rejected for the given reason.
    Reject(String),
    /// The event failed with the given error.
    Fail(Box<error::Error>),
    /// The director should be restarted.
    Restart,
    /// The event was the last one which should be processed.
    Done,
}

impl JobStatus {
    /// Combine two handler results into one.
    pub fn combine(self, other: Self) -> Self {
        match (self, other) {
            // Acceptance defers to the other.
            (JobStatus::Accept, next) |
            (next, JobStatus::Accept) => next,
            // Completion overrides any other status.
            (JobStatus::Done, _) |
            (_, JobStatus::Done) => JobStatus::Done,
            // Once completion is handled, restart takes precedence.
            (JobStatus::Restart, _) |
            (_, JobStatus::Restart) => JobStatus::Restart,
            // Deferring is next.
            (JobStatus::Defer(left), JobStatus::Defer(right)) => {
                JobStatus::Defer(format!("{}\n{}", left, right))
            },
            (defer @ JobStatus::Defer(_), _) |
            (_, defer @ JobStatus::Defer(_)) => defer,
            // Failures are handled next.
            (fail @ JobStatus::Fail(_), _) |
            (_, fail @ JobStatus::Fail(_)) => fail,
            // All we have left are rejections; combine their messages.
            (JobStatus::Reject(left), JobStatus::Reject(right)) => {
                JobStatus::Reject(format!("{}\n{}", left, right))
            },
        }
    }
}

/// A future from running a job handler.
pub type JobHandlerFuture<T> = BoxFuture<T, Error>;

/// Interface for handling events.
pub trait JobHandler {
    /// Adds the handler to a director.
    fn add_to_director<'a>(&'a self, director: &mut Director<'a>) -> Result<()>;

    /// The JSON object is passed in and acted upon.
    fn handle(&self, kind: &str, object: &Value) -> JobHandlerFuture<JobStatus>;

    /// The retry limit for a job kind.
    fn retry_limit(&self, _kind: &str) -> usize {
        5
    }

    /// The JSON object which has been retried is passed in and acted upon.
    fn handle_retry(&self, kind: &str, object: &Value, reasons: Vec<String>)
                    -> JobHandlerFuture<JobStatus> {
        if reasons.len() > self.retry_limit(kind) {
            let msg = format!("retry limit ({}) reached for {}",
                              reasons.len(),
                              kind);
            futures::finished(JobStatus::Reject(msg)).boxed()
        } else {
            self.handle(kind, object)
        }
    }
}
