// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Git hosting service traits
//!
//! The traits in this module are meant to help keep service-specific knowledge away from the
//! workflow implementation.
//!
//! These traits might be split into a separate library in the future.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate kitware_git;
use self::kitware_git::{CommitId, GitContext, Identity};

use std::error;
use std::io;
use std::rc::Rc;

/// A commit status created by a `HostedCommit` or `HostedMergeRequest`.
pub struct CommitStatus<'a> {
    /// The commit the status applies to.
    pub commit: &'a HostedCommit,
    /// The state of the commit status.
    pub state: CommitStatusState,
    /// The refname the commit status is intended for.
    pub refname: Option<&'a str>,
    /// The name of the status check.
    pub name: &'a str,
    /// A description for the status.
    pub description: &'a str,
}

/// A commit hosted on the service provider.
pub trait HostedCommit {
    /// The repository where the commit lives.
    fn project(&self) -> &HostedRepo;
    /// The refname for the commit (if available).
    fn refname(&self) -> Option<&str>;
    /// The object id of the commit.
    fn id(&self) -> &CommitId;

    /// Create a commit status for the commit.
    fn create_commit_status<'a>(&'a self, state: CommitStatusState, name: &'a str,
                                description: &'a str)
                                -> CommitStatus<'a> {
        CommitStatus {
            commit: self._self_ref_hack(),
            state: state,
            refname: self.refname(),
            name: name,
            description: description,
        }
    }

    /// A hack to get a reference to a `HostedCommit` for the default implementation of
    /// `create_commit_status`.
    fn _self_ref_hack(&self) -> &HostedCommit;
}

/// A repository hosted on the service.
pub trait HostedRepo {
    /// The name of the project.
    fn name(&self) -> &str;

    /// The URL which should be used to fetch from the repository.
    ///
    /// Whether this uses HTTPS or SSH is dependent on the service, but it should not require
    /// interaction in order to use (whether through an SSH key or administrator privileges).
    fn url(&self) -> &str;

    /// The ID of the repository.
    fn id(&self) -> u64;
}

/// An issue on the service.
pub trait HostedIssue {
    /// The source repository for the merge request.
    fn repo(&self) -> &HostedRepo;
    /// The internal ID of the issue.
    ///
    /// Service-specific.
    fn id(&self) -> u64;
    /// The URL of the merge request.
    fn url(&self) -> &str;
    /// The description for the issue.
    fn description(&self) -> &str;
    /// The labels for the issue.
    fn labels(&self) -> &[String];
    /// The milestone for the issue.
    fn milestone(&self) -> Option<&str>;
    /// The username of the assignee.
    fn assignee(&self) -> Option<&str>;
}

/// A merge request on the service.
pub trait HostedMergeRequest {
    /// The source repository for the merge request.
    fn source_repo(&self) -> &HostedRepo;
    /// The name of the branch requested for merging.
    fn source_branch(&self) -> &str;
    /// The repository the merge request will be merged into.
    fn target_repo(&self) -> &HostedRepo;
    /// The target branch for the request.
    fn target_branch(&self) -> &str;
    /// The internal ID of the merge request.
    ///
    /// Service-specific.
    fn id(&self) -> u64;
    /// The URL of the merge request.
    fn url(&self) -> &str;
    /// Whether the merge request is a "work-in-progress" or not.
    fn work_in_progress(&self) -> bool;
    /// The description for the merge request.
    fn description(&self) -> &str;
    /// The previous commit of the merge request (if available).
    ///
    /// This is particularly important for the stage action. Not so important otherwise.
    fn old_commit(&self) -> Option<&HostedCommit>;
    /// The commit which has been requested for merging.
    fn commit(&self) -> &HostedCommit;
    /// The author of the merge request.
    fn author(&self) -> &HostedUser;
    /// Build a string which may be used in a comment to refer to the merge request.
    fn reference(&self) -> String;

    /// Create a commit status for the merge request.
    fn create_commit_status<'a>(&'a self, state: CommitStatusState, name: &'a str,
                                description: &'a str)
                                -> CommitStatus<'a> {
        CommitStatus {
            commit: self.commit(),
            state: state,
            refname: Some(self.source_branch()),
            name: name,
            description: description,
        }
    }

    /// Convenience method for getting a commit id structure for the old commit.
    fn old_commit_id(&self) -> Option<&CommitId> {
        self.old_commit().map(|c| c.id())
    }

    /// Convenience method for getting a commit id structure for the commit.
    fn commit_id(&self) -> &CommitId {
        self.commit().id()
    }
}

#[derive(Clone, Copy, PartialEq, Eq, Debug)]
/// States for a commit status.
pub enum CommitStatusState {
    /// The check is expected, but has not started yet.
    Pending,
    /// The check is currently running.
    Running,
    /// The check is a success.
    Success,
    /// The check is a failure.
    Failed,
}

/// A commit status for a specific commit.
pub trait HostedCommitStatus {
    /// The state of the commit status.
    fn state(&self) -> &CommitStatusState;
    /// The author of the commit status.
    fn author(&self) -> &HostedUser;
    /// The refname of the commit (if applicable).
    fn refname(&self) -> Option<&str>;
    /// The name of the check being performed.
    fn name(&self) -> &str;
    /// A description of the check.
    fn description(&self) -> &str;
}

/// A user on the service.
pub trait HostedUser {
    /// The internal ID of the user.
    ///
    /// Service-specific.
    fn id(&self) -> u64;
    /// The username on the service of the user (used for mentioning the user).
    fn handle(&self) -> &str;
    /// The real name of the user.
    fn name(&self) -> &str;
    /// The email address of the user.
    fn email(&self) -> &str;

    /// Convenience method for getting an identity for the user.
    fn identity(&self) -> Identity {
        Identity::new(self.name(), self.email())
    }
}

/// A membership of a user to a project, team, or group.
pub trait HostedMembership {
    /// The user with access.
    fn user(&self) -> &HostedUser;

    /// The access level of the user.
    fn access_level(&self) -> u64;

    /// When the membership expires (if at all).
    fn expiration(&self) -> Option<&DateTime<UTC>>;
}

/// A comment on the service.
pub trait HostedComment {
    /// The ID of the note.
    fn id(&self) -> u64;
    /// Indicates whether the comment is autogenerated (via activity or mentions) or not.
    fn is_system(&self) -> bool;
    /// Indicates whether the comment indicates a branch update or not.
    ///
    /// This is used to separate the comment stream into before and after for an update to its
    /// source topic.
    fn is_branch_update(&self) -> bool;
    /// When the comment was created.
    fn created_at(&self) -> &DateTime<UTC>;
    /// When the comment was last updated.
    fn updated_at(&self) -> &DateTime<UTC>;
    /// The author of the comment.
    fn author(&self) -> &HostedUser;
    /// The content of the comment.
    fn content(&self) -> &str;
}

/// An award on the service.
pub trait HostedAward {
    /// The name of the award.
    fn name(&self) -> &str;
    /// The author of the award.
    fn author(&self) -> &HostedUser;
}

quick_error! {
    #[derive(Debug)]
    /// An error from a hosting service.
    pub enum HostError {
        /// An error occurred when communicating with the service.
        Communication(err: Box<error::Error>) {
            display("service communication error: {:?}", err)
            cause(err.as_ref())
        }
        /// An error occurred on the host itself.
        Host(err: Box<error::Error>) {
            display("service error: {:?}", err)
            cause(err.as_ref())
        }
        /// The service returned an invalidly formatted result.
        InvalidFormat(err: Box<error::Error>) {
            display("invalid format error: {:?}", err)
            cause(err.as_ref())
        }
    }
}

/// A result from a hosting service request.
pub type HostedResult<T> = Result<T, HostError>;

/// A project hosted on a service.
pub struct HostedProject {
    /// The name of the project.
    pub name: String,
    /// The service the project is hosted on.
    pub service: Rc<HostingService>,
}

impl HostedProject {
    /// Add a member to the project.
    pub fn add_member(&self, user: &HostedUser, level: u64) -> HostedResult<()> {
        self.service.add_member(&self.name, user, level)
    }

    /// Get the membership list of the project.
    pub fn members(&self) -> HostedResult<Vec<Box<HostedMembership>>> {
        self.service.members(&self.name)
    }

    /// Add a wehhook to the project.
    pub fn add_hook(&self, url: &str) -> HostedResult<()> {
        self.service.add_hook(&self.name, url)
    }

    /// Get a commit for a project.
    pub fn commit(&self, commit: &CommitId) -> HostedResult<Box<HostedCommit>> {
        self.service.commit(&self.name, commit)
    }

    /// Get a merge request on a project.
    pub fn merge_request(&self, id: u64) -> HostedResult<Box<HostedMergeRequest>> {
        self.service.merge_request(&self.name, id)
    }
}

/// A hosting service.
pub trait HostingService {
    /// Fetch a commit into a given git context.
    ///
    /// The default implementation requires that the commit have a valid refname, otherwise the
    /// fetch will fail.
    fn fetch_commit(&self, git: &GitContext, commit: &HostedCommit) -> Result<(), io::Error> {
        git.fetch(commit.project().url(), &[commit.refname().unwrap()])
    }
    /// Fetch a merge request into a given git context.
    fn fetch_mr(&self, git: &GitContext, mr: &HostedMergeRequest) -> Result<(), io::Error> {
        git.fetch(mr.source_repo().url(), &[mr.source_branch()])
    }

    /// The user the service is acting as.
    fn service_user(&self) -> &HostedUser;

    /// Add a user to a project with the given access level.
    fn add_member(&self, project: &str, user: &HostedUser, level: u64) -> HostedResult<()>;

    /// Return the list of memberships to a project.
    ///
    /// This should include *all* memberships which provide a permission to the project above that
    /// of an anonymous user.
    fn members(&self, project: &str) -> HostedResult<Vec<Box<HostedMembership>>>;

    /// Register a webhook for a project.
    ///
    /// The webhook should listen for all events that matter for the workflow (e.g., pushes,
    /// updates to merge requests, etc.).
    fn add_hook(&self, project: &str, url: &str) -> HostedResult<()>;

    /// Get a user by name.
    fn user(&self, user: &str) -> HostedResult<Box<HostedUser>>;
    /// Get a commit for a project.
    fn commit(&self, project: &str, commit: &CommitId) -> HostedResult<Box<HostedCommit>>;
    /// Get an issue on a project.
    fn issue(&self, project: &str, id: u64) -> HostedResult<Box<HostedIssue>>;
    /// Get a merge request on a project.
    fn merge_request(&self, project: &str, id: u64) -> HostedResult<Box<HostedMergeRequest>>;
    /// Get a repository by name.
    fn repo(&self, project: &str) -> HostedResult<Box<HostedRepo>>;

    /// Get a user by ID.
    fn user_by_id(&self, user: u64) -> HostedResult<Box<HostedUser>>;
    /// Get a commit for a project by ID.
    fn commit_by_id(&self, project: u64, commit: &CommitId) -> HostedResult<Box<HostedCommit>>;
    /// Get an issue on a project by ID.
    fn issue_by_id(&self, project: u64, id: u64) -> HostedResult<Box<HostedIssue>>;
    /// Get a merge request on a project by ID.
    fn merge_request_by_id(&self, project: u64, id: u64) -> HostedResult<Box<HostedMergeRequest>>;
    /// Get a repository by ID.
    fn repo_by_id(&self, project: u64) -> HostedResult<Box<HostedRepo>>;

    /// Get comments for an issue.
    ///
    /// Comments are ordered from oldest to newest.
    fn get_issue_comments(&self, issue: &HostedIssue) -> HostedResult<Vec<Box<HostedComment>>>;
    /// Add a comment to an issue.
    fn post_issue_comment(&self, issue: &HostedIssue, content: &str) -> HostedResult<()>;
    /// Get comments for a merge request.
    ///
    /// Comments are ordered from oldest to newest.
    fn get_mr_comments(&self, mr: &HostedMergeRequest) -> HostedResult<Vec<Box<HostedComment>>>;
    /// Add a comment to a merge request.
    fn post_mr_comment(&self, mr: &HostedMergeRequest, content: &str) -> HostedResult<()>;
    /// Add a comment to a commit.
    fn post_commit_comment(&self, commit: &HostedCommit, content: &str) -> HostedResult<()>;
    /// Get the latest commit statuses for a commit.
    fn get_commit_statuses(&self, commit: &HostedCommit)
                           -> HostedResult<Vec<Box<HostedCommitStatus>>>;
    /// Create a commit status.
    fn post_commit_status(&self, status: CommitStatus) -> HostedResult<()>;

    /// Award an emoji to a merge request comment.
    fn get_mr_comment_awards(&self, mr: &HostedMergeRequest, comment: &HostedComment)
                             -> HostedResult<Vec<Box<HostedAward>>>;
    /// Award an emoji to a merge request comment.
    fn award_mr_comment(&self, mr: &HostedMergeRequest, comment: &HostedComment, award: &str)
                        -> HostedResult<()>;
}
