// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate itertools;
use self::itertools::Itertools;

extern crate gitlab;
use self::gitlab::{CommitStatusInfo, Gitlab, GitlabResult};

extern crate kitware_git;
use self::kitware_git::CommitId;

extern crate regex;
use self::regex::Regex;

use super::traits::*;

use std::cmp::{self, Ord, PartialOrd};

lazy_static! {
    static ref MR_UPDATE_RE: Regex =
        Regex::new("^Added [0-9][0-9]* (new )?commits?:\n\
                    \n\
                    (\\* [0-9a-f]* - [^\n]*\n)*\
                    (\n\\[Compare with previous versions\\]\\(.*\\))?\
                    $'").unwrap();
}

struct GitlabCommit {
    repo: GitlabRepo,
    commit_id: CommitId,
}

impl HostedCommit for GitlabCommit {
    fn project(&self) -> &HostedRepo {
        &self.repo
    }

    fn refname(&self) -> Option<&str> {
        // TODO
        None
    }

    fn id(&self) -> &CommitId {
        &self.commit_id
    }

    fn _self_ref_hack(&self) -> &HostedCommit {
        self
    }
}

struct GitlabRepo {
    project: gitlab::Project,
}

impl HostedRepo for GitlabRepo {
    fn name(&self) -> &str {
        &self.project.path_with_namespace
    }

    fn url(&self) -> &str {
        &self.project.ssh_url_to_repo
    }

    fn id(&self) -> u64 {
        self.project.id.value()
    }
}

struct GitlabIssue {
    repo: GitlabRepo,
    issue: gitlab::Issue,
}

impl HostedIssue for GitlabIssue {
    fn repo(&self) -> &HostedRepo {
        &self.repo
    }

    fn id(&self) -> u64 {
        self.issue.id.value()
    }

    fn url(&self) -> &str {
        &self.issue.web_url
    }

    fn description(&self) -> &str {
        &self.issue.description
    }

    fn labels(&self) -> &[String] {
        &self.issue.labels
    }

    fn milestone(&self) -> Option<&str> {
        self.issue.milestone.as_ref().map(|m| m.title.as_str())
    }

    fn assignee(&self) -> Option<&str> {
        self.issue.assignee.as_ref().map(|u| u.username.as_str())
    }
}

struct GitlabMergeRequest {
    source_repo: GitlabRepo,
    target_repo: GitlabRepo,
    mr: gitlab::MergeRequest,
    commit: GitlabCommit,
    author: GitlabUser,
}

impl HostedMergeRequest for GitlabMergeRequest {
    fn source_repo(&self) -> &HostedRepo {
        &self.source_repo
    }

    fn source_branch(&self) -> &str {
        &self.mr.source_branch
    }

    fn target_repo(&self) -> &HostedRepo {
        &self.target_repo
    }

    fn target_branch(&self) -> &str {
        &self.mr.target_branch
    }

    fn id(&self) -> u64 {
        self.mr.id.value()
    }

    fn url(&self) -> &str {
        &self.mr.web_url
    }

    fn work_in_progress(&self) -> bool {
        self.mr.work_in_progress
    }

    fn description(&self) -> &str {
        self.mr
            .description
            .as_ref()
            .map(|s| s.as_str())
            .unwrap_or("")
    }

    fn old_commit(&self) -> Option<&HostedCommit> {
        // TODO
        None
    }

    fn commit(&self) -> &HostedCommit {
        &self.commit
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn reference(&self) -> String {
        format!("!{}", self.mr.id)
    }
}

struct GitlabCommitStatus {
    commit_status: gitlab::CommitStatus,
    state: CommitStatusState,
    author: GitlabUser,
}

impl GitlabCommitStatus {
    fn new(status: gitlab::CommitStatus, author: gitlab::UserFull) -> Self {
        GitlabCommitStatus {
            state: convert_state_from_gitlab(&status.status),
            commit_status: status,
            author: GitlabUser::new(author),
        }
    }
}

impl HostedCommitStatus for GitlabCommitStatus {
    fn state(&self) -> &CommitStatusState {
        &self.state
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn refname(&self) -> Option<&str> {
        self.commit_status
            .ref_
            .as_ref()
            .map(|ref r| r.as_str())
    }

    fn name(&self) -> &str {
        &self.commit_status.name
    }

    fn description(&self) -> &str {
        self.commit_status
            .description
            .as_ref()
            .map(|ref d| d.as_str())
            .unwrap_or("")
    }
}

fn gitlab_status_info<'a>(status: CommitStatus<'a>) -> CommitStatusInfo<'a> {
    CommitStatusInfo {
        refname: status.refname,
        name: Some(status.name),
        target_url: None,
        description: Some(status.description),
    }
}

pub struct GitlabUser {
    user: gitlab::UserFull,
}

impl GitlabUser {
    /// Create a new GitlabUser from a user instance.
    pub fn new(user: gitlab::UserFull) -> Self {
        GitlabUser {
            user: user,
        }
    }
}

impl HostedUser for GitlabUser {
    fn id(&self) -> u64 {
        self.user.id.value()
    }

    fn handle(&self) -> &str {
        &self.user.username
    }

    fn name(&self) -> &str {
        &self.user.name
    }

    fn email(&self) -> &str {
        &self.user.email
    }
}

struct GitlabMembership {
    user: GitlabUser,
    access_level: u64,
    expiration: Option<DateTime<UTC>>,
}

impl HostedMembership for GitlabMembership {
    fn user(&self) -> &HostedUser {
        &self.user
    }

    fn access_level(&self) -> u64 {
        self.access_level
    }

    fn expiration(&self) -> Option<&DateTime<UTC>> {
        self.expiration.as_ref()
    }
}

struct GitlabNote {
    note: gitlab::Note,
    author: GitlabUser,
}

impl GitlabNote {
    fn new(gitlab: &Gitlab, note: gitlab::Note) -> GitlabResult<Self> {
        let author = try!(gitlab.user::<gitlab::UserFull>(note.author.id));

        Ok(GitlabNote {
            note: note,
            author: GitlabUser::new(author),
        })
    }
}

impl PartialEq for GitlabNote {
    fn eq(&self, other: &Self) -> bool {
        self.id() == other.id()
    }
}

impl Eq for GitlabNote {}

impl PartialOrd for GitlabNote {
    fn partial_cmp(&self, other: &Self) -> Option<cmp::Ordering> {
        Some(self.cmp(other))
    }
}

impl Ord for GitlabNote {
    fn cmp(&self, other: &Self) -> cmp::Ordering {
        self.id().cmp(&other.id())
    }
}

impl HostedComment for GitlabNote {
    fn id(&self) -> u64 {
        self.note.id.value()
    }

    fn is_system(&self) -> bool {
        self.note.system
    }

    fn is_branch_update(&self) -> bool {
        MR_UPDATE_RE.is_match(&self.note.body)
    }

    fn created_at(&self) -> &DateTime<UTC> {
        &self.note.created_at
    }

    fn updated_at(&self) -> &DateTime<UTC> {
        &self.note.updated_at
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn content(&self) -> &str {
        &self.note.body
    }
}

struct GitlabAward {
    award: gitlab::AwardEmoji,
    author: GitlabUser,
}

impl GitlabAward {
    fn new(award: gitlab::AwardEmoji, author: gitlab::UserFull) -> Self {
        GitlabAward {
            award: award,
            author: GitlabUser::new(author),
        }
    }
}

impl HostedAward for GitlabAward {
    fn name(&self) -> &str {
        &self.award.name
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }
}

/// Structure used to communicate with a GitLab instance.
pub struct GitlabService {
    gitlab: Gitlab,
    user: GitlabUser,
}

impl GitlabService {
    /// Create a new GitLab communication channel.
    pub fn new(gitlab: Gitlab) -> HostedResult<Self> {
        Ok(GitlabService {
            user: GitlabUser::new(try!(gitlab.current_user())),
            gitlab: gitlab,
        })
    }

    fn commit_from_project(&self, project: gitlab::Project, commit: &CommitId)
                           -> HostedResult<Box<HostedCommit>> {
        let commit = try!(self.gitlab.commit(project.id, &commit.0));

        Ok(Box::new(GitlabCommit {
            repo: GitlabRepo {
                project: project,
            },
            commit_id: CommitId::new(commit.id.value()),
        }))
    }

    fn issue_from_project(&self, project: gitlab::Project, id: u64)
                          -> HostedResult<Box<HostedIssue>> {
        let issue = try!(self.gitlab.issue(project.id, gitlab::IssueId::new(id)));

        Ok(Box::new(GitlabIssue {
            repo: GitlabRepo {
                project: project,
            },
            issue: issue,
        }))
    }

    fn merge_request_from_project(&self, project: gitlab::Project, id: u64)
                                  -> HostedResult<Box<HostedMergeRequest>> {
        let mr = try!(self.gitlab
            .merge_request(project.id, gitlab::MergeRequestId::new(id)));
        let source_project = try!(self.gitlab.project(mr.source_project_id));
        let source_project_commit = try!(self.gitlab.project(mr.source_project_id));
        let source_branch = try!(self.gitlab.branch(mr.source_project_id, &mr.source_branch));
        let commit = try!(self.gitlab.commit(mr.source_project_id,
                                             &source_branch.commit.unwrap().id.value()));
        let author = try!(self.gitlab.user(mr.author.id));

        Ok(Box::new(GitlabMergeRequest {
            source_repo: GitlabRepo {
                project: source_project,
            },
            target_repo: GitlabRepo {
                project: project,
            },
            mr: mr,
            commit: GitlabCommit {
                repo: GitlabRepo {
                    project: source_project_commit,
                },
                commit_id: CommitId::new(commit.id.value()),
            },
            author: GitlabUser::new(author),
        }))
    }

    fn sort_notes(&self, notes: Vec<gitlab::Note>) -> HostedResult<Vec<Box<HostedComment>>> {
        Ok(try!(notes.into_iter()
            .map(|note| GitlabNote::new(&self.gitlab, note))
            .collect::<Result<Vec<_>, _>>())
            .into_iter()
            .sorted()
            .into_iter()
            .map(|note| Box::new(note) as Box<HostedComment>)
            .collect())
    }
}

impl HostingService for GitlabService {
    fn service_user(&self) -> &HostedUser {
        &self.user
    }

    fn add_member(&self, project: &str, user: &HostedUser, level: u64) -> HostedResult<()> {
        let project = try!(self.gitlab.project_by_name(project));

        try!(self.gitlab
            .add_user_to_project(project.id, gitlab::UserId::new(user.id()), level.into()));

        Ok(())
    }

    fn members(&self, project: &str) -> HostedResult<Vec<Box<HostedMembership>>> {
        let project = try!(self.gitlab.project_by_name(project));

        let project_members = try!(self.gitlab.project_members(project.id));
        let namespace_id = project.namespace.owner_id();
        let group_members = if let gitlab::NamespaceId::Group(gid) = namespace_id {
            try!(self.gitlab.group_members(gid))
        } else {
            vec![]
        };

        project_members.into_iter()
            // Chain with the group membership list.
            .chain(group_members.into_iter())
            // Sort by the user IDs.
            .sorted_by(|a, b| Ord::cmp(&a.id.value(), &b.id.value()))
            .into_iter()
            // Group all members in the list by the ID.
            .group_by(|member| member.id)
            .into_iter()
            // Create GitlabMembership structs.
            .map(|(_, members)| {
                // Find the maximum access level from any membership.
                let max_access = members.max_by_key(|member| member.access_level)
                    .unwrap();

                Ok(Box::new(GitlabMembership {
                    user: GitlabUser::new(try!(self.gitlab.user(max_access.id))),
                    access_level: max_access.access_level,
                    expiration: max_access.expires_at,
                }) as Box<HostedMembership>)
            })
            .collect()
    }

    fn add_hook(&self, project: &str, url: &str) -> HostedResult<()> {
        let project = try!(self.gitlab.project_by_name(project));
        let events = gitlab::WebhookEvents::new()
            .with_issues()
            .with_merge_requests()
            .with_note()
            .with_push();

        let _ = try!(self.gitlab.add_hook(project.id, url, events));

        Ok(())
    }

    fn user(&self, user: &str) -> HostedResult<Box<HostedUser>> {
        Ok(Box::new(GitlabUser::new(try!(self.gitlab.user_by_name(user)))))
    }

    fn commit(&self, project: &str, commit: &CommitId) -> HostedResult<Box<HostedCommit>> {
        let project = try!(self.gitlab.project_by_name(project));
        self.commit_from_project(project, commit)
    }

    fn issue(&self, project: &str, id: u64) -> HostedResult<Box<HostedIssue>> {
        let project = try!(self.gitlab.project_by_name(project));
        self.issue_from_project(project, id)
    }

    fn merge_request(&self, project: &str, id: u64) -> HostedResult<Box<HostedMergeRequest>> {
        let project = try!(self.gitlab.project_by_name(project));
        self.merge_request_from_project(project, id)
    }

    fn repo(&self, project: &str) -> HostedResult<Box<HostedRepo>> {
        Ok(Box::new(GitlabRepo {
            project: try!(self.gitlab.project_by_name(project)),
        }))
    }

    fn user_by_id(&self, user: u64) -> HostedResult<Box<HostedUser>> {
        Ok(Box::new(GitlabUser::new(try!(self.gitlab.user(gitlab::UserId::new(user))))))
    }

    fn commit_by_id(&self, project: u64, commit: &CommitId) -> HostedResult<Box<HostedCommit>> {
        let project = try!(self.gitlab.project(gitlab::ProjectId::new(project)));
        self.commit_from_project(project, commit)
    }

    fn issue_by_id(&self, project: u64, id: u64) -> HostedResult<Box<HostedIssue>> {
        let project = try!(self.gitlab.project(gitlab::ProjectId::new(project)));
        self.issue_from_project(project, id)
    }

    fn merge_request_by_id(&self, project: u64, id: u64) -> HostedResult<Box<HostedMergeRequest>> {
        let project = try!(self.gitlab.project(gitlab::ProjectId::new(project)));
        self.merge_request_from_project(project, id)
    }

    fn repo_by_id(&self, project: u64) -> HostedResult<Box<HostedRepo>> {
        Ok(Box::new(GitlabRepo {
            project: try!(self.gitlab.project(gitlab::ProjectId::new(project))),
        }))
    }

    fn get_issue_comments(&self, issue: &HostedIssue) -> HostedResult<Vec<Box<HostedComment>>> {
        let notes = try!(self.gitlab.issue_notes(gitlab::ProjectId::new(issue.repo().id()),
                                                 gitlab::IssueId::new(issue.id())));

        self.sort_notes(notes)
    }

    fn post_issue_comment(&self, issue: &HostedIssue, content: &str) -> HostedResult<()> {
        try!(self.gitlab.create_issue_note(gitlab::ProjectId::new(issue.repo().id()),
                                           gitlab::IssueId::new(issue.id()),
                                           content));

        Ok(())
    }

    fn get_mr_comments(&self, mr: &HostedMergeRequest) -> HostedResult<Vec<Box<HostedComment>>> {
        let notes = try!(self.gitlab
            .merge_request_notes(gitlab::ProjectId::new(mr.target_repo().id()),
                                 gitlab::MergeRequestId::new(mr.id())));

        self.sort_notes(notes)
    }

    fn post_mr_comment(&self, mr: &HostedMergeRequest, content: &str) -> HostedResult<()> {
        try!(self.gitlab.create_merge_request_note(gitlab::ProjectId::new(mr.target_repo().id()),
                                                   gitlab::MergeRequestId::new(mr.id()),
                                                   content));

        Ok(())
    }

    fn post_commit_comment(&self, commit: &HostedCommit, content: &str) -> HostedResult<()> {
        try!(self.gitlab.create_commit_comment(gitlab::ProjectId::new(commit.project().id()),
                                               commit.id().as_str(),
                                               content));

        Ok(())
    }

    fn get_commit_statuses(&self, commit: &HostedCommit)
                           -> HostedResult<Vec<Box<HostedCommitStatus>>> {
        try!(self.gitlab
            .commit_statuses(gitlab::ProjectId::new(commit.project().id()),
                             commit.id().as_str()))
            .into_iter()
            .map(|status| {
                Ok(try!(self.gitlab
                    .user(status.author.id)
                    .map(|author| {
                        let gitlab_status = GitlabCommitStatus::new(status, author);
                        Box::new(gitlab_status) as Box<HostedCommitStatus>
                    })))
            })
            .collect::<Result<Vec<_>, _>>()
    }

    fn post_commit_status(&self, status: CommitStatus) -> HostedResult<()> {
        try!(self.gitlab
            .create_commit_status(gitlab::ProjectId::new(status.commit.project().id()),
                                  status.commit.id().as_str(),
                                  convert_state_to_gitlab(&status.state),
                                  &gitlab_status_info(status)));

        Ok(())
    }

    fn get_mr_comment_awards(&self, mr: &HostedMergeRequest, comment: &HostedComment)
                             -> HostedResult<Vec<Box<HostedAward>>> {
        try!(self.gitlab
            .merge_request_note_awards(gitlab::ProjectId::new(mr.target_repo().id()),
                                       gitlab::MergeRequestId::new(mr.id()),
                                       gitlab::NoteId::new(comment.id())))
            .into_iter()
            .map(|award| {
                Ok(try!(self.gitlab
                    .user(award.user.id)
                    .map(|author| {
                        let gitlab_award = GitlabAward::new(award, author);
                        Box::new(gitlab_award) as Box<HostedAward>
                    })))
            })
            .collect::<Result<Vec<_>, _>>()
    }

    fn award_mr_comment(&self, mr: &HostedMergeRequest, comment: &HostedComment, award: &str)
                        -> HostedResult<()> {
        try!(self.gitlab
            .award_merge_request_note(gitlab::ProjectId::new(mr.target_repo().id()),
                                      gitlab::MergeRequestId::new(mr.id()),
                                      gitlab::NoteId::new(comment.id()),
                                      award));

        Ok(())
    }
}

fn convert_state_to_gitlab(state: &CommitStatusState) -> gitlab::StatusState {
    match *state {
        CommitStatusState::Pending => gitlab::StatusState::Pending,
        CommitStatusState::Running => gitlab::StatusState::Running,
        CommitStatusState::Success => gitlab::StatusState::Success,
        CommitStatusState::Failed => gitlab::StatusState::Failed,
    }
}

fn convert_state_from_gitlab(state: &gitlab::StatusState) -> CommitStatusState {
    match *state {
        gitlab::StatusState::Canceled |
        gitlab::StatusState::Pending => CommitStatusState::Pending,
        gitlab::StatusState::Running => CommitStatusState::Running,
        gitlab::StatusState::Success => CommitStatusState::Success,
        gitlab::StatusState::Failed => CommitStatusState::Failed,
    }
}

impl From<gitlab::Error> for HostError {
    fn from(error: gitlab::Error) -> Self {
        let wrap_err = Box::new(error);

        match *wrap_err.as_ref() {
            gitlab::Error::Ease(_) |
            gitlab::Error::UrlParse(_) => HostError::Communication(wrap_err),
            gitlab::Error::Gitlab(_) => HostError::Host(wrap_err),
            gitlab::Error::Deserialize(_) => HostError::InvalidFormat(wrap_err),
        }
    }
}
