// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `merge` action.
//!
//! TODO

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate itertools;
use self::itertools::Itertools;

extern crate kitware_git;
use self::kitware_git::{CommitId, GitContext, Identity, MergeResult, MergeStatus};

use super::super::host::*;
use super::super::utils::Trailer;

use std::io::{self, Write};
use std::iter;

quick_error! {
    #[derive(Debug)]
    /// An error within the `merge` action.
    pub enum Error {
        /// An error occurred while performing a git action.
        Git(err: io::Error) {
            cause(err)
            display("git error: {:?}", err)
            from()
        }
        /// An error occurred while communicating with the hosting service.
        Host(err: HostError) {
            cause(err)
            display("hosting error: {:?}", err)
            from()
        }
    }
}

/// Implementation of the `merge` action.
///
/// TODO
pub struct Merge {
    branch: String,
    ctx: GitContext,
    project: HostedProject,
    quiet: bool,
    log_limit: Option<usize>,
}

impl Merge {
    /// Create a new stage action.
    pub fn new<B: ToString>(ctx: GitContext, branch: B, project: HostedProject) -> Self {
        Merge {
            branch: branch.to_string(),
            ctx: ctx,
            project: project,
            quiet: false,
            log_limit: None,
        }
    }

    /// Reduce the number of comments made by the stage action.
    ///
    /// The comments created by this action can be a bit much. This reduces the comments to those
    /// which are errors or are important.
    pub fn quiet(&mut self) -> &mut Self {
        self.quiet = true;
        self
    }

    /// Limit the number of log entries in merge commit messages.
    ///
    /// Anything beyond this limit (if present) is elided.
    pub fn log_limit(&mut self, log_limit: Option<usize>) -> &mut Self {
        self.log_limit = log_limit;
        self
    }

    /// Merge a merge request into the branch.
    ///
    /// Information for the merge commit is gathered from the comment stream as well as the merge
    /// request itself. Comments from before the last update are ignored since they do not apply to
    /// the latest incarnation of the topic.
    pub fn merge_mr(&mut self, mr: &HostedMergeRequest, who: &Identity, when: &DateTime<UTC>)
                    -> Result<(), Error> {
        if mr.work_in_progress() {
            self.send_mr_comment(mr,
                                 "This merge request is marked as a Work in Progress and may not \
                                  be merged. Please remove the Work in Progress state first.");
            return Ok(());
        }

        // Fetch the commit into the stager's git context.
        try!(self.project.service.fetch_mr(&self.ctx, mr));

        let branch_id = CommitId::new(&self.branch);

        // Determine if the topic is mergeable at all.
        let merge_status = try!(self.ctx.mergeable(&branch_id, mr.commit().id()));
        let bases = if let MergeStatus::Mergeable(bases) = merge_status {
            bases
        } else {
            self.send_mr_comment(mr, &unmerged_status_message(&merge_status));
            return Ok(());
        };

        // Prepare a work area to perform the actual merge.
        let workarea = try!(self.ctx.prepare(&branch_id));
        let merge_result = try!(workarea.setup_merge(&bases, &branch_id, mr.commit().id()));
        let mut merge_command = match merge_result {
            MergeResult::Conflict(conflicts) => {
                let mut conflict_paths = conflicts.iter()
                    .map(|conflict| conflict.path().to_string_lossy())
                    .dedup();
                self.send_mr_comment(mr,
                                     &format!("This merge request contains conflicts in the \
                                               following paths:  \n`{}`",
                                              conflict_paths.join("`  \n`")));
                return Ok(());
            },
            MergeResult::Ready(command) => command,
            MergeResult::_Phantom(_) => unreachable!(),
        };

        // Add authorship information to the commit message. Committer information is provided by
        // the default git environment.
        merge_command.env("GIT_AUTHOR_NAME", &who.name)
            .env("GIT_AUTHOR_EMAIL", &who.email)
            .env("GIT_AUTHOR_DATE", when.to_rfc2822());

        // Gather trailer information from the merge request.
        let comments = try!(self.project.service.get_mr_comments(mr));
        let all_trailers = try!(comments.into_iter()
            // Look at comments from newest to oldest.
            .rev()
            // Stop when we have a branch update comment.
            //
            // TODO: This should instead probably be "newer than the last branch update" since it
            // looks like not all hosting services (e.g., Github) support seeing the "system"
            // comments via the API. However, it seems that the `updated_at` field provided by
            // Gitlab and Github changes with things like editing the description. May require
            // feature requests to Github and a patch to Gitlab.
            .take_while(|comment| !comment.is_branch_update())
            // Ignore system comments.
            .filter(|comment| !comment.is_system())
            // Gather information from each comment.
            .map(|comment| {
                // TODO: Get awards from the MR itself.
                let awards = try!(self.project.service.get_mr_comment_awards(mr, comment.as_ref()));
                // Parse trailers from the comment itself.
                Ok(parse_comment_for_trailers(self.project.service.as_ref(), comment.as_ref())
                    .into_iter()
                    // Create trailers out of awards on the comment.
                    .chain(awards.into_iter()
                        .filter_map(|award| parse_awards_as_trailers(award.as_ref())))
                    .collect::<Vec<_>>())
            })
            .collect::<Result<Vec<_>, Error>>())
            // Now that we have all the trailers, gather them up.
            .into_iter()
            // Put them back into chronological order.
            .rev()
            // Put all of the trailers together into a single vector.
            .flatten()
            // Remove duplicate trailers.
            .unique();

        // TODO: Look for Rejected-by trailers.
        // let rejection_users = all_trailers.iter()
        //     .filter(|trailer| trailer.token == "Rejected-by")
        //     .map(|trailer| trailer.value)
        //     .collect::<Vec<_>>();
        //
        // if !rejection_users.is_empty() {
        //     self.send_mr_comment(mr, format!("This merge request may not be merged because it has \
        //                                       been rejected by the following users: {}. Please \
        //                                       work with them to resolve their comments first.",
        //                                      rejection_users.join(", ")));
        //     return Ok(());
        // }

        // TODO: Check for a required set of trailers.
        // TODO: Check for assignee approval.

        let mut merge_process = try!(merge_command.spawn());
        let commit_message = try!(self.build_commit_message(mr, all_trailers));
        try!(write!(merge_process.stdin.as_mut().unwrap(), "{}", commit_message));
        let merge_commit = try!(merge_process.wait_with_output());
        try!(kitware_git::check_status(merge_commit.status, "merge commit failed"));
        let commit_id = String::from_utf8_lossy(&merge_commit.stdout);

        let push = try!(self.ctx
            .git()
            .arg("push")
            .arg("--porcelain")
            .arg("origin")
            .arg(format!("{}:{}", commit_id, self.branch))
            .status());
        try!(kitware_git::check_status(push, "push failed"));

        self.send_info_mr_comment(mr, "Topic successfully merged and pushed!");

        Ok(())
    }

    fn build_commit_message<T>(&self, mr: &HostedMergeRequest, trailers: T)
                               -> Result<String, io::Error>
        where T: Iterator<Item = Trailer>,
    {
        let mut branch_summary = mr.description()
            // Break it into lines.
            .lines()
            // Find the message block.
            .skip_while(|&line| line != "```message")
            // Skip the entry line.
            .skip(1)
            // Take it until the end of the block.
            .take_while(|&line| line != "```")
            // Join the lines together.
            .join("\n");
        if !branch_summary.is_empty() {
            // Append a separator if we have a branch description.
            branch_summary.push_str("\n\n");
        }

        let mut log_command = self.ctx.git();
        log_command
            .arg("log")
            .arg("--date-order")
            .arg("--format=%h %s")
            // FIXME: Use the new "auto abbrev" functionality in git. It appears that this will
            // give us "7" by default, but it is likely better than this once we outgrow `8`.
            .arg("--abbrev=8");

        if let Some(limit) = self.log_limit {
            // Get up to one more than the maximum. This is done so that we can detect that there
            // are more so that an elision indicator may be added.
            log_command.arg(format!("--max-count={}", limit + 1));
        }

        let log = try!(log_command
            .arg(format!("{}..{}", self.branch, mr.commit().id()))
            .output());
        try!(kitware_git::check_status(log.status, "log failed"));
        let log_output = String::from_utf8_lossy(&log.stdout);
        let mut log_lines = log_output.lines().collect::<Vec<_>>();
        // Elide the log if there are too many entries.
        if let Some(limit) = self.log_limit {
            if limit == 0 {
                log_lines.clear()
            } else if log_lines.len() > limit {
                log_lines[limit] = "...";
            }
        }
        let mut log_summary = log_lines.into_iter()
            .join("\n");
        if !log_summary.is_empty() {
            // Append a separator if we have a log summary.
            log_summary.push_str("\n\n");
        }

        let trailer_summary =
            trailers.chain(iter::once(Trailer::new("Merge-request", mr.reference())))
                .map(|trailer| format!("{}", trailer))
                .join("  \n");

        Ok(format!("Merge topic \'{}\'\n\
                    \n\
                    {}{}{}",
                   mr.source_branch(),
                   branch_summary,
                   log_summary,
                   trailer_summary))
    }

    fn send_mr_comment(&self, mr: &HostedMergeRequest, content: &str) {
        if let Err(err) = self.project.service.post_mr_comment(mr, content) {
            error!(target: "workflow/merge",
                   "failed to post a comment to merge request: {}, {}: {:?}",
                   self.project.name,
                   mr.id(),
                   err);
        }
    }

    fn send_info_mr_comment(&self, mr: &HostedMergeRequest, content: &str) {
        if !self.quiet {
            self.send_mr_comment(mr, content)
        }
    }
}

fn unmerged_status_message(reason: &MergeStatus) -> String {
    let reason_message = match *reason {
        MergeStatus::NoCommonHistory => "there is no common history",
        MergeStatus::AlreadyMerged => "it has already been merged",
        MergeStatus::Mergeable(_) => "it is…mergeable? Sorry, something went wrong",
    };

    format!("This merge request may not be merged because {}.", reason_message)
}

fn parse_awards_as_trailers(award: &HostedAward) -> Option<Trailer> {
    // TODO: Implement.
    None
}

fn parse_comment_for_trailers(service: &HostingService, comment: &HostedComment) -> Vec<Trailer> {
    // TODO: Implement.
    vec![]
}
