// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `check` action.
//!
//! This action checks that commits pass to a set of git hooks.

extern crate git_hooks;
use self::git_hooks::{GitHookConfiguration, HookResult};

extern crate itertools;
use self::itertools::Itertools;

extern crate kitware_git;
use self::kitware_git::{CommitId, GitContext, Identity};

use super::super::host::*;

use std::io;
use std::rc::Rc;

quick_error! {
    #[derive(Debug)]
    /// An error within the `check` action.
    pub enum Error {
        /// An error occurred while performing a git action.
        Git(err: io::Error) {
            cause(err)
            display("git error: {:?}", err)
            from()
        }
        /// An error occurred while communicating with the hosting service.
        Host(err: HostError) {
            cause(err)
            display("hosting error: {:?}", err)
            from()
        }
    }
}

#[derive(Clone, Copy, PartialEq, Eq, Debug)]
/// States for a check result.
pub enum CheckStatus {
    /// The checks passed.
    Pass,
    /// The checks failed.
    Fail,
}

struct TopicCommit<'a> {
    commit: &'a HostedCommit,
    sha: CommitId,
}

impl<'a> TopicCommit<'a> {
    fn new(commit: &'a HostedCommit, sha: CommitId) -> Self {
        TopicCommit {
            commit: commit,
            sha: sha,
        }
    }
}

impl<'a> HostedCommit for TopicCommit<'a> {
    fn project(&self) -> &HostedRepo {
        self.commit.project()
    }

    fn refname(&self) -> Option<&str> {
        self.commit.refname()
    }

    fn id(&self) -> &CommitId {
        &self.sha
    }

    fn _self_ref_hack(&self) -> &HostedCommit {
        self
    }
}

/// Implementation of the `check` action.
pub struct Check<'a> {
    ctx: GitContext,
    service: Rc<HostingService>,
    config: GitHookConfiguration<'a>,
    admins: &'a [String],
}

impl<'a> Check<'a> {
    /// Create a new check action.
    pub fn new(ctx: GitContext, service: Rc<HostingService>, config: GitHookConfiguration<'a>,
               admins: &'a [String])
               -> Self {
        Check {
            ctx: ctx,
            service: service,
            config: config,
            admins: admins,
        }
    }

    /// Check a range of commits.
    pub fn check_mr<R>(self, reason: R, base: &CommitId, mr: &HostedMergeRequest)
                       -> Result<CheckStatus, Error>
        where R: AsRef<str>,
    {
        let refs = try!(self.config.list(&self.ctx, reason.as_ref(), base, mr.commit().id()));

        let mut result = try!(refs.into_iter()
            .map(|sha| {
                let result = try!(self.config
                    .run(&self.ctx, &[sha.clone()], &mr.author().identity()));

                let state = if result.pass() {
                    CommitStatusState::Success
                } else {
                    CommitStatusState::Failed
                };
                let topic_commit = TopicCommit::new(mr.commit(), sha);
                let status = topic_commit.create_commit_status(state,
                                                               "kwrobot-commit-check",
                                                               "basic content checks");
                try!(self.service.post_commit_status(status));

                Ok(result)
            })
            .collect::<Result<Vec<_>, Error>>())
            .into_iter()
            .fold(HookResult::new(), HookResult::combine);

        if mr.work_in_progress() {
            result.add_warning("the merge request is marked as a work-in-progress");
        }

        let state = if result.pass() {
            CommitStatusState::Success
        } else {
            CommitStatusState::Failed
        };
        let status = mr.create_commit_status(state,
                                             "kwrobot-branch-check",
                                             "overall branch status for basic content checks");
        try!(self.service.post_commit_status(status));

        self.report_to_mr(mr, result)
    }

    /// Check a single commit.
    pub fn check_mr_merge(self, candidate_merge: &CommitId, merger: &Identity,
                          mr: &HostedMergeRequest)
                          -> Result<CheckStatus, Error> {
        let mut result = try!(self.config.run(&self.ctx, &[candidate_merge.clone()], merger));

        // Just let whitelisted merge commits through.
        if result.allowed() {
            return Ok(CheckStatus::Pass);
        }

        if mr.work_in_progress() {
            result.add_warning("the merge request is marked as a work-in-progress");
        }

        let comment = self.hook_result_comment(result, false);
        Ok(if comment.is_empty() {
            CheckStatus::Pass
        } else {
            let merge_comment = format!("The proposed merge commit {} failed content \
                                         checks:\n\n{}",
                                        candidate_merge,
                                        comment);
            try!(self.service.post_mr_comment(mr, &merge_comment));

            CheckStatus::Fail
        })
    }

    // Post the results of a check as a merge request comment.
    fn report_to_mr(&self, mr: &HostedMergeRequest, result: HookResult)
                    -> Result<CheckStatus, Error> {
        // Just silently accept allowed MRs.
        if result.allowed() {
            return Ok(CheckStatus::Pass);
        }

        let pass = result.pass();

        let comment = self.hook_result_comment(result, true);
        if !comment.is_empty() {
            try!(self.service.post_mr_comment(mr, &comment));
        }

        Ok(if pass {
            CheckStatus::Pass
        } else {
            CheckStatus::Fail
        })
    }

    fn hook_result_comment(&self, result: HookResult, with_assist: bool) -> String {
        let mut comment = String::new();

        // This scope is necessary so that the borrow in `push_results` ends before we use
        // `comment` again at the end of the function.
        {
            let mut push_results = |label, items: &Vec<String>| {
                if !items.is_empty() {
                    comment.push_str(&Self::comment_fragment(label, items));
                }
            };

            push_results("Errors", result.errors());
            push_results("Warnings", result.warnings());
            push_results("Alerts", result.alerts());
        }

        if with_assist {
            if !result.warnings().is_empty() {
                comment.push_str("The warnings may be temporary; if they are, a comment with the \
                                  text `Do: check` at the end of the comment will rerun the \
                                  checks.\n\n");
            }

            if !result.errors().is_empty() {
                comment.push_str("Please rewrite commits to fix the errors listed above (adding \
                                  fixup commits will not resolve the errors) and force-push the \
                                  branch again to update the merge request.\n\n");
            }
        }

        if !result.alerts().is_empty() {
            comment.push_str(&format!("Alert: @{}.\n\n", self.admins.join(" @")));
        }

        // Remove trailing whitespace from the comment.
        let non_ws_len = comment.trim_right().len();
        comment.truncate(non_ws_len);

        comment
    }

    fn comment_fragment(label: &str, items: &[String]) -> String {
        format!("{}:  \n  - {}\n\n", label, items.iter().join("\n  - "))
    }
}
