// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate git_workarea;
use self::git_workarea::{GitContext, Identity};

extern crate itertools;
use self::itertools::Itertools;

use actions::merge::*;
use actions::tests::host::MockService;
use actions::tests::utils::test_workspace_dir;
use host::{HostedProject, HostingService, User};
use utils::Trailer;

use std::collections::HashMap;
use std::path::Path;
use std::process::Command;
use std::rc::Rc;

static BRANCH_NAME: &'static str = "test-merge";
static BASE: &'static str = "58b2dee73ab6e6b1f3587b41d0ccdbe2ded785dd";
static BASE_UPDATE: &'static str = "11bfbf44147015650afe6f95508ecfb1a77443cd";

lazy_static! {
    static ref MERGER_IDENT: Identity =
        Identity::new("merger", "merger@example.com");
    static ref AUTHOR_DATE: DateTime<UTC> = UTC::now();
}

fn git_context(workspace_path: &Path, commit: &str) -> (GitContext, GitContext) {
    // Here, we create two clones of the current repository: one to act as the remote and another
    // to be the repository the merge action uses. The first is cloned from the source tree's
    // directory while the second is cloned from that first clone. This sets up the `origin` remote
    // properly for the `merge` command.

    let origindir = workspace_path.join("origin");
    let clone = Command::new("git")
        .arg("clone")
        .arg("--bare")
        .arg(concat!(env!("CARGO_MANIFEST_DIR"), "/.git"))
        .arg(&origindir)
        .output()
        .unwrap();
    if !clone.status.success() {
        panic!("origin clone failed: {}",
               String::from_utf8_lossy(&clone.stderr));
    }

    // Make the branch point to the expected place.
    let origin_ctx = GitContext::new(&origindir);
    let update_ref = origin_ctx.git()
        .arg("update-ref")
        .arg(format!("refs/heads/{}", BRANCH_NAME))
        .arg(commit)
        .output()
        .unwrap();
    if !update_ref.status.success() {
        panic!("update-ref failed: {}",
               String::from_utf8_lossy(&update_ref.stderr));
    }

    let gitdir = workspace_path.join("git");
    let clone = Command::new("git")
        .arg("clone")
        .arg("--bare")
        .arg(origindir)
        .arg(&gitdir)
        .output()
        .unwrap();
    if !clone.status.success() {
        panic!("working clone failed: {}",
               String::from_utf8_lossy(&clone.stderr));
    }

    (origin_ctx, GitContext::new(gitdir))
}

fn make_commit(ctx: &GitContext) {
    let commit = ctx.git()
        .arg("commit-tree")
        .arg("-p").arg(BRANCH_NAME)
        .arg("-m").arg("blocking commit")
        .arg(format!("{}^{{tree}}", BRANCH_NAME))
        .output()
        .unwrap();
    if !commit.status.success() {
        panic!("blocking commit creation failed: {}",
               String::from_utf8_lossy(&commit.stderr));
    }
    let commit = String::from_utf8_lossy(&commit.stdout);
    let update_ref = ctx.git()
        .arg("update-ref")
        .arg(format!("refs/heads/{}", BRANCH_NAME))
        .arg(commit.trim())
        .status()
        .unwrap();
    assert!(update_ref.success());
}

#[derive(Debug)]
struct TestMergePolicy {
    trailers: Vec<Trailer>,
    needs_reviews_from: HashMap<u64, String>,
    rejections: Vec<String>,
}

impl Default for TestMergePolicy {
    fn default() -> Self {
        TestMergePolicy {
            trailers: vec![
                Trailer::new("Acked-by", "Ghostflow <ghostflow@example.com>"),
            ],
            needs_reviews_from: HashMap::new(),
            rejections: Vec::new(),
        }
    }
}

impl MergePolicyFilter for TestMergePolicy {
    fn process_trailer(&mut self, trailer: Trailer, user: Option<&User>) {
        // Skip trailers which do not end in `-by`.
        if !trailer.token.ends_with("-by") {
            return;
        }

        // Ignore trailers by the `ignore` user.
        if let Some(user) = user {
            if user.handle == "ignore" {
                return;
            }
        }

        if trailer.token == "Require-review-by" {
            if let Some(user) = user {
                let msg = format!("review is required by @{}", user.handle);
                self.needs_reviews_from.insert(user.id, msg);
            }
        } else if trailer.token == "Rejected-by" {
            // Block if anyone else says `Rejected-by`.
            let reason = if let Some(user) = user {
                format!("rejected by @{}", user.handle)
            } else {
                format!("rejected by {}", trailer.value)
            };

            self.rejections.push(reason);
        } else {
            // Add the trailer.
            self.trailers.push(trailer);
        }
    }

    fn result(self) -> ::std::result::Result<Vec<Trailer>, Vec<String>> {
        let reasons = self.rejections
            .into_iter()
            .chain(self.needs_reviews_from
                .into_iter()
                .map(|(_, value)| value))
            .collect::<Vec<_>>();

        if reasons.is_empty() {
            Ok(self.trailers)
        } else {
            Err(reasons)
        }
    }
}

fn create_merge(git: &GitContext, base: &str, service: &Rc<MockService>) -> Merge<TestMergePolicy> {
    let project = HostedProject {
        name: "base".to_string(),
        service: service.clone(),
    };
    Merge::new(git.clone(), base, project, TestMergePolicy::default())
}

fn check_mr_commit_message(ctx: &GitContext, expected: &str) {
    let cat_file = ctx.git()
        .arg("cat-file")
        .arg("-p")
        .arg(format!("refs/heads/{}", BRANCH_NAME))
        .output()
        .unwrap();
    let actual = String::from_utf8_lossy(&cat_file.stdout)
        .splitn(2, "\n\n")
        .skip(1)
        .join("\n\n");
    assert_eq!(actual, expected);
}

fn check_mr_commit(ctx: &GitContext, expected: &str) {
    check_mr_commit_message(ctx, expected);

    let author_log = ctx.git()
        .arg("log")
        .arg("--pretty=%an%n%ae%n%ad")
        .arg("--max-count=1")
        .arg(format!("refs/heads/{}", BRANCH_NAME))
        .output()
        .unwrap();
    let actual = String::from_utf8_lossy(&author_log.stdout);
    let author_log_lines = actual.lines().collect::<Vec<_>>();
    assert_eq!(author_log_lines[0], MERGER_IDENT.name);
    assert_eq!(author_log_lines[1], MERGER_IDENT.email);
    assert_eq!(author_log_lines[2],
               AUTHOR_DATE.format("%a %b %-d %H:%M:%S %Y %z").to_string());
}

#[test]
// Merging should succeed with a different branch name.
fn test_merge_simple_branch_name() {
    let tempdir = test_workspace_dir("test_merge_simple_branch_name");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr_named(&mr, "topic-1-renamed", &MERGER_IDENT, &AUTHOR_DATE)
        .unwrap();
    assert_eq!(res, MergeActionResult::Success);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "Topic successfully merged and pushed!");

    check_mr_commit(&origin_ctx,
                    "Merge topic \'topic-1-renamed\'\n\
                     \n\
                     A simple message\n\
                     \n\
                     7189cf55 topic-1: make a change\n\
                     \n\
                     Acked-by: Ghostflow <ghostflow@example.com>\n\
                     Tested-by: other <other@example.com>\n\
                     Tested-by: maint <maint@example.com>\n\
                     Acked-by: maint <maint@example.com>\n\
                     Acked-by: other <other@example.com>\n\
                     Merge-request: !1\n");
}

#[test]
// Merging should succeed.
fn test_merge_simple() {
    let tempdir = test_workspace_dir("test_merge_simple");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Success);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "Topic successfully merged and pushed!");

    check_mr_commit(&origin_ctx,
                    "Merge topic \'topic-1\'\n\
                     \n\
                     A simple message\n\
                     \n\
                     7189cf55 topic-1: make a change\n\
                     \n\
                     Acked-by: Ghostflow <ghostflow@example.com>\n\
                     Tested-by: other <other@example.com>\n\
                     Tested-by: maint <maint@example.com>\n\
                     Acked-by: maint <maint@example.com>\n\
                     Acked-by: other <other@example.com>\n\
                     Merge-request: !1\n");
}

#[test]
// Merging should not comment when quiet.
fn test_merge_simple_quiet() {
    let tempdir = test_workspace_dir("test_merge_simple_quiet");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let mut merge = create_merge(&ctx, BRANCH_NAME, &service);
    merge.quiet();

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Success);

    assert_eq!(service.remaining_data(), 0);

    check_mr_commit(&origin_ctx,
                    "Merge topic \'topic-1\'\n\
                     \n\
                     A simple message\n\
                     \n\
                     7189cf55 topic-1: make a change\n\
                     \n\
                     Acked-by: Ghostflow <ghostflow@example.com>\n\
                     Tested-by: other <other@example.com>\n\
                     Tested-by: maint <maint@example.com>\n\
                     Acked-by: maint <maint@example.com>\n\
                     Acked-by: other <other@example.com>\n\
                     Merge-request: !1\n");
}

#[test]
// Merging should create a log without elision if the limit is over the length.
fn test_merge_simple_log_limit_over() {
    let tempdir = test_workspace_dir("test_merge_simple_log_limit_over");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let mut merge = create_merge(&ctx, BRANCH_NAME, &service);
    merge.log_limit(Some(1));

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Success);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "Topic successfully merged and pushed!");

    check_mr_commit(&origin_ctx,
                    "Merge topic \'topic-1\'\n\
                     \n\
                     A simple message\n\
                     \n\
                     7189cf55 topic-1: make a change\n\
                     \n\
                     Acked-by: Ghostflow <ghostflow@example.com>\n\
                     Tested-by: other <other@example.com>\n\
                     Tested-by: maint <maint@example.com>\n\
                     Acked-by: maint <maint@example.com>\n\
                     Acked-by: other <other@example.com>\n\
                     Merge-request: !1\n");
}

#[test]
// Merging should not elide if the limit is exactly the topic length.
fn test_merge_simple_log_limit_exact() {
    let tempdir = test_workspace_dir("test_merge_simple_log_limit_exact");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let mut merge = create_merge(&ctx, BRANCH_NAME, &service);
    merge.log_limit(Some(1));

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Success);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "Topic successfully merged and pushed!");

    check_mr_commit(&origin_ctx,
                    "Merge topic \'topic-1\'\n\
                     \n\
                     A simple message\n\
                     \n\
                     7189cf55 topic-1: make a change\n\
                     \n\
                     Acked-by: Ghostflow <ghostflow@example.com>\n\
                     Tested-by: other <other@example.com>\n\
                     Tested-by: maint <maint@example.com>\n\
                     Acked-by: maint <maint@example.com>\n\
                     Acked-by: other <other@example.com>\n\
                     Merge-request: !1\n");
}

#[test]
// Merging should elide if the topic is longer than the limit.
fn test_merge_simple_log_limit_elide() {
    let tempdir = test_workspace_dir("test_merge_simple_log_limit_elide");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let mut merge = create_merge(&ctx, BRANCH_NAME, &service);
    merge.log_limit(Some(1));

    service.step(2);

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Success);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "Topic successfully merged and pushed!");

    check_mr_commit(&origin_ctx,
                    "Merge topic \'topic-1\'\n\
                     \n\
                     A simple message\n\
                     \n\
                     fe70f127 topic-1: update\n\
                     ...\n\
                     \n\
                     Acked-by: Ghostflow <ghostflow@example.com>\n\
                     Merge-request: !1\n");
}

#[test]
// Merging not log if the limit is zero.
fn test_merge_simple_log_limit_elide_zero() {
    let tempdir = test_workspace_dir("test_merge_simple_log_limit_elide_zero");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let mut merge = create_merge(&ctx, BRANCH_NAME, &service);
    merge.log_limit(Some(0));

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Success);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "Topic successfully merged and pushed!");

    check_mr_commit(&origin_ctx,
                    "Merge topic \'topic-1\'\n\
                     \n\
                     A simple message\n\
                     \n\
                     Acked-by: Ghostflow <ghostflow@example.com>\n\
                     Tested-by: other <other@example.com>\n\
                     Tested-by: maint <maint@example.com>\n\
                     Acked-by: maint <maint@example.com>\n\
                     Acked-by: other <other@example.com>\n\
                     Merge-request: !1\n");
}

#[test]
// Merging WIP branches should fail.
fn test_merge_wip() {
    let tempdir = test_workspace_dir("test_merge_wip");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    let mr = service.merge_request("base", 2).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Failed);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "This merge request is marked as a Work in Progress and may not be merged. Please \
                remove the Work in Progress state first.");

    check_mr_commit_message(&origin_ctx, "base: add a base commit\n");
}

#[test]
// Test failure to push to the remote server.
fn test_merge_push_fail() {
    let tempdir = test_workspace_dir("test_merge_push_fail");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    // Make a change on remote the clone is not aware of.
    make_commit(&origin_ctx);

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::PushFailed);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "Automatic merge succeeded, but pushing to the remote failed!");

    check_mr_commit_message(&origin_ctx, "blocking commit\n");
}

#[test]
// Test failure to push to the remote server.
fn test_merge_push_fail_quiet() {
    let tempdir = test_workspace_dir("test_merge_push_fail_quiet");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let mut merge = create_merge(&ctx, BRANCH_NAME, &service);
    merge.quiet();

    // Make a change on remote the clone is not aware of.
    make_commit(&origin_ctx);

    let mr = service.merge_request("base", 1).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::PushFailed);

    assert_eq!(service.remaining_data(), 0);

    check_mr_commit_message(&origin_ctx, "blocking commit\n");
}

#[test]
// Already merged branches should be ignored.
fn test_merge_already_merged() {
    let tempdir = test_workspace_dir("test_merge_already_merged");
    let (_, ctx) = git_context(tempdir.path(), BASE_UPDATE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    let mr = service.merge_request("base", 5).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Failed);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "This merge request may not be merged because it has already been merged.");
}

#[test]
// Branches which do not share history should not be allowed.
fn test_merge_no_common_history() {
    let tempdir = test_workspace_dir("test_merge_no_common_history");
    let (_, ctx) = git_context(tempdir.path(), BASE_UPDATE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    service.step(1);

    let mr = service.merge_request("base", 5).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Failed);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "This merge request may not be merged because there is no common history.");
}

#[test]
// Conflicts should be reported.
fn test_merge_conflict() {
    let tempdir = test_workspace_dir("test_merge_conflict");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE_UPDATE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    let mr = service.merge_request("base", 3).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Failed);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "This merge request contains conflicts in the following paths:  \n\
                `base`");

    check_mr_commit_message(&origin_ctx, "base: update base commit\n");
}

#[test]
// Policies should be allowed to reject merge requests.
fn test_merge_policy_rejection() {
    let tempdir = test_workspace_dir("test_merge_policy_rejection");
    let (origin_ctx, ctx) = git_context(tempdir.path(), BASE);
    let service = MockService::test_service();
    let merge = create_merge(&ctx, BRANCH_NAME, &service);

    let mr = service.merge_request("base", 3).unwrap();
    let res = merge.merge_mr(&mr, &MERGER_IDENT, &AUTHOR_DATE).unwrap();
    assert_eq!(res, MergeActionResult::Failed);

    let mr_commit_comments = service.mr_comments(mr.id);

    assert_eq!(service.remaining_data(), 0);

    assert_eq!(mr_commit_comments.len(), 1);
    assert_eq!(mr_commit_comments[0],
               "This merge request may not be merged because:  \n  \
                - rejected by @maint");

    check_mr_commit_message(&origin_ctx, "base: add a base commit\n");
}
