// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use actions::clone::*;
use actions::tests::host::MockService;
use actions::tests::utils::{check_git_config_value, check_git_config_values, test_workspace_dir};
use host::{HostedProject, HostingService};

use std::fs::read_link;
use std::path::{Path, PathBuf};
use std::rc::Rc;

static PROJECT_NAME: &'static str = "self";

fn create_clone<'a>(path: &Path, service: &Rc<MockService>) -> Clone_ {
    let project = HostedProject {
        name: PROJECT_NAME.to_string(),
        service: service.clone(),
    };

    Clone_::new(path, project)
}

#[test]
fn test_clone_watched_repo() {
    let tempdir = test_workspace_dir("test_clone_watched_repo");
    let service = MockService::test_service();
    let clone = create_clone(tempdir.path(), &service);
    assert!(!clone.exists());

    let ctx = clone.clone_watched_repo().unwrap();

    check_git_config_value(&ctx, "remote.origin.tagopt", "--no-tags");
    check_git_config_value(&ctx,
                           "remote.origin.url",
                           &service.repo(PROJECT_NAME).unwrap().url);
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");

    let clone_after = create_clone(tempdir.path(), &service);
    assert!(clone_after.exists());

    let ctx = clone_after.clone_watched_repo().unwrap();

    check_git_config_value(&ctx, "remote.origin.tagopt", "--no-tags");
    check_git_config_value(&ctx,
                           "remote.origin.url",
                           &service.repo(PROJECT_NAME).unwrap().url);
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
}

#[test]
fn test_clone_watched_repo_submodules() {
    let tempdir = test_workspace_dir("test_clone_watched_repo_submodules");
    let service = MockService::test_service();
    let mut clone = create_clone(tempdir.path(), &service);
    let internal = "submodule";
    let external = "/dev";
    clone.with_submodule("internal", CloneSubmoduleLink::new(internal));
    clone.with_submodule("external", CloneSubmoduleLink::new(external));
    clone.with_submodule("subdir/internal", CloneSubmoduleLink::new(internal));
    clone.with_submodule("subdir/external", CloneSubmoduleLink::new(external));
    assert!(!clone.exists());

    let ctx = clone.clone_watched_repo().unwrap();

    assert_eq!(read_link(ctx.gitdir().join("modules").join("internal")).unwrap(),
               tempdir.path().join(format!("{}.git", internal)));
    assert_eq!(read_link(ctx.gitdir().join("modules").join("external")).unwrap(),
               PathBuf::from(external));
    assert_eq!(read_link(ctx.gitdir().join("modules").join("subdir/internal")).unwrap(),
               tempdir.path().join(format!("{}.git", internal)));
    assert_eq!(read_link(ctx.gitdir().join("modules").join("subdir/external")).unwrap(),
               PathBuf::from(external));
}

#[test]
fn test_clone_mirror_repo() {
    let tempdir = test_workspace_dir("test_clone_mirror_repo");
    let service = MockService::test_service();
    let clone = create_clone(tempdir.path(), &service);
    assert!(!clone.exists());

    let refs = [
        "refs/heads/*",
        "refs/tags/*",
    ];
    let ctx = clone.clone_mirror_repo(&refs).unwrap();

    check_git_config_value(&ctx,
                           "remote.origin.url",
                           &service.repo(PROJECT_NAME).unwrap().url);
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
    check_git_config_values(&ctx,
                            "remote.origin.fetch",
                            &["+refs/heads/*:refs/heads/*", "+refs/tags/*:refs/tags/*"]);

    let clone_after = create_clone(tempdir.path(), &service);
    assert!(clone_after.exists());

    let ctx = clone_after.clone_mirror_repo(&refs).unwrap();

    check_git_config_value(&ctx,
                           "remote.origin.url",
                           &service.repo(PROJECT_NAME).unwrap().url);
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
    check_git_config_values(&ctx,
                            "remote.origin.fetch",
                            &["+refs/heads/*:refs/heads/*", "+refs/tags/*:refs/tags/*"]);

    let clone_change = create_clone(tempdir.path(), &service);
    assert!(clone_change.exists());

    let new_refs = [
        "refs/heads/*",
        "refs/tags/*",
        "refs/notes/*",
    ];
    let ctx = clone_change.clone_mirror_repo(&new_refs).unwrap();

    check_git_config_value(&ctx,
                           "remote.origin.url",
                           &service.repo(PROJECT_NAME).unwrap().url);
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
    check_git_config_values(&ctx,
                            "remote.origin.fetch",
                            &["+refs/heads/*:refs/heads/*",
                              "+refs/tags/*:refs/tags/*",
                              "+refs/notes/*:refs/notes/*"]);
}
