// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The `check` action.
//!
//! This action checks that commits pass to a set of git checks.

extern crate git_checks;
use self::git_checks::{CheckResult, GitCheckConfiguration};

extern crate git_workarea;
use self::git_workarea::{CommitId, GitContext, Identity};

extern crate itertools;
use self::itertools::Itertools;

use host::{self, CommitStatusState, HostingService, MergeRequest};

use std::fmt::{self, Debug};
use std::rc::Rc;

error_chain! {
    links {
        GitChecks(git_checks::Error, git_checks::ErrorKind)
            #[doc = "Errors from the git-checks crate."];
        Host(host::Error, host::ErrorKind)
            #[doc = "Errors from the service host."];
    }

    errors {
        /// An error occurred when executing git commands.
        Git(msg: String) {
            display("git error: {}", msg)
        }
    }
}

/// The name of the status the `check` action will use.
pub const STATUS_NAME: &'static str = "ghostflow-branch-check";

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
/// States for a check result.
pub enum CheckStatus {
    /// The checks passed.
    Pass,
    /// The checks failed.
    Fail,
}

/// Implementation of the `check` action.
pub struct Check<'a> {
    /// The context to use for checking commits.
    ctx: GitContext,
    /// The service which hosts the project.
    service: Rc<HostingService>,
    /// The configuration to use for the check.
    ///
    /// This contains the actual checks to use.
    config: GitCheckConfiguration<'a>,
    /// The administrators of the project.
    ///
    /// These users are notified when checks find problems which should be brought to an
    /// administrator's attention.
    admins: &'a [String],
}

impl<'a> Check<'a> {
    /// Create a new check action.
    pub fn new(ctx: GitContext, service: Rc<HostingService>, config: GitCheckConfiguration<'a>,
               admins: &'a [String])
               -> Self {
        Check {
            ctx: ctx,
            service: service,
            config: config,
            admins: admins,
        }
    }

    /// Check a range of commits.
    pub fn check_mr<R>(self, reason: R, base: &CommitId, mr: &MergeRequest) -> Result<CheckStatus>
        where R: AsRef<str>,
    {
        info!(target: "ghostflow/check",
              "checking merge request {}",
              mr.url);

        let refs = self.config.list(&self.ctx, reason.as_ref(), base, &mr.commit.id)?;

        let mut result = refs.into_iter()
            .map(|sha| {
                let result = self.config
                    .run(&self.ctx, &[sha.clone()], &mr.author.identity())?;

                let state = if result.pass() {
                    CommitStatusState::Success
                } else {
                    CommitStatusState::Failed
                };
                let topic_commit = {
                    let mut commit = mr.commit.clone();
                    commit.id = sha;
                    commit
                };
                let status =
                    topic_commit.create_commit_status(state,
                                                      "ghostflow-commit-check",
                                                      "basic content checks");
                self.service.post_commit_status(status)?;

                Ok(result)
            })
            .collect::<Result<Vec<_>>>()?
            .into_iter()
            .fold(CheckResult::new(), CheckResult::combine);

        if mr.work_in_progress {
            result.add_warning("the merge request is marked as a work-in-progress.");
        }

        let state = if result.pass() {
            CommitStatusState::Success
        } else {
            CommitStatusState::Failed
        };
        let status = mr.create_commit_status(state,
                                             STATUS_NAME,
                                             "overall branch status for basic content checks");
        self.service.post_commit_status(status)?;

        self.report_to_mr(mr, result)
    }

    /// Check a single commit.
    pub fn check_mr_merge(self, candidate_merge: &CommitId, merger: &Identity, mr: &MergeRequest)
                          -> Result<CheckStatus> {
        info!(target: "ghostflow/check",
              "checking the merge of merge request {}",
              mr.url);

        let mut result = self.config.run(&self.ctx, &[candidate_merge.clone()], merger)?;

        // Just let whitelisted merge commits through.
        if result.allowed() {
            return Ok(CheckStatus::Pass);
        }

        if mr.work_in_progress {
            result.add_warning("the merge request is marked as a work-in-progress.");
        }

        let comment = self.check_result_comment(result, false);
        Ok(if comment.is_empty() {
            // TODO: Add a `quiet` option.
            try!(self.service.post_mr_comment(mr, "Basic content checks passed!"));

            CheckStatus::Pass
        } else {
            let merge_comment = format!("The proposed merge commit {} failed content \
                                         checks:\n\n{}",
                                        candidate_merge,
                                        comment);
            self.service.post_mr_comment(mr, &merge_comment)?;

            CheckStatus::Fail
        })
    }

    /// Post the results of a check as a merge request comment.
    fn report_to_mr(&self, mr: &MergeRequest, result: CheckResult) -> Result<CheckStatus> {
        // Just silently accept allowed MRs.
        if result.allowed() {
            return Ok(CheckStatus::Pass);
        }

        let pass = result.pass();

        let comment = self.check_result_comment(result, true);
        if !comment.is_empty() {
            self.service.post_mr_comment(mr, &comment)?;
        }

        Ok(if pass {
            CheckStatus::Pass
        } else {
            CheckStatus::Fail
        })
    }

    /// Create a comment for the given check result.
    fn check_result_comment(&self, result: CheckResult, with_assist: bool) -> String {
        let mut comment = String::new();

        // This scope is necessary so that the borrow in `push_results` ends before we use
        // `comment` again at the end of the function.
        {
            let mut push_results = |label, items: &Vec<String>| {
                if !items.is_empty() {
                    comment.push_str(&Self::comment_fragment(label, items));
                }
            };

            push_results("Errors", result.errors());
            push_results("Warnings", result.warnings());
            push_results("Alerts", result.alerts());
        }

        if with_assist {
            if !result.warnings().is_empty() {
                comment.push_str("The warnings may be temporary; if they are, a comment with the \
                                  text `Do: check` at its end will rerun the checks.\n\n");
            }

            if !result.errors().is_empty() {
                comment.push_str("Please rewrite commits to fix the errors listed above (adding \
                                  fixup commits will not resolve the errors) and force-push the \
                                  branch again to update the merge request.\n\n");
            }
        }

        if !result.alerts().is_empty() {
            comment.push_str(&format!("Alert: @{}.\n\n", self.admins.join(" @")));
        }

        // Remove trailing whitespace from the comment.
        let non_ws_len = comment.trim_right().len();
        comment.truncate(non_ws_len);

        comment
    }

    /// Create a fragment of the comment.
    fn comment_fragment(label: &str, items: &[String]) -> String {
        format!("{}:  \n  - {}\n\n", label, items.iter().join("\n  - "))
    }
}

impl<'a> Debug for Check<'a> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f,
               "Check {{ gitdir: {}, admins: ['{}'] }}",
               self.ctx.gitdir().to_string_lossy(),
               self.admins.iter().join("', '"))
    }
}
