// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Git hosting service traits
//!
//! The traits in this module are meant to help keep service-specific knowledge away from the
//! workflow implementation.
//!
//! These traits might be split into a separate library in the future.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate kitware_git;
use self::kitware_git::{CommitId, GitContext, Identity};

use std::error;
use std::io;
use std::rc::Rc;

/// A commit hosted on the service provider.
pub trait HostedCommit {
    /// The repository where the commit lives.
    fn project(&self) -> &HostedRepo;
    /// The refname for the commit (if available).
    fn refname(&self) -> Option<&str>;
    /// The object id of the commit.
    fn id(&self) -> &str;

    /// Create a commit status for the commit.
    fn create_commit_status<'a>(&'a self, state: CommitStatusState, name: &'a str,
                                description: &'a str)
                                -> Box<HostedCommitStatus<'a>> {
        Box::new(HostedCommitStatus {
            commit: self._self_ref_hack(),
            state: state,
            refname: self.refname(),
            name: name,
            description: description,
        })
    }

    /// Convenience method for getting a commit id structure for the commit.
    fn commit_id(&self) -> CommitId {
        CommitId::new(self.id())
    }

    /// A hack to get a reference to a `HostedCommit` for the default implementation of
    /// `create_commit_status`.
    fn _self_ref_hack(&self) -> &HostedCommit;
}

/// A repository hosted on the service.
pub trait HostedRepo {
    /// The name of the project.
    fn name(&self) -> &str;

    /// The URL which should be used to fetch from the repository.
    ///
    /// Whether this uses HTTPS or SSH is dependent on the service, but it should not require
    /// interaction in order to use (whether through an SSH key or administrator privileges).
    fn url(&self) -> &str;

    /// The ID of the repository.
    fn id(&self) -> u64;
}

/// An issue on the service.
pub trait HostedIssue {
    /// The source repository for the merge request.
    fn repo(&self) -> &HostedRepo;
    /// The internal ID of the issue.
    ///
    /// Service-specific.
    fn id(&self) -> u64;
    /// The URL of the merge request.
    fn url(&self) -> &str;
    /// The description for the issue.
    fn description(&self) -> &str;
    /// The labels for the issue.
    fn labels(&self) -> &[String];
    /// The milestone for the issue.
    fn milestone(&self) -> Option<&str>;
    /// The username of the assignee.
    fn assignee(&self) -> Option<&str>;
}

/// A merge request on the service.
pub trait HostedMergeRequest {
    /// The source repository for the merge request.
    fn source_repo(&self) -> &HostedRepo;
    /// The name of the branch requested for merging.
    fn source_branch(&self) -> &str;
    /// The repository the merge request will be merged into.
    fn target_repo(&self) -> &HostedRepo;
    /// The target branch for the request.
    fn target_branch(&self) -> &str;
    /// The internal ID of the merge request.
    ///
    /// Service-specific.
    fn id(&self) -> u64;
    /// The URL of the merge request.
    fn url(&self) -> &str;
    /// Whether the merge request is a "work-in-progress" or not.
    fn work_in_progress(&self) -> bool;
    /// The description for the merge request.
    fn description(&self) -> &str;
    /// The previous commit of the merge request (if available).
    ///
    /// This is particularly important for the stage action. Not so important otherwise.
    fn old_commit(&self) -> Option<&HostedCommit>;
    /// The commit which has been requested for merging.
    fn commit(&self) -> &HostedCommit;

    /// Create a commit status for the merge request.
    fn create_commit_status<'a>(&'a self, state: CommitStatusState, name: &'a str,
                                description: &'a str)
                                -> Box<HostedCommitStatus<'a>> {
        Box::new(HostedCommitStatus {
            commit: self.commit(),
            state: state,
            refname: Some(self.source_branch()),
            name: name,
            description: description,
        })
    }

    /// Convenience method for getting a commit id structure for the old commit.
    fn old_commit_id(&self) -> Option<CommitId> {
        self.old_commit().map(|c| c.commit_id())
    }

    /// Convenience method for getting a commit id structure for the commit.
    fn commit_id(&self) -> CommitId {
        self.commit().commit_id()
    }
}

#[derive(Clone, Copy, PartialEq, Eq, Debug)]
/// States for a commit status.
pub enum CommitStatusState {
    /// The check is expected, but has not started yet.
    Pending,
    /// The check is currently running.
    Running,
    /// The check is a success.
    Success,
    /// The check is a failure.
    Failed,
}

/// A commit status for a specific commit.
pub struct HostedCommitStatus<'a> {
    /// The commit the status should apply to.
    pub commit: &'a HostedCommit,
    /// The state of the commit status.
    pub state: CommitStatusState,
    /// The refname of the commit (if applicable).
    pub refname: Option<&'a str>,
    /// The name of the check being performed.
    pub name: &'a str,
    /// A description of the check.
    pub description: &'a str,
}

/// A user on the service.
pub trait HostedUser {
    /// The internal ID of the user.
    ///
    /// Service-specific.
    fn id(&self) -> u64;
    /// The username on the service of the user (used for mentioning the user).
    fn handle(&self) -> &str;
    /// The real name of the user.
    fn name(&self) -> &str;
    /// The email address of the user.
    fn email(&self) -> &str;

    /// Convenience method for getting an identity for the user.
    fn identity(&self) -> Identity {
        Identity::new(self.name(), self.email())
    }
}

/// A comment on the service.
pub trait HostedComment {
    /// Indicates whether the comment is autogenerated (via activity or mentions) or not.
    fn is_system(&self) -> bool;
    /// When the comment was created.
    fn created_at(&self) -> &DateTime<UTC>;
    /// When the comment was last updated.
    fn updated_at(&self) -> &DateTime<UTC>;
    /// The author of the comment.
    fn author(&self) -> &HostedUser;
    /// The content of the comment.
    fn content(&self) -> &str;
}

quick_error! {
    #[derive(Debug)]
    /// An error from a hosting service.
    pub enum HostError {
        /// An error occurred when communicating with the service.
        Communication(err: Box<error::Error>) {
            display("service communication error: {:?}", err)
            cause(err.as_ref())
        }
        /// An error occurred on the host itself.
        Host(err: Box<error::Error>) {
            display("service error: {:?}", err)
            cause(err.as_ref())
        }
        /// The service returned an invalidly formatted result.
        InvalidFormat(err: Box<error::Error>) {
            display("invalid format error: {:?}", err)
            cause(err.as_ref())
        }
    }
}

/// A project hosted on a service.
pub struct HostedProject {
    /// The name of the project.
    pub name: String,
    /// The service the project is hosted on.
    pub service: Rc<HostingService>,
}

impl HostedProject {
    /// Get a commit for a project.
    pub fn commit(&self, commit: &CommitId) -> Result<Box<HostedCommit>, HostError> {
        self.service.commit(&self.name, commit)
    }

    /// Get a merge request on a project.
    pub fn merge_request(&self, id: u64) -> Result<Box<HostedMergeRequest>, HostError> {
        self.service.merge_request(&self.name, id)
    }
}

/// A hosting service.
pub trait HostingService {
    /// Fetch a commit into a given git context.
    ///
    /// The default implementation requires that the commit have a valid refname, otherwise the
    /// fetch will fail.
    fn fetch_commit(&self, git: &GitContext, commit: &HostedCommit) -> Result<(), io::Error> {
        git.fetch(commit.project().url(), &[commit.refname().unwrap()])
    }
    /// Fetch a merge request into a given git context.
    fn fetch_mr(&self, git: &GitContext, mr: &HostedMergeRequest) -> Result<(), io::Error> {
        git.fetch(mr.source_repo().url(), &[mr.source_branch()])
    }

    /// Get a commit for a project.
    fn commit(&self, project: &str, commit: &CommitId) -> Result<Box<HostedCommit>, HostError>;
    /// Get an issue on a project.
    fn issue(&self, project: &str, id: u64) -> Result<Box<HostedIssue>, HostError>;
    /// Get a merge request on a project.
    fn merge_request(&self, project: &str, id: u64) -> Result<Box<HostedMergeRequest>, HostError>;
    /// Get a repository by name.
    fn repo(&self, project: &str) -> Result<Box<HostedRepo>, HostError>;

    /// Get comments for an issue.
    fn get_issue_comments(&self, issue: &HostedIssue)
                          -> Result<Vec<Box<HostedComment>>, HostError>;
    /// Add a comment to an issue.
    fn post_issue_comment(&self, issue: &HostedIssue, content: &str) -> Result<(), HostError>;
    /// Get comments for a merge request.
    fn get_mr_comments(&self, mr: &HostedMergeRequest)
                       -> Result<Vec<Box<HostedComment>>, HostError>;
    /// Add a comment to a merge request.
    fn post_mr_comment(&self, mr: &HostedMergeRequest, content: &str) -> Result<(), HostError>;
    /// Add a comment to a commit.
    fn post_commit_comment(&self, commit: &HostedCommit, content: &str) -> Result<(), HostError>;
    /// Create a commit status.
    fn post_commit_status(&self, status: &HostedCommitStatus) -> Result<(), HostError>;

    /// Get the access level of a user to a project.
    fn user_access(&self, project: &HostedRepo, user: &HostedUser) -> Result<u64, HostError>;
}
