// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate gitlab;
use self::gitlab::{CommitStatusInfo, Gitlab, GitlabResult};

extern crate kitware_git;
use self::kitware_git::CommitId;

use super::traits::*;

struct GitlabCommit {
    repo: GitlabRepo,
    commit: gitlab::RepoCommitDetail,
}

impl HostedCommit for GitlabCommit {
    fn project(&self) -> &HostedRepo {
        &self.repo
    }

    fn refname(&self) -> Option<&str> {
        // TODO
        None
    }

    fn id(&self) -> &str {
        self.commit.id.value()
    }

    fn _self_ref_hack(&self) -> &HostedCommit {
        self
    }
}

struct GitlabRepo {
    project: gitlab::Project,
}

impl HostedRepo for GitlabRepo {
    fn name(&self) -> &str {
        &self.project.path_with_namespace
    }

    fn url(&self) -> &str {
        &self.project.ssh_url_to_repo
    }

    fn id(&self) -> u64 {
        self.project.id.value()
    }
}

struct GitlabIssue {
    repo: GitlabRepo,
    issue: gitlab::Issue,

    // FIXME: Waiting on upstream to provide this.
    // https://gitlab.com/gitlab-org/gitlab-ce/merge_requests/5631
    url: String,
}

impl HostedIssue for GitlabIssue {
    fn repo(&self) -> &HostedRepo {
        &self.repo
    }

    fn id(&self) -> u64 {
        self.issue.id.value()
    }

    fn url(&self) -> &str {
        //&self.issue.web_url
        &self.url
    }

    fn description(&self) -> &str {
        &self.issue.description
    }

    fn labels(&self) -> &[String] {
        &self.issue.labels
    }

    fn milestone(&self) -> Option<&str> {
        self.issue.milestone.as_ref().map(|m| m.title.as_str())
    }

    fn assignee(&self) -> Option<&str> {
        self.issue.assignee.as_ref().map(|u| u.username.as_str())
    }
}

struct GitlabMergeRequest {
    source_repo: GitlabRepo,
    target_repo: GitlabRepo,
    mr: gitlab::MergeRequest,
    commit: GitlabCommit,

    // FIXME: Waiting on upstream to provide this.
    // https://gitlab.com/gitlab-org/gitlab-ce/merge_requests/5631
    url: String,
}

impl HostedMergeRequest for GitlabMergeRequest {
    fn source_repo(&self) -> &HostedRepo {
        &self.source_repo
    }

    fn source_branch(&self) -> &str {
        &self.mr.source_branch
    }

    fn target_repo(&self) -> &HostedRepo {
        &self.target_repo
    }

    fn target_branch(&self) -> &str {
        &self.mr.target_branch
    }

    fn id(&self) -> u64 {
        self.mr.id.value()
    }

    fn url(&self) -> &str {
        //&self.mr.web_url
        &self.url
    }

    fn work_in_progress(&self) -> bool {
        self.mr.work_in_progress
    }

    fn description(&self) -> &str {
        &self.mr.description
    }

    fn old_commit(&self) -> Option<&HostedCommit> {
        // TODO
        None
    }

    fn commit(&self) -> &HostedCommit {
        &self.commit
    }
}

fn gitlab_status_info<'a>(status: &'a HostedCommitStatus) -> CommitStatusInfo<'a> {
    CommitStatusInfo {
        refname: status.refname,
        name: Some(status.name),
        target_url: None,
        description: Some(status.description),
    }
}

struct GitlabUser {
    user: gitlab::UserFull,
}

impl GitlabUser {
    fn new(user: gitlab::UserFull) -> Self {
        GitlabUser {
            user: user,
        }
    }
}

impl HostedUser for GitlabUser {
    fn id(&self) -> u64 {
        self.user.id.value()
    }

    fn handle(&self) -> &str {
        &self.user.username
    }

    fn name(&self) -> &str {
        &self.user.name
    }

    fn email(&self) -> &str {
        &self.user.email
    }
}

struct GitlabNote {
    note: gitlab::Note,
    author: GitlabUser,
}

impl GitlabNote {
    fn new(gitlab: &Gitlab, note: gitlab::Note) -> GitlabResult<Self> {
        let author = try!(gitlab.user::<gitlab::UserFull>(note.author.id));

        Ok(GitlabNote {
            note: note,
            author: GitlabUser::new(author),
        })
    }
}

impl HostedComment for GitlabNote {
    fn is_system(&self) -> bool {
        self.note.system
    }

    fn created_at(&self) -> &DateTime<UTC> {
        &self.note.created_at
    }

    fn updated_at(&self) -> &DateTime<UTC> {
        &self.note.updated_at
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn content(&self) -> &str {
        &self.note.body
    }
}

/// Structure used to communicate with a GitLab instance.
pub struct GitlabService {
    gitlab: Gitlab,
}

impl GitlabService {
    /// Create a new GitLab communication channel.
    pub fn new(gitlab: Gitlab) -> Self {
        GitlabService {
            gitlab: gitlab,
        }
    }
}

impl HostingService for GitlabService {
    fn commit(&self, project: &str, commit: &CommitId) -> Result<Box<HostedCommit>, HostError> {
        let project = try!(self.gitlab.project_by_name(project));
        let commit = try!(self.gitlab.commit(project.id, &commit.0));

        Ok(Box::new(GitlabCommit {
            repo: GitlabRepo {
                project: project,
            },
            commit: commit,
        }))
    }

    fn issue(&self, project: &str, id: u64) -> Result<Box<HostedIssue>, HostError> {
        let issue_project = try!(self.gitlab.project_by_name(project));
        let issue = try!(self.gitlab.issue(issue_project.id, gitlab::IssueId::new(id)));

        Ok(Box::new(GitlabIssue {
            repo: GitlabRepo {
                project: issue_project,
            },
            url: format!("{}/issues/{}", project, issue.id),
            issue: issue,
        }))
    }

    fn merge_request(&self, project: &str, id: u64) -> Result<Box<HostedMergeRequest>, HostError> {
        let target_project = try!(self.gitlab.project_by_name(project));
        let mr =
            try!(self.gitlab.merge_request(target_project.id, gitlab::MergeRequestId::new(id)));
        let source_project = try!(self.gitlab.project(mr.source_project_id));
        let source_project_commit = try!(self.gitlab.project(mr.source_project_id));
        let source_branch = try!(self.gitlab.branch(mr.source_project_id, &mr.source_branch));
        let commit = try!(self.gitlab.commit(mr.source_project_id,
                                             &source_branch.commit.unwrap().id.value()));

        Ok(Box::new(GitlabMergeRequest {
            source_repo: GitlabRepo {
                project: source_project,
            },
            target_repo: GitlabRepo {
                project: target_project,
            },
            url: format!("{}/merge_requests/{}", project, mr.iid),
            mr: mr,
            commit: GitlabCommit {
                repo: GitlabRepo {
                    project: source_project_commit,
                },
                commit: commit,
            },
        }))
    }

    fn repo(&self, project: &str) -> Result<Box<HostedRepo>, HostError> {
        Ok(Box::new(GitlabRepo {
            project: try!(self.gitlab.project_by_name(project)),
        }))
    }

    fn get_issue_comments(&self, issue: &HostedIssue)
                          -> Result<Vec<Box<HostedComment>>, HostError> {
        let notes = try!(self.gitlab.issue_notes(
            gitlab::ProjectId::new(issue.repo().id()),
            gitlab::IssueId::new(issue.id())));

        Ok(try!(notes.into_iter().map(|note| {
            GitlabNote::new(&self.gitlab, note)
                .map(|note| Box::new(note) as Box<HostedComment>)
        }).collect()))
    }

    fn post_issue_comment(&self, issue: &HostedIssue, content: &str) -> Result<(), HostError> {
        try!(self.gitlab.create_issue_note(
            gitlab::ProjectId::new(issue.repo().id()),
            gitlab::IssueId::new(issue.id()),
            content));

        Ok(())
    }

    fn get_mr_comments(&self, mr: &HostedMergeRequest)
                       -> Result<Vec<Box<HostedComment>>, HostError> {
        let notes = try!(self.gitlab.merge_request_notes(
            gitlab::ProjectId::new(mr.target_repo().id()),
            gitlab::MergeRequestId::new(mr.id())));

        Ok(try!(notes.into_iter().map(|note| {
            GitlabNote::new(&self.gitlab, note)
                .map(|note| Box::new(note) as Box<HostedComment>)
        }).collect()))
    }

    fn post_mr_comment(&self, mr: &HostedMergeRequest, content: &str) -> Result<(), HostError> {
        try!(self.gitlab.create_merge_request_note(
            gitlab::ProjectId::new(mr.target_repo().id()),
            gitlab::MergeRequestId::new(mr.id()),
            content));

        Ok(())
    }

    fn post_commit_comment(&self, commit: &HostedCommit, content: &str) -> Result<(), HostError> {
        try!(self.gitlab.create_commit_comment(
            gitlab::ProjectId::new(commit.project().id()),
            commit.id(),
            content));

        Ok(())
    }

    fn post_commit_status(&self, status: &HostedCommitStatus) -> Result<(), HostError> {
        try!(self.gitlab.create_commit_status(
            gitlab::ProjectId::new(status.commit.project().id()),
            status.commit.id(),
            status.state.into(),
            &gitlab_status_info(status)));

        Ok(())
    }

    fn user_access(&self, repo: &HostedRepo, user: &HostedUser) -> Result<u64, HostError> {
        let pid = gitlab::ProjectId::new(repo.id());
        let uid = gitlab::UserId::new(user.id());
        let project_access = self.gitlab.project_access(pid, uid).ok();

        let project = self.gitlab.project(pid);

        let gid = if let Ok(project) = project {
                match project.namespace.namespace_id() {
                    gitlab::NamespaceId::Group(gid) => Some(gid),
                    gitlab::NamespaceId::User(_) => None,
                }
            } else {
                None
            };

        let group_access = if let Some(gid) = gid {
                self.gitlab.group_access(gid, uid).ok()
            } else {
                None
            };

        Ok([project_access, group_access]
           .iter()
           .filter_map(|&a| a)
           .max()
           .unwrap_or(0))
    }
}

impl From<CommitStatusState> for gitlab::StatusState {
    fn from(state: CommitStatusState) -> Self {
        match state {
            CommitStatusState::Pending => gitlab::StatusState::Pending,
            CommitStatusState::Running => gitlab::StatusState::Running,
            CommitStatusState::Success => gitlab::StatusState::Success,
            CommitStatusState::Failed => gitlab::StatusState::Failed,
        }
    }
}

impl From<gitlab::Error> for HostError {
    fn from(error: gitlab::Error) -> Self {
        let wrap_err = Box::new(error);

        match *wrap_err.as_ref() {
            gitlab::Error::Ease(_) |
            gitlab::Error::UrlParse(_) => HostError::Communication(wrap_err),
            gitlab::Error::Gitlab(_) => HostError::Host(wrap_err),
            gitlab::Error::Deserialize(_) => HostError::InvalidFormat(wrap_err),
        }
    }
}
