// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

extern crate chrono;
use self::chrono::{DateTime, UTC};

extern crate kitware_git;
use self::kitware_git::{CommitId, GitContext};

use super::super::super::host::*;

use std::cell::{Cell, RefCell};
use std::cmp::min;
use std::collections::hash_map::HashMap;
use std::error;
use std::fmt::{self, Display, Formatter};
use std::io;
use std::rc::Rc;

#[derive(Clone)]
struct MockCommit {
    repo: MockRepo,
    id: CommitId,
}

impl HostedCommit for MockCommit {
    fn project(&self) -> &HostedRepo {
        &self.repo
    }

    fn refname(&self) -> Option<&str> {
        // TODO
        None
    }

    fn id(&self) -> &str {
        self.id.as_str()
    }

    fn _self_ref_hack(&self) -> &HostedCommit {
        self
    }
}

#[derive(Clone)]
struct MockRepo {
    name: String,
    url: String,
    id: u64,
}

impl HostedRepo for MockRepo {
    fn name(&self) -> &str {
        &self.name
    }

    fn url(&self) -> &str {
        &self.url
    }

    fn id(&self) -> u64 {
        self.id
    }
}

#[derive(Clone)]
struct MockIssue {
    repo: MockRepo,
    id: u64,
    url: String,
    description: String,
    labels: Vec<String>,
    milestone: Option<String>,
    assignee: Option<String>,

    comments: Vec<MockComment>,
}

impl HostedIssue for MockIssue {
    fn repo(&self) -> &HostedRepo {
        &self.repo
    }

    fn id(&self) -> u64 {
        self.id
    }

    fn url(&self) -> &str {
        &self.url
    }

    fn description(&self) -> &str {
        &self.description
    }

    fn labels(&self) -> &[String] {
        &self.labels
    }

    fn milestone(&self) -> Option<&str> {
        self.milestone.as_ref().map(|m| m.as_str())
    }

    fn assignee(&self) -> Option<&str> {
        self.assignee.as_ref().map(|a| a.as_str())
    }
}

#[derive(Clone)]
struct MockMergeRequest {
    source_repo: MockRepo,
    source_branch: String,
    target_repo: MockRepo,
    target_branch: String,
    id: u64,
    url: String,
    work_in_progress: bool,
    description: String,
    old_commit: Option<MockCommit>,
    commit: MockCommit,

    comments: Vec<MockComment>,
}

impl HostedMergeRequest for MockMergeRequest {
    fn source_repo(&self) -> &HostedRepo {
        &self.source_repo
    }

    fn source_branch(&self) -> &str {
        &self.source_branch
    }

    fn target_repo(&self) -> &HostedRepo {
        &self.target_repo
    }

    fn target_branch(&self) -> &str {
        &self.target_branch
    }

    fn id(&self) -> u64 {
        self.id
    }

    fn url(&self) -> &str {
        &self.url
    }

    fn work_in_progress(&self) -> bool {
        self.work_in_progress
    }

    fn description(&self) -> &str {
        &self.description
    }

    fn old_commit(&self) -> Option<&HostedCommit> {
        self.old_commit.as_ref().map(|oc| oc as &HostedCommit)
    }

    fn commit(&self) -> &HostedCommit {
        &self.commit
    }
}

#[derive(Clone)]
struct MockUser {
    id: u64,
    handle: String,
    name: String,
    email: String,
}

impl HostedUser for MockUser {
    fn id(&self) -> u64 {
        self.id
    }

    fn handle(&self) -> &str {
        &self.handle
    }

    fn name(&self) -> &str {
        &self.name
    }

    fn email(&self) -> &str {
        &self.email
    }
}

#[derive(Clone)]
struct MockComment {
    is_system: bool,
    created_at: DateTime<UTC>,
    updated_at: DateTime<UTC>,
    author: MockUser,
    content: String,
}

impl HostedComment for MockComment {
    fn is_system(&self) -> bool {
        self.is_system
    }

    fn created_at(&self) -> &DateTime<UTC> {
        &self.created_at
    }

    fn updated_at(&self) -> &DateTime<UTC> {
        &self.updated_at
    }

    fn author(&self) -> &HostedUser {
        &self.author
    }

    fn content(&self) -> &str {
        &self.content
    }
}

#[derive(Clone)]
pub struct MockCommitStatus {
    commit: MockCommit,
    pub state: CommitStatusState,
    pub refname: Option<String>,
    pub name: String,
    pub description: String,
}

impl MockCommitStatus {
    fn from_hosted_status(status: &HostedCommitStatus) -> Self {
        MockCommitStatus {
            commit: MockCommit {
                repo: MockRepo {
                    name: status.commit.project().name().to_string(),
                    url: status.commit.project().url().to_string(),
                    id: status.commit.project().id(),
                },
                id: CommitId::new(status.commit.id()),
            },
            state: status.state,
            refname: status.refname.map(|r| r.to_string()),
            name: status.name.to_string(),
            description: status.description.to_string(),
        }
    }
}

pub struct MockService {
    projects: HashMap<String, MockRepo>,
    issues: HashMap<u64, Vec<MockIssue>>,
    merge_requests: HashMap<u64, Vec<MockMergeRequest>>,

    step: Cell<usize>,

    issue_comments: RefCell<HashMap<u64, Vec<String>>>,
    mr_comments: RefCell<HashMap<u64, Vec<String>>>,
    commit_comments: RefCell<HashMap<String, Vec<String>>>,
    commit_statuses: RefCell<HashMap<String, Vec<MockCommitStatus>>>,
}

impl MockService {
    fn new() -> Self {
        MockService {
            projects: HashMap::new(),
            issues: HashMap::new(),
            merge_requests: HashMap::new(),
            step: Cell::new(0),

            issue_comments: RefCell::new(HashMap::new()),
            mr_comments: RefCell::new(HashMap::new()),
            commit_comments: RefCell::new(HashMap::new()),
            commit_statuses: RefCell::new(HashMap::new()),
        }
    }

    fn add_project(mut self, project: MockRepo) -> Self {
        self.projects.insert(project.name.clone(), project);

        self
    }

    fn add_issue(mut self, issue: MockIssue) -> Self {
        self.issues
            .entry(issue.id)
            .or_insert_with(|| Vec::new())
            .push(issue);

        self
    }

    fn add_merge_request(mut self, mr: MockMergeRequest) -> Self {
        self.merge_requests
            .entry(mr.id)
            .or_insert_with(|| Vec::new())
            .push(mr);

        self
    }

    fn find_project(&self, project: &str) -> Result<&MockRepo, HostError> {
        match self.projects.get(project) {
            Some(ref p) => Ok(p),
            None => Err(HostError::Host(MockError::new("invalid project"))),
        }
    }

    fn find_issue(&self, id: u64) -> Result<&MockIssue, HostError> {
        match self.issues.get(&id) {
            Some(ref p) => {
                let issue_step = min(self.step.get(), p.len() - 1);
                Ok(&p[issue_step])
            },
            None => Err(HostError::Host(MockError::new("invalid issue"))),
        }
    }

    fn find_merge_request(&self, id: u64) -> Result<&MockMergeRequest, HostError> {
        match self.merge_requests.get(&id) {
            Some(ref p) => {
                let mr_step = min(self.step.get(), p.len() - 1);
                Ok(&p[mr_step])
            },
            None => Err(HostError::Host(MockError::new("invalid merge request"))),
        }
    }

    pub fn test_service() -> Rc<Self> {
        let base_repo = MockRepo {
                name: "base".to_string(),
                url: "base".to_string(),
                id: 1,
            };
        let fork_repo = MockRepo {
                name: "fork".to_string(),
                url: "fork".to_string(),
                id: 2,
            };
        let self_repo = MockRepo {
                name: "self".to_string(),
                url: concat!(env!("CARGO_MANIFEST_DIR"), "/.git").to_string(),
                id: 3,
            };

        let make_mr = |old_commit: Option<&str>, commit: &str, id: u64, desc: &str| {
                MockMergeRequest {
                    source_repo: fork_repo.clone(),
                    source_branch: format!("topic-{}", id),
                    target_repo: base_repo.clone(),
                    target_branch: "master".to_string(),
                    id: id,
                    url: format!("mr{}", id),
                    work_in_progress: false,
                    description: desc.to_string(),
                    old_commit: old_commit.map(|c| {
                        MockCommit {
                            repo: fork_repo.clone(),
                            id: CommitId::new(c),
                        }
                    }),
                    commit: MockCommit {
                        repo: fork_repo.clone(),
                        id: CommitId::new(commit),
                    },

                    comments: Vec::new(),
                }
            };
        let mr1 = |old_commit: Option<&str>, commit: &str| {
                make_mr(old_commit, commit, 1, "a simple topic")
            };
        let mr2 = |old_commit: Option<&str>, commit: &str| {
                make_mr(old_commit, commit, 2, "a different, simple topic")
            };
        let mr3 = |old_commit: Option<&str>, commit: &str| {
                make_mr(old_commit,
                        commit,
                        3,
                        "a topic which conflicts with the update based")
            };

        static TOPIC1: &'static str = "7189cf557ba2c7c61881ff8669158710b94d8df1";
        static TOPIC1_CONFLICT: &'static str = "755842266dcc5739c06d61433241f44b9306f24c";
        static TOPIC1_UPDATE: &'static str = "fe70f127605efb6032cacea0bd336428d67ed5a3";
        static TOPIC1_MISSED_UPDATE: &'static str = "1b340d2edcf19077ab3e27ddda7430a6612c2f62";

        static TOPIC2: &'static str = "f6f8de8c7c5f1a081b14f5a47c7798268f383222";

        static TOPIC3: &'static str = "7a28f8ea8759ff4f125ddbca825f976927a61310";

        let mr1_base = mr1(None, TOPIC1);
        let mr1_conflict = mr1(Some(TOPIC1), TOPIC1_CONFLICT);
        let mr1_update = mr1(Some(TOPIC1), TOPIC1_UPDATE);
        let mr1_missed_update = mr1(Some(TOPIC1_UPDATE), TOPIC1_MISSED_UPDATE);

        let mr2_base = mr2(None, TOPIC2);

        let mr3_base = mr3(None, TOPIC3);

        Rc::new(Self::new()
            .add_project(base_repo.clone())
            .add_project(fork_repo.clone())
            .add_project(self_repo.clone())
            .add_merge_request(mr1_base)
            .add_merge_request(mr1_conflict)
            .add_merge_request(mr1_update)
            .add_merge_request(mr1_missed_update)
            .add_merge_request(mr2_base)
            .add_merge_request(mr3_base))
    }

    pub fn step(&self, step: usize) {
        self.step.set(step);
    }

    pub fn reset_data(&self) {
        self.issue_comments.borrow_mut().clear();
        self.mr_comments.borrow_mut().clear();
        self.commit_comments.borrow_mut().clear();
        self.commit_statuses.borrow_mut().clear();
    }

    pub fn issue_comments(&self, id: u64) -> Vec<String> {
        self.issue_comments
            .borrow_mut()
            .remove(&id)
            .unwrap_or_else(|| Vec::new())
    }

    pub fn mr_comments(&self, id: u64) -> Vec<String> {
        self.mr_comments
            .borrow_mut()
            .remove(&id)
            .unwrap_or_else(|| Vec::new())
    }

    pub fn commit_comments(&self, commit: &str) -> Vec<String> {
        self.commit_comments
            .borrow_mut()
            .remove(commit)
            .unwrap_or_else(|| Vec::new())
    }

    pub fn commit_statuses(&self, commit: &str) -> Vec<MockCommitStatus> {
        self.commit_statuses
            .borrow_mut()
            .remove(commit)
            .unwrap_or_else(|| Vec::new())
    }

    pub fn remaining_data(&self) -> usize {
        0 +
            self.issue_comments.borrow().len() +
            self.mr_comments.borrow().len() +
            self.commit_comments.borrow().len() +
            self.commit_statuses.borrow().len()
    }
}

#[derive(Debug)]
struct MockError(String);

impl MockError {
    fn new(desc: &str) -> Box<Self> {
        Box::new(MockError(desc.to_string()))
    }
}

impl Display for MockError {
    fn fmt(&self, f: &mut Formatter) -> fmt::Result {
        write!(f, "mock service error: {}", self.0)
    }
}

impl error::Error for MockError {
    fn description(&self) -> &str {
        "Mock service error"
    }
}

impl HostingService for MockService {
    fn fetch_commit(&self, _: &GitContext, _: &HostedCommit) -> Result<(), io::Error> {
        Ok(())
    }

    fn fetch_mr(&self, _: &GitContext, _: &HostedMergeRequest) -> Result<(), io::Error> {
        Ok(())
    }

    fn commit(&self, project: &str, commit: &CommitId) -> Result<Box<HostedCommit>, HostError> {
        let project = try!(self.find_project(project));

        Ok(Box::new(MockCommit {
            repo: project.clone(),
            // Just act as if the commit is a part of the project.
            id: commit.clone(),
        }))
    }

    fn issue(&self, project: &str, id: u64) -> Result<Box<HostedIssue>, HostError> {
        let _ = try!(self.find_project(project));

        self.find_issue(id)
            .map(|issue| Box::new(issue.clone()) as Box<HostedIssue>)
    }

    fn merge_request(&self, project: &str, id: u64) -> Result<Box<HostedMergeRequest>, HostError> {
        let _ = try!(self.find_project(project));

        self.find_merge_request(id)
            .map(|mr| Box::new(mr.clone()) as Box<HostedMergeRequest>)
    }

    fn repo(&self, project: &str) -> Result<Box<HostedRepo>, HostError> {
        self.find_project(project)
            .map(|project| Box::new(project.clone()) as Box<HostedRepo>)
    }

    fn get_issue_comments(&self, issue: &HostedIssue)
                          -> Result<Vec<Box<HostedComment>>, HostError> {
        let issue = try!(self.find_issue(issue.id()));

        issue.comments
            .iter()
            .map(|comment| Ok(Box::new(comment.clone()) as Box<HostedComment>))
            .collect()
    }

    fn post_issue_comment(&self, issue: &HostedIssue, content: &str) -> Result<(), HostError> {
        let mut comment_map = self.issue_comments.borrow_mut();
        let mut comments = comment_map.entry(issue.id()).or_insert_with(|| Vec::new());

        info!(target: "test/service",
              "posting a comment to Issue#{}: {}",
              issue.id(), content);

        Ok(comments.push(content.to_string()))
    }

    fn get_mr_comments(&self, mr: &HostedMergeRequest)
                       -> Result<Vec<Box<HostedComment>>, HostError> {
        let mr = try!(self.find_merge_request(mr.id()));

        mr.comments
            .iter()
            .map(|comment| Ok(Box::new(comment.clone()) as Box<HostedComment>))
            .collect()
    }

    fn post_mr_comment(&self, mr: &HostedMergeRequest, content: &str) -> Result<(), HostError> {
        let mut comment_map = self.mr_comments.borrow_mut();
        let mut comments = comment_map.entry(mr.id()).or_insert_with(|| Vec::new());

        info!(target: "test/service",
              "posting a comment to MR#{}: {}",
              mr.id(), content);

        Ok(comments.push(content.to_string()))
    }

    fn post_commit_comment(&self, commit: &HostedCommit, content: &str) -> Result<(), HostError> {
        let mut comment_map = self.commit_comments.borrow_mut();
        let mut comments = comment_map.entry(commit.id().to_string()).or_insert_with(|| Vec::new());

        info!(target: "test/service",
              "posting a comment to commit {}: {}",
              commit.id(), content);

        Ok(comments.push(content.to_string()))
    }

    fn post_commit_status(&self, status: &HostedCommitStatus) -> Result<(), HostError> {
        let mut comment_map = self.commit_statuses.borrow_mut();
        let mut comments = comment_map.entry(status.commit.id().to_string())
            .or_insert_with(|| Vec::new());

        info!(target: "test/service",
              "posting a {:?} status to commit {} ({:?}) by {}: {}",
              status.state, status.commit.id(), status.refname,
              status.name, status.description);

        Ok(comments.push(MockCommitStatus::from_hosted_status(status)))
    }
}
