// Copyright 2016 Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use super::host::MockService;
use super::utils::{check_git_config_value, check_git_config_values, test_workspace_dir};
use super::super::clone::*;
use super::super::super::host::{HostedProject, HostingService};

use std::path::Path;
use std::rc::Rc;

static PROJECT_NAME: &'static str = "self";

fn create_clone<'a>(path: &Path, service: &Rc<MockService>) -> Clone_ {
    let project = HostedProject {
            name: PROJECT_NAME.to_string(),
            service: service.clone(),
        };

    Clone_::new(path, project)
}

#[test]
fn test_clone_watched_repo() {
    let tempdir = test_workspace_dir("test_clone_watched_repo");
    let service = MockService::test_service();
    let clone = create_clone(tempdir.path(), &service);
    assert!(!clone.exists());

    let ctx = clone.clone_watched_repo().unwrap();

    check_git_config_value(&ctx, "remote.origin.tagopt", "--no-tags");
    check_git_config_value(&ctx,
                           "remote.origin.url",
                           service.repo(PROJECT_NAME).unwrap().url());
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");

    let clone_after = create_clone(tempdir.path(), &service);
    assert!(clone_after.exists());

    let ctx = clone_after.clone_watched_repo().unwrap();

    check_git_config_value(&ctx, "remote.origin.tagopt", "--no-tags");
    check_git_config_value(&ctx,
                           "remote.origin.url",
                           service.repo(PROJECT_NAME).unwrap().url());
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
}

#[test]
fn test_clone_mirror_repo() {
    let tempdir = test_workspace_dir("test_clone_mirror_repo");
    let service = MockService::test_service();
    let clone = create_clone(tempdir.path(), &service);
    assert!(!clone.exists());

    let refs = [
            "refs/heads/*",
            "refs/tags/*",
        ];
    let ctx = clone.clone_mirror_repo(&refs).unwrap();

    check_git_config_value(&ctx,
                           "remote.origin.url",
                           service.repo(PROJECT_NAME).unwrap().url());
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
    check_git_config_values(&ctx,
                            "remote.origin.fetch",
                            &["+refs/heads/*:refs/heads/*", "+refs/tags/*:refs/tags/*"]);

    let clone_after = create_clone(tempdir.path(), &service);
    assert!(clone_after.exists());

    let ctx = clone_after.clone_mirror_repo(&refs).unwrap();

    check_git_config_value(&ctx,
                           "remote.origin.url",
                           service.repo(PROJECT_NAME).unwrap().url());
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
    check_git_config_values(&ctx,
                            "remote.origin.fetch",
                            &["+refs/heads/*:refs/heads/*", "+refs/tags/*:refs/tags/*"]);

    let clone_change = create_clone(tempdir.path(), &service);
    assert!(clone_change.exists());

    let new_refs = [
            "refs/heads/*",
            "refs/tags/*",
            "refs/notes/*",
        ];
    let ctx = clone_change.clone_mirror_repo(&new_refs).unwrap();

    check_git_config_value(&ctx,
                           "remote.origin.url",
                           service.repo(PROJECT_NAME).unwrap().url());
    check_git_config_value(&ctx, "core.logAllRefUpdates", "true");
    check_git_config_values(&ctx,
                            "remote.origin.fetch",
                            &["+refs/heads/*:refs/heads/*",
                              "+refs/tags/*:refs/tags/*",
                              "+refs/notes/*:refs/notes/*"]);
}
